import React from "react";
import { Link } from "@inertiajs/react";
import { motion } from "framer-motion";
import PublicLayout from "@/Layouts/PublicLayout";
import HeroVideoBackground from "@/Components/HeroVideoBackground";
import { BookOpen, Users, Target, TrendingUp, Award, Microscope, ChevronRight } from "lucide-react";

// Importa el MP4 para que Vite lo gestione en producción (hash + cache busting)
import heroVideo from "@/assets/video/video_investigacion.mp4";

export default function PresentacionProyectos() {
  const features = [
    { icon: BookOpen, title: "Presentación de proyectos", description: "Ficha institucional por proyecto: título, resumen, objetivos, equipo, ODS e impacto esperado." },
    { icon: Target, title: "Líneas y ejes estratégicos", description: "Agrupación por áreas temáticas y líneas de investigación para facilitar la exploración temática." },
    { icon: TrendingUp, title: "Transparencia e impacto", description: "Resultados esperados, indicadores y aportes a la comunidad científica y a la sociedad en general." },
  ];

  const stats = [
    { number: "50+", label: "Proyectos Activos", icon: Microscope },
    { number: "200+", label: "Investigadores", icon: Users },
    { number: "15", label: "Líneas de Investigación", icon: Target },
    { number: "12", label: "ODS Impactados", icon: Award },
  ];

  return (
    <PublicLayout title="Proyectos · Presentación">
      {/* HERO con componente reutilizable */}
      <HeroVideoBackground src={heroVideo} minHeight="min-h-[75vh]" overlayClass="bg-slate-900/45">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-20 sm:py-24 lg:py-32">
          <div className="max-w-3xl">
            <motion.div initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.5 }} className="inline-flex items-center gap-2 px-4 py-2 rounded-md bg-white/10 backdrop-blur border border-white/20 shadow-sm mb-8">
              <div className="w-1.5 h-1.5 bg-blue-300 rounded-full" />
              <span className="text-sm font-medium text-white tracking-wide">Portal Institucional de Investigación</span>
            </motion.div>

            <motion.h1 initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.5, delay: 0.05 }} className="text-4xl sm:text-5xl lg:text-6xl font-serif font-bold mb-6 text-white leading-tight">
              Proyectos de Investigación Científica
            </motion.h1>

            <motion.div initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.5, delay: 0.1 }} className="w-24 h-1 bg-blue-300 mb-6" />

            <motion.p initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.5, delay: 0.15 }} className="text-lg text-white/90 mb-8 leading-relaxed font-light max-w-2xl">
              Descubra cómo nuestros proyectos de investigación contribuyen al avance del conocimiento
              científico y al desarrollo sostenible. Explore objetivos, equipos de trabajo, líneas de
              investigación y su alineación con los Objetivos de Desarrollo Sostenible.
            </motion.p>

            <motion.div initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.5, delay: 0.2 }} className="flex flex-col sm:flex-row gap-4">
              <Link href="/proyectos" className="group inline-flex items-center justify-center gap-2 px-8 py-3.5 bg-blue-900 text-white font-medium hover:bg-blue-800 transition-all duration-200 shadow-sm">
                Catálogo de Proyectos
                <ChevronRight className="w-4 h-4 group-hover:translate-x-1 transition-transform" />
              </Link>
              <Link href="/login" className="inline-flex items-center justify-center gap-2 px-8 py-3.5 bg-white/10 text-white font-medium border border-white/20 hover:bg-white/15 transition-all duration-200 backdrop-blur">
                Acceso Institucional
              </Link>
            </motion.div>
          </div>
        </div>
      </HeroVideoBackground>

      {/* ESTADÍSTICAS */}
      <section className="bg-white py-4 border-y border-slate-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <motion.div initial={{ opacity: 0, y: 20 }} whileInView={{ opacity: 1, y: 0 }} viewport={{ once: true }} transition={{ duration: 0.5 }} className="grid grid-cols-2 lg:grid-cols-4 gap-8">
            {stats.map((stat, index) => (
              <motion.div key={stat.label} initial={{ opacity: 0, y: 20 }} whileInView={{ opacity: 1, y: 0 }} viewport={{ once: true }} transition={{ duration: 0.4, delay: index * 0.1 }} className="text-center py-6">
                <stat.icon className="w-8 h-8 mx-auto mb-3 text-blue-900" strokeWidth={1.5} />
                <div className="text-4xl font-serif font-bold text-slate-900 mb-2">{stat.number}</div>
                <div className="text-sm text-slate-600 font-medium uppercase tracking-wider">{stat.label}</div>
              </motion.div>
            ))}
          </motion.div>
        </div>
      </section>

      {/* CARACTERÍSTICAS */}
      <section className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-20 sm:py-24 lg:py-28">
        <motion.div initial={{ opacity: 0, y: 20 }} whileInView={{ opacity: 1, y: 0 }} viewport={{ once: true }} className="text-center mb-16">
          <h2 className="text-3xl sm:text-4xl lg:text-5xl font-serif font-bold text-slate-900 mb-4">Contenido del Portal</h2>
          <div className="w-24 h-1 bg-blue-900 mx-auto mb-6" />
          <p className="text-lg text-slate-600 max-w-3xl mx-auto font-light">Información detallada y estructurada sobre los proyectos de investigación institucionales</p>
        </motion.div>

        <div className="grid md:grid-cols-3 gap-8">
          {features.map((feature, index) => (
            <motion.div
              key={feature.title}
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.5, delay: index * 0.1 }}
              className="group bg-white p-8 border border-slate-200 hover:border-blue-900 hover:shadow-lg transition-all duration-300"
            >
              <div className="inline-flex p-3 bg-slate-100 mb-6 group-hover:bg-blue-900 transition-colors duration-300">
                <feature.icon className="w-6 h-6 text-slate-700 group-hover:text-white transition-colors duration-300" strokeWidth={1.5} />
              </div>
              <h3 className="text-xl font-serif font-bold text-slate-900 mb-4">{feature.title}</h3>
              <p className="text-slate-600 leading-relaxed font-light mb-4">{feature.description}</p>
              <div className="inline-flex items-center text-blue-900 font-medium text-sm opacity-0 group-hover:opacity-100 transition-opacity">
                Más información <ChevronRight className="w-4 h-4 ml-1" />
              </div>
            </motion.div>
          ))}
        </div>
      </section>

      {/* CTA */}
      <section className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16 sm:py-20">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.5 }}
          className="relative overflow-hidden bg-gradient-to-r from-slate-50 to-white border-2 border-slate-200 shadow-sm"
        >
          <div className="relative z-10 p-8 sm:p-12 lg:p-16 flex flex-col lg:flex-row items-center justify-between gap-6 lg:gap-8">
            <div className="text-center lg:text-left">
              <h3 className="text-2xl sm:text-3xl lg:text-4xl font-serif font-bold text-slate-900 mb-3">Explore el Catálogo Completo</h3>
              <p className="text-slate-600 text-base sm:text-lg max-w-2xl font-light">Consulte la lista completa de proyectos activos y acceda al detalle de cada iniciativa de investigación</p>
            </div>

            <Link href="/proyectos" className="group inline-flex items-center gap-3 px-8 py-4 bg-blue-900 text-white font-medium hover:bg-blue-800 transition-colors whitespace-nowrap shadow-sm">
              Ver Proyectos
              <ChevronRight className="w-5 h-5 group-hover:translate-x-1 transition-transform" />
            </Link>
          </div>

          {/* Líneas decorativas */}
          <div className="absolute top-0 left-0 w-2 h-full bg-blue-900" />
          <div className="absolute bottom-0 right-0 w-32 h-1 bg-blue-900" />
        </motion.div>
      </section>
    </PublicLayout>
  );
}
