// resources/js/Pages/ShowProyecto.jsx
import React, { useState } from "react";
import { Link } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import {
  FiArrowLeft, FiClock, FiUsers, FiBookOpen, FiFileText,
  FiDownload, FiGlobe, FiChevronDown, FiPlayCircle, FiTarget, FiAward,
  FiCheckCircle, FiCalendar, FiTrendingUp, FiExternalLink, FiMapPin,
  FiZap, FiLayers, FiGrid, FiShare2, FiMail, FiMaximize2, FiX
} from "react-icons/fi";
import PublicLayout from "@/Layouts/PublicLayout";

/* ===== Componentes Reutilizables ===== */
const Badge = ({ children, variant = "default", icon: Icon }) => {
  const variants = {
    default: "bg-blue-50 text-blue-700 border border-blue-700",
    success: "bg-slate-100 text-slate-800 border border-slate-400",
    info: "bg-slate-50 text-slate-700 border border-slate-300",
  };

  return (
    <span className={`inline-flex items-center gap-2 px-4 py-2 text-xs font-bold uppercase tracking-wider ${variants[variant]} mr-3 mb-3`}>
      {Icon && <Icon className="w-4 h-4" strokeWidth={2} />}
      {children}
    </span>
  );
};

const StatCard = ({ icon: Icon, label, value }) => {
  return (
    <motion.div whileHover={{ y: -4 }} className="border-2 border-slate-200 bg-white p-6 hover:border-blue-700 hover:shadow-md transition-all rounded-xl">
      <div className="flex items-start gap-4">
        <div className="p-3 bg-slate-100 text-slate-700 rounded-lg">
          <Icon className="w-6 h-6" strokeWidth={1.5} />
        </div>
        <div className="flex-1">
          <p className="text-sm text-slate-600 mb-1 font-medium uppercase tracking-wide">{label}</p>
          <p className="text-3xl font-serif font-bold text-slate-700">{value}</p>
        </div>
      </div>
    </motion.div>
  );
};

const Section = ({ title, subtitle, children, icon: Icon }) => (
  <section className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16 sm:py-20">
    <motion.div initial={{ opacity: 0, y: 20 }} whileInView={{ opacity: 1, y: 0 }} viewport={{ once: true }} className="mb-12">
      <div className="flex items-center gap-4 mb-4">
        {Icon && (
          <div className="p-3 bg-slate-100 text-slate-700 rounded-lg">
            <Icon className="w-6 h-6" strokeWidth={1.5} />
          </div>
        )}
        <div>
          <h2 className="text-3xl sm:text-4xl font-serif font-bold text-slate-700">{title}</h2>
          <div className="w-24 h-1 bg-blue-700 mt-3" />
        </div>
      </div>
      {subtitle && <p className="text-slate-600 text-lg font-light ml-20">{subtitle}</p>}
    </motion.div>
    <div>{children}</div>
  </section>
);

/* ===== Componente Principal ===== */
export default function ShowProyecto({
  proyecto = {
    id: "appstrada",
    titulo: "Aplicaciones con Privacidad para Sistemas de Transporte impulsados por Datos (APPSTRADA)",
    slug: "appstrada",
    periodo: "2025–2027",
    estado: "En ejecución",
    linea: "Sistemas Inteligentes de Transporte",
    ods: ["Salud y bienestar", "Ciudades y comunidades sostenibles"],
    resumen:
      "APPSTRADA mejora la privacidad y confidencialidad en SIT mediante técnicas de anonimización y aprendizaje colaborativo, evaluando su impacto en utilidad de datos y precisión de modelos con fuentes reales (ECU911) y sintéticas.",
    objetivoGeneral:
      "Mejorar la privacidad y confidencialidad de la información de usuarios de SIT combinando técnicas de anonimización e IA distribuida.",
    objetivosEspecificos: [
      "Levantar el estado del arte de problemas de privacidad en SIT.",
      "Proponer aplicaciones SIT basadas en ML con datos reales y sintéticos.",
      "Formular criterios de diseño para mecanismos de anonimización.",
      "Definir estrategias de aprendizaje colaborativo consciente de privacidad.",
      "Evaluar desempeño de los modelos de privacidad propuestos.",
      "Difundir resultados (artículos, congresos, comunidad).",
    ],
    hipotesis:
      "La combinación de Privacidad Diferencial (DP) y Aprendizaje Federado (FL) preserva utilidad y precisión de modelos ML garantizando privacidad en SIT.",
    resultadosEsperados: [
      "Escenarios SIT modernos basados en ML para evaluar confidencialidad.",
      "Modelos de privacidad escalables aplicados a SIT impulsados por datos.",
      "Criterios de diseño para implementación de privacidad en SIT.",
      "Curvas de desempeño (métricas, escenarios, redes de acceso).",
      "≥ 2 artículos Q2 o superior; 2 artículos Scopus (≥ B CORE).",
      "≥ 1 Proyecto de Titulación (maestría o grado).",
      "Disertación a la comunidad.",
    ],
    equipo: {
      directores: [
        { rol: "Director", nombre: "Xavier Calderón Hinojosa", avatar: "/assets/img/people/image.png" },
        { rol: "Co-Director", nombre: "Luis Felipe Urquiza", avatar: "/assets/img/people/lfurquiza.jpg" },
      ],
      colaboradores: [
        { rol: "Colaborador Interno", nombre: "Pablo Lupera Morillo", avatar: "/assets/img/people/plupera.jpg" },
        { rol: "Colaborador Externo", nombre: "Eduardo M. Barbecho Bautista", avatar: "/assets/img/people/ebarbecho.jpg" },
      ],
    },
    hero: {
      image: "https://images.unsplash.com/photo-1558494949-ef010cbdcc31?w=1200&q=80",
      video: null,
      poster: "https://images.unsplash.com/photo-1558494949-ef010cbdcc31?w=1200&q=80",
      youtubeId: null,
    },
    galeria: [
      "https://images.unsplash.com/photo-1451187580459-43490279c0fa?w=800&q=80",
      "https://images.unsplash.com/photo-1573164713988-8665fc963095?w=800&q=80",
      "https://images.unsplash.com/photo-1460925895917-afdab827c52f?w=800&q=80",
      "https://images.unsplash.com/photo-1504868584819-f8e8b4b6d7e3?w=800&q=80",
      "https://images.unsplash.com/photo-1551288049-bebda4e38f71?w=800&q=80",
      "https://images.unsplash.com/photo-1551434678-e076c223a692?w=800&q=80",
    ],
    enlaces: {
      web: "#",
      pdfResumen: "/storage/proyectos/appstrada/resumen.pdf",
      pdfPoster: "/storage/proyectos/appstrada/poster.pdf",
      repo: "#",
    },
    milestones: [
      { fecha: "2025-Q1", hito: "Diseño experimental y protocolo de privacidad diferencial", completado: true },
      { fecha: "2025-Q2", hito: "Prototipo FL + DP con datos sintéticos (simuladores)", completado: true },
      { fecha: "2025-Q4", hito: "Piloto con datos reales seccionados (ECU911)", completado: false },
      { fecha: "2026-Q2", hito: "Artículo Q2 someter · Demo SIT con privacidad", completado: false },
      { fecha: "2026-Q4", hito: "Validación ampliada · Segunda publicación", completado: false },
      { fecha: "2027-Q2", hito: "Informe final · Disertación a la comunidad", completado: false },
    ],
    estadoArteBreve:
      "Se revisan enfoques k-anonimato (y extensiones l-diversity, t-closeness) y su vulnerabilidad a ataques de reidentificación; se propone DP (central/local) y FL como alternativas robustas para SIT, evaluando utilidad-privacidad en tráfico vehicular.",
  },
}) {
  const [openEA, setOpenEA] = useState(false);
  const [selectedImage, setSelectedImage] = useState(null);
  const [showShareMenu, setShowShareMenu] = useState(false);

  const estadoConfig = {
    "En ejecución": {
      color: "bg-blue-700",
      pill: "bg-blue-50 text-blue-700 border-blue-700",
    },
    Propuesta: {
      color: "bg-slate-600",
      pill: "bg-slate-100 text-slate-700 border-slate-400",
    },
    Finalizado: {
      color: "bg-slate-400",
      pill: "bg-slate-50 text-slate-600 border-slate-300",
    },
  };

  const cfg = estadoConfig[proyecto.estado] || estadoConfig["En ejecución"];

  return (
    <PublicLayout title={`Proyecto · ${proyecto.titulo}`}>
      {/* ===== HERO ACADÉMICO ===== */}
      <section className="relative overflow-hidden bg-slate-700">
        <div className="relative h-[65vh] min-h-[550px] max-h-[750px]">
          <motion.div initial={{ scale: 1.1 }} animate={{ scale: 1 }} transition={{ duration: 1.5 }} className="absolute inset-0">
            <img src={proyecto.hero.poster || proyecto.hero.image} alt={proyecto.titulo} className="w-full h-full object-cover opacity-40" />
          </motion.div>

          {/* Overlay institucional */}
          <div className="absolute inset-0 bg-gradient-to-r from-slate-700/95 via-slate-700/80 to-slate-700/60" />

          {/* Líneas decorativas */}
          <div className="absolute top-0 left-0 w-2 h-full bg-blue-700" />
          <div className="absolute top-0 right-0 w-2 h-full bg-blue-700" />

          {/* Contenido */}
          <div className="absolute inset-0 flex items-end">
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 pb-16 sm:pb-20 w-full">
              <motion.div initial={{ opacity: 0, y: 30 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.7, delay: 0.2 }} className="max-w-4xl">
                <Link
                  href="/proyectos/lista"
                  className="inline-flex items-center gap-2 text-sm text-white/90 hover:text-white mb-6 transition-colors bg-white/10 px-4 py-2 border border-white/20 font-medium uppercase tracking-wider"
                >
                  <FiArrowLeft className="w-4 h-4" />
                  Volver al Catálogo
                </Link>

                <h1 className="text-3xl sm:text-4xl lg:text-5xl font-serif font-bold text-white mb-8 leading-tight">{proyecto.titulo}</h1>

                <div className="flex flex-wrap items-center">
                  <Badge variant="success" icon={FiClock}>
                    {proyecto.periodo}
                  </Badge>
                  <Badge variant={proyecto.estado === "En ejecución" ? "default" : "success"} icon={FiZap}>
                    {proyecto.estado}
                  </Badge>
                  <Badge variant="info" icon={FiLayers}>
                    {proyecto.linea}
                  </Badge>
                </div>

                <div className="mt-8 flex flex-wrap gap-4">
                  {proyecto.enlaces.pdfResumen && (
                    <a
                      href={proyecto.enlaces.pdfResumen}
                      className="inline-flex items-center gap-2 px-6 py-3 bg-blue-700 text-white font-medium hover:bg-blue-800 transition-all uppercase tracking-wider text-sm"
                    >
                      <FiDownload className="w-4 h-4" />
                      Descargar Resumen
                    </a>
                  )}
                  <button
                    onClick={() => setShowShareMenu(!showShareMenu)}
                    className="inline-flex items-center gap-2 px-6 py-3 bg-white/10 border-2 border-white/20 text-white font-medium hover:bg-white/20 transition-all uppercase tracking-wider text-sm"
                  >
                    <FiShare2 className="w-4 h-4" />
                    Compartir
                  </button>
                </div>
              </motion.div>
            </div>
          </div>
        </div>
      </section>

      {/* VIDEO (opcional) */}
      {(proyecto.hero.video || proyecto.hero.youtubeId) && (
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 -mt-16 relative z-20 mb-16">
          <motion.div initial={{ opacity: 0, y: 30 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.6 }} className="overflow-hidden shadow-xl border-4 border-white bg-black rounded-2xl">
            {proyecto.hero.video ? (
              <video controls poster={proyecto.hero.poster} className="w-full h-[360px] sm:h-[480px] lg:h-[600px] object-cover">
                <source src={proyecto.hero.video} type="video/mp4" />
              </video>
            ) : proyecto.hero.youtubeId ? (
              <div className="relative w-full aspect-video">
                <iframe
                  className="absolute inset-0 w-full h-full"
                  src={`https://www.youtube.com/embed/${proyecto.hero.youtubeId}`}
                  title="Video del proyecto"
                  allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                  allowFullScreen
                />
              </div>
            ) : null}
          </motion.div>
        </div>
      )}

      {/* ESTADÍSTICAS */}
      <section className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="grid grid-cols-2 lg:grid-cols-4 gap-6">
          <StatCard icon={FiUsers} label="Equipo" value={(proyecto.equipo.directores?.length || 0) + (proyecto.equipo.colaboradores?.length || 0)} />
          <StatCard icon={FiTarget} label="Objetivos Específicos" value={proyecto.objetivosEspecificos.length} />
          <StatCard icon={FiAward} label="Resultados Esperados" value={proyecto.resultadosEsperados.length} />
          <StatCard icon={FiTrendingUp} label="Hitos Planificados" value={proyecto.milestones.length} />
        </div>
      </section>

      {/* RESUMEN + ODS */}
      <Section title="Resumen Ejecutivo" icon={FiFileText}>
        <div className="grid lg:grid-cols-3 gap-8">
          <motion.div initial={{ opacity: 0, x: -20 }} whileInView={{ opacity: 1, x: 0 }} viewport={{ once: true }} className="lg:col-span-2">
            <div className="border-l-4 border-blue-700 pl-6">
              <p className="text-slate-700 leading-relaxed text-lg font-light">{proyecto.resumen}</p>
            </div>
          </motion.div>

          <motion.div initial={{ opacity: 0, x: 20 }} whileInView={{ opacity: 1, x: 0 }} viewport={{ once: true }} className="space-y-6">
            <div className="border-2 border-slate-200 bg-slate-50 p-6 rounded-xl">
              <h3 className="font-bold text-slate-700 mb-4 flex items-center gap-2 uppercase tracking-wide text-sm">
                <FiAward className="w-5 h-5" strokeWidth={2} />
                Objetivos de Desarrollo Sostenible
              </h3>
              <div className="space-y-3">
                {proyecto.ods.map((o) => (
                  <div key={o} className="flex items-center gap-3 p-3 bg-white border-2 border-slate-200 text-sm font-medium text-slate-800 rounded-lg">
                    <div className="w-2 h-2 bg-blue-700" />
                    {o}
                  </div>
                ))}
              </div>
            </div>
          </motion.div>
        </div>
      </Section>

      {/* EQUIPO (CORREGIDO) */}
      <Section title="Equipo de Investigación" icon={FiUsers}>
        <div className="grid md:grid-cols-2 gap-12">
          {/* Directores */}
          <div>
            <div className="flex items-center gap-3 mb-6">
              <div className="h-1 w-16 bg-blue-700" />
              <h3 className="text-xl font-serif font-bold text-slate-700">Dirección del Proyecto</h3>
            </div>

            <div className="space-y-4">
              {(proyecto.equipo.directores || []).map((p, idx) => (
                <motion.div
                  key={p.nombre}
                  initial={{ opacity: 0, x: -20 }}
                  whileInView={{ opacity: 1, x: 0 }}
                  viewport={{ once: true }}
                  transition={{ duration: 0.3, delay: idx * 0.05 }}
                  className="group border-2 border-slate-200 bg-white p-5 rounded-xl hover:border-blue-700 hover:shadow-md transition-all flex gap-4 items-center"
                >
                  <img
                    src={p.avatar || "https://ui-avatars.com/api/?name=" + encodeURIComponent(p.nombre)}
                    className="h-16 w-16 rounded-full object-cover ring-4 ring-slate-100"
                    alt={p.nombre}
                  />
                  <div className="flex-1">
                    <p className="font-bold text-slate-700">{p.nombre}</p>
                    <p className="text-sm text-blue-700 font-medium">{p.rol}</p>
                  </div>
                  <FiExternalLink className="w-5 h-5 text-slate-400 group-hover:text-blue-700 transition-colors" />
                </motion.div>
              ))}
            </div>
          </div>

          {/* Colaboradores */}
          <div>
            <div className="flex items-center gap-3 mb-6">
              <div className="h-1 w-16 bg-blue-700" />
              <h3 className="text-xl font-serif font-bold text-slate-700">Colaboradores</h3>
            </div>

            <div className="space-y-4">
              {(proyecto.equipo.colaboradores || []).map((p, idx) => (
                <motion.div
                  key={p.nombre}
                  initial={{ opacity: 0, x: 20 }}
                  whileInView={{ opacity: 1, x: 0 }}
                  viewport={{ once: true }}
                  transition={{ duration: 0.3, delay: idx * 0.05 }}
                  className="group border-2 border-slate-200 bg-white p-5 rounded-xl hover:border-blue-700 hover:shadow-md transition-all flex gap-4 items-center"
                >
                  <img
                    src={p.avatar || "https://ui-avatars.com/api/?name=" + encodeURIComponent(p.nombre)}
                    className="h-16 w-16 rounded-full object-cover ring-4 ring-slate-100"
                    alt={p.nombre}
                  />
                  <div className="flex-1">
                    <p className="font-bold text-slate-700">{p.nombre}</p>
                    <p className="text-sm text-slate-700 font-medium">{p.rol}</p>
                  </div>
                  <FiExternalLink className="w-5 h-5 text-slate-400 group-hover:text-blue-700 transition-colors" />
                </motion.div>
              ))}
            </div>
          </div>
        </div>
      </Section>

      {/* ESTADO DEL ARTE (MOVIDO AQUÍ) */}
      <Section title="Estado del arte" icon={FiBookOpen} subtitle="Marco teórico y fundamentos científicos">
        <div className="mt-2 border-2 border-slate-200 bg-slate-50 p-8 rounded-2xl">
          <div className="prose prose-lg max-w-none">
            <p className="text-slate-700 leading-relaxed mb-8 font-light">{proyecto.estadoArteBreve}</p>

            <h4 className="text-lg font-serif font-bold text-slate-700 mb-6 flex items-center gap-3">
              <div className="w-1 h-6 bg-blue-700" />
              Aspectos Clave Revisados
            </h4>

            <div className="grid md:grid-cols-2 gap-4">
              {[
                "Limitaciones del k-anonimato ante ataques de vinculación y homogeneidad",
                "DP central vs local; ruido (Laplaciano/Gaussiano), canales y curadores confiables",
                "Aprendizaje Federado (FL): entrenamiento descentralizado, privacidad y eficiencia",
                "Aplicaciones a tráfico vehicular, predicción de flujos y reservas (EV charging)",
              ].map((item, idx) => (
                <div key={idx} className="flex items-start gap-3 p-4 bg-white border-2 border-slate-200 rounded-lg">
                  <FiCheckCircle className="w-5 h-5 text-blue-700 flex-shrink-0 mt-0.5" strokeWidth={2} />
                  <span className="text-slate-700 text-sm font-light">{item}</span>
                </div>
              ))}
            </div>
          </div>
        </div>
      </Section>

      {/* GALERÍA */}
      <Section title="Galería Multimedia" icon={FiGrid} subtitle="Imágenes, diagramas y recursos visuales">
        <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
          {proyecto.galeria.map((src, i) => (
            <motion.div
              key={i}
              initial={{ opacity: 0, scale: 0.8 }}
              whileInView={{ opacity: 1, scale: 1 }}
              viewport={{ once: true }}
              transition={{ delay: i * 0.05 }}
              whileHover={{ scale: 1.05 }}
              onClick={() => setSelectedImage(src)}
              className="group relative overflow-hidden border-2 border-slate-200 bg-white hover:border-blue-700 hover:shadow-md transition-all cursor-pointer aspect-square rounded-xl"
            >
              <img src={src} alt={`Galería ${i + 1}`} className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300" />
              <div className="absolute inset-0 bg-gradient-to-t from-slate-700/80 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity flex items-end p-4">
                <span className="text-white text-sm font-medium flex items-center gap-2 uppercase tracking-wide">
                  <FiMaximize2 className="w-4 h-4" />
                  Ampliar
                </span>
              </div>
            </motion.div>
          ))}
        </div>

        {proyecto.hero.video && (
          <motion.div initial={{ opacity: 0, y: 20 }} whileInView={{ opacity: 1, y: 0 }} viewport={{ once: true }} className="mt-8 overflow-hidden border-4 border-slate-200 rounded-2xl">
            <div className="relative bg-black">
              <video controls className="w-full h-[360px] md:h-[440px] object-cover" poster={proyecto.hero.poster}>
                <source src={proyecto.hero.video} type="video/mp4" />
              </video>
            </div>
          </motion.div>
        )}
      </Section>

      {/* Modal imagen */}
      <AnimatePresence>
        {selectedImage && (
          <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} exit={{ opacity: 0 }} onClick={() => setSelectedImage(null)} className="fixed inset-0 bg-black/95 z-50 flex items-center justify-center p-4 cursor-pointer">
            <motion.div initial={{ scale: 0.8 }} animate={{ scale: 1 }} exit={{ scale: 0.8 }} onClick={(e) => e.stopPropagation()} className="relative max-w-6xl w-full">
              <button onClick={() => setSelectedImage(null)} className="absolute -top-12 right-0 text-white hover:text-slate-300 transition-colors">
                <FiX className="w-8 h-8" />
              </button>
              <img src={selectedImage} alt="Ampliada" className="w-full h-auto border-4 border-white rounded-xl" />
            </motion.div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* RECURSOS */}
      <Section title="Recursos y Documentación" icon={FiDownload}>
        <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
          {[
            {
              icon: FiFileText,
              title: "Resumen Ejecutivo",
              description: "Ficha técnica del proyecto",
              link: proyecto.enlaces.pdfResumen,
              action: "Descargar PDF",
            },
            {
              icon: FiBookOpen,
              title: "Póster Científico",
              description: "Versión para eventos académicos",
              link: proyecto.enlaces.pdfPoster,
              action: "Descargar Póster",
            },
            {
              icon: FiGlobe,
              title: "Repositorio",
              description: "Documentación técnica completa",
              link: proyecto.enlaces.web,
              action: "Visitar Sitio",
            },
            {
              icon: FiMail,
              title: "Contacto",
              description: "Información de contacto institucional",
              link: "/contacto",
              action: "Contactar",
            },
          ].map((recurso, idx) => (
            <motion.a
              key={idx}
              href={recurso.link}
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ delay: idx * 0.05 }}
              whileHover={{ y: -4 }}
              className="group relative border-2 border-slate-200 bg-white p-6 hover:border-blue-700 hover:shadow-md transition-all overflow-hidden rounded-xl"
            >
              <div className="relative">
                <div className="inline-flex p-4 bg-slate-100 text-slate-700 group-hover:bg-blue-700 group-hover:text-white transition-colors mb-6 rounded-lg">
                  <recurso.icon className="w-6 h-6" strokeWidth={1.5} />
                </div>
                <h3 className="font-bold text-slate-700 mb-2 leading-tight">{recurso.title}</h3>
                <p className="text-sm text-slate-600 mb-4 font-light">{recurso.description}</p>
                <div className="flex items-center gap-2 text-sm font-medium text-blue-700 group-hover:text-blue-800 uppercase tracking-wide">
                  {recurso.action}
                  <FiExternalLink className="w-4 h-4 group-hover:translate-x-1 transition-transform" strokeWidth={2} />
                </div>
              </div>
            </motion.a>
          ))}
        </div>
      </Section>

    </PublicLayout>
  );
}
