import React, { useMemo, useState } from "react";
import { Link } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import {
  FiFilter, FiSearch, FiX, FiUsers, FiArrowRight, FiTag, FiLayers, FiCalendar,
  FiGrid, FiList, FiMapPin, FiAward, FiTrendingUp, FiChevronDown
} from "react-icons/fi";
import PublicLayout from "@/Layouts/PublicLayout";
import ProyectoCard from "./Proyectos/Components/ProyectoCard";
import ProyectoListItem from "./Proyectos/Components/ProyectoListItem";

/* ========= Utilidades ========= */
const uniq = (arr) => [...new Set(arr.filter(Boolean))];
const norm = (s) => (s || "").toString().toLowerCase().normalize("NFD").replace(/\p{Diacritic}/gu, "");

export default function ProyectosLista({ proyectos: proyectosFromServer = [], filtros: filtrosIniciales = {} }) {
  const demo = [
    {
      id: "appstrada",
      titulo: "Aplicaciones con Privacidad para Sistemas de Transporte impulsados por Datos (APPSTRADA)",
      resumen: "Privacidad en SIT mediante anonimización y aprendizaje distribuido; evaluación con datos reales y sintéticos.",
      directores: [
        { rol: "Director", nombre: "Xavier Calderón Hinojosa" },
        { rol: "Co-Director", nombre: "Luis Felipe Urquiza" },
      ],
      colaboradores: [
        { rol: "Colaborador Interno", nombre: "Pablo Lupera Morillo" },
        { rol: "Colaborador Externo", nombre: "Eduardo Marcelo Barbecho Bautista" },
      ],
      ods: ["Salud y bienestar", "Ciudades y comunidades sostenibles"],
      linea: "Sistemas Inteligentes de Transporte",
      estado: "En ejecución",
      periodo: "2025–2027",
      link: "/proyectos/appstrada",
    },
    {
      id: "iot-energy",
      titulo: "Optimización Energética con IoT para Edificios Inteligentes",
      resumen: "Reducción del consumo energético con redes IoT y ML en campus universitarios.",
      directores: [{ rol: "Director", nombre: "Dra. Soraya Sinche" }],
      colaboradores: [{ rol: "Colaborador Externo", nombre: "Dr. Michael Stein" }],
      ods: ["Energía asequible y no contaminante", "Industria, innovación e infraestructura"],
      linea: "IoT y Energía",
      estado: "Propuesta",
      periodo: "2026–2028",
      link: "/proyectos/iot-energy",
    },
    {
      id: "vision-drone",
      titulo: "Visión por Computador y Drones para Monitoreo Ambiental",
      resumen: "Drones con sensores ópticos y térmicos para monitoreo de ecosistemas y detección temprana.",
      directores: [{ rol: "Director", nombre: "Dr. Andrés Flores" }],
      colaboradores: [{ rol: "Colaborador Interno", nombre: "Mg. Paola Andrade" }],
      ods: ["Vida de ecosistemas terrestres", "Acción por el clima"],
      linea: "Visión por Computador",
      estado: "En ejecución",
      periodo: "2024–2026",
      link: "/proyectos/vision-drone",
    },
  ];

  const proyectos = proyectosFromServer.length ? proyectosFromServer : demo;

  /* ======= Estado ======= */
  const [q, setQ] = useState(filtrosIniciales.q || "");
  const [odsSel, setOdsSel] = useState(filtrosIniciales.ods || "Todos");
  const [directorSel, setDirectorSel] = useState(filtrosIniciales.director || "Todos");
  const [lineaSel, setLineaSel] = useState(filtrosIniciales.linea || "Todos");
  const [estadoSel, setEstadoSel] = useState("Todos");
  const [viewMode, setViewMode] = useState("grid");
  const [page, setPage] = useState(1);
  const [filtersOpen, setFiltersOpen] = useState(true);
  const pageSize = 9;

  /* ======= Opciones dinámicas ======= */
  const odsOptions = useMemo(() => ["Todos", ...uniq(proyectos.flatMap((p) => p.ods || []))], [proyectos]);
  const directoresOptions = useMemo(() => {
    const noms = proyectos.flatMap((p) => (p.directores || []).map((d) => d.nombre));
    return ["Todos", ...uniq(noms)];
  }, [proyectos]);
  const lineasOptions = useMemo(() => ["Todos", ...uniq(proyectos.map((p) => p.linea))], [proyectos]);
  const estadoOptions = ["Todos", "En ejecución", "Propuesta", "Finalizado"];

  /* ======= Filtrado ======= */
  const filtrados = useMemo(() => {
    const nq = norm(q);
    return proyectos.filter((p) => {
      const hitQ =
        !nq ||
        norm(p.titulo).includes(nq) ||
        norm(p.resumen).includes(nq) ||
        norm((p.linea || "")).includes(nq) ||
        (p.directores || []).some((d) => norm(d.nombre).includes(nq)) ||
        (p.colaboradores || []).some((c) => norm(c.nombre).includes(nq));

      const hitOds = odsSel === "Todos" || (p.ods || []).includes(odsSel);
      const hitDir = directorSel === "Todos" || (p.directores || []).some((d) => d.nombre === directorSel);
      const hitLinea = lineaSel === "Todos" || p.linea === lineaSel;
      const hitEstado = estadoSel === "Todos" || p.estado === estadoSel;

      return hitQ && hitOds && hitDir && hitLinea && hitEstado;
    });
  }, [proyectos, q, odsSel, directorSel, lineaSel, estadoSel]);

  /* ======= Paginación ======= */
  const total = filtrados.length;
  const totalPages = Math.max(1, Math.ceil(total / pageSize));
  const pageSafe = Math.min(page, totalPages);
  const slice = filtrados.slice((pageSafe - 1) * pageSize, pageSafe * pageSize);

  React.useEffect(() => setPage(1), [q, odsSel, directorSel, lineaSel, estadoSel]);

  const activeFiltersCount = [q, odsSel !== "Todos", directorSel !== "Todos", lineaSel !== "Todos", estadoSel !== "Todos"].filter(Boolean).length;

  return (
    <PublicLayout title="Proyectos · Listado">
      {/* Hero académico */}
      <section className="relative bg-gradient-to-b from-slate-50 to-white border-b border-slate-200">
        <div className="absolute top-0 left-0 w-1 h-24 bg-blue-900" />
        <div className="absolute top-0 right-0 w-1 h-24 bg-blue-900" />

        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16 sm:py-20 lg:py-24 relative">
          <div className="max-w-3xl">
            <motion.div initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.5 }}>
              <div className="inline-flex items-center gap-2 px-4 py-2 bg-white border border-slate-200 shadow-sm mb-6">
                <div className="w-1.5 h-1.5 bg-blue-900 rounded-full" />
                <span className="text-sm font-medium text-slate-700 tracking-wide">{proyectos.length} proyectos registrados</span>
              </div>
            </motion.div>

            <motion.h1
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5, delay: 0.1 }}
              className="text-3xl sm:text-4xl lg:text-5xl font-serif font-bold mb-4 text-slate-900"
            >
              Catálogo de Proyectos de Investigación
            </motion.h1>

            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5, delay: 0.2 }}
              className="w-24 h-1 bg-blue-900 mb-6"
            />

            <motion.p
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5, delay: 0.3 }}
              className="text-lg text-slate-600 leading-relaxed font-light"
            >
              Explore el portafolio completo de proyectos de investigación institucionales.
              Utilice los filtros para buscar por Objetivos de Desarrollo Sostenible, líneas temáticas,
              directores académicos y estado de ejecución.
            </motion.p>
          </div>
        </div>
      </section>

      {/* Barra de herramientas académica */}
      <section className="sticky top-0 z-40 bg-white border-b-2 border-slate-200 shadow-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-between py-4 gap-4">
            <div className="flex items-center gap-4">
              <button
                onClick={() => setFiltersOpen(!filtersOpen)}
                className="inline-flex items-center gap-2 px-4 py-2.5 border-2 border-slate-300 hover:border-blue-900 hover:bg-slate-50 transition-all font-medium text-slate-700"
              >
                <FiFilter className="w-4 h-4" />
                <span className="hidden sm:inline">Filtros de Búsqueda</span>
                {activeFiltersCount > 0 && (
                  <span className="px-2 py-0.5 bg-blue-900 text-white text-xs font-bold">
                    {activeFiltersCount}
                  </span>
                )}
                <FiChevronDown className={`w-4 h-4 transition-transform ${filtersOpen ? "rotate-180" : ""}`} />
              </button>

              <div className="text-sm text-slate-600">
                <span className="font-bold text-slate-900">{total}</span> proyecto{total === 1 ? "" : "s"} encontrado{total === 1 ? "" : "s"}
              </div>
            </div>

            <div className="flex items-center gap-2">
              <button
                onClick={() => setViewMode("grid")}
                className={`p-2.5 border transition-all ${
                  viewMode === "grid" ? "bg-blue-900 text-white border-blue-900" : "text-slate-600 border-slate-300 hover:border-blue-900 hover:bg-slate-50"
                }`}
              >
                <FiGrid className="w-5 h-5" />
              </button>
              <button
                onClick={() => setViewMode("list")}
                className={`p-2.5 border transition-all ${
                  viewMode === "list" ? "bg-blue-900 text-white border-blue-900" : "text-slate-600 border-slate-300 hover:border-blue-900 hover:bg-slate-50"
                }`}
              >
                <FiList className="w-5 h-5" />
              </button>
            </div>
          </div>
        </div>
      </section>

      {/* Panel de Filtros académico */}
      <AnimatePresence>
        {filtersOpen && (
          <motion.section
            initial={{ height: 0, opacity: 0 }}
            animate={{ height: "auto", opacity: 1 }}
            exit={{ height: 0, opacity: 0 }}
            transition={{ duration: 0.3 }}
            className="bg-slate-50 border-b-2 border-slate-200 overflow-hidden"
          >
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
              <div className="grid lg:grid-cols-5 sm:grid-cols-2 gap-6">
                {/* Búsqueda */}
                <div className="lg:col-span-2">
                  <label className="block">
                    <span className="text-xs font-bold text-slate-700 uppercase tracking-wider mb-3 block">
                      Búsqueda General
                    </span>
                    <div className="relative">
                      <FiSearch className="absolute left-4 top-1/2 -translate-y-1/2 text-slate-400 w-5 h-5" />
                      <input
                        type="text"
                        value={q}
                        onChange={(e) => setQ(e.target.value)}
                        placeholder="Título, director, palabras clave..."
                        className="w-full pl-12 pr-4 py-3 border-2 border-slate-300 focus:border-blue-900 focus:ring-0 bg-white font-light"
                      />
                      {q && (
                        <button
                          onClick={() => setQ("")}
                          className="absolute right-4 top-1/2 -translate-y-1/2 text-slate-400 hover:text-slate-700"
                        >
                          <FiX className="w-5 h-5" />
                        </button>
                      )}
                    </div>
                  </label>
                </div>

                {/* Estado */}
                <label className="block">
                  <span className="text-xs font-bold text-slate-700 uppercase tracking-wider mb-3 block">
                    Estado del Proyecto
                  </span>
                  <select
                    value={estadoSel}
                    onChange={(e) => setEstadoSel(e.target.value)}
                    className="w-full py-3 border-2 border-slate-300 focus:border-blue-900 focus:ring-0 bg-white font-light"
                  >
                    {estadoOptions.map((o) => (
                      <option key={o} value={o}>{o}</option>
                    ))}
                  </select>
                </label>

                {/* Línea */}
                <label className="block">
                  <span className="text-xs font-bold text-slate-700 uppercase tracking-wider mb-3 block">
                    Línea de Investigación
                  </span>
                  <select
                    value={lineaSel}
                    onChange={(e) => setLineaSel(e.target.value)}
                    className="w-full py-3 border-2 border-slate-300 focus:border-blue-900 focus:ring-0 bg-white font-light"
                  >
                    {lineasOptions.map((o) => (
                      <option key={o} value={o}>{o}</option>
                    ))}
                  </select>
                </label>

                {/* ODS */}
                <label className="block">
                  <span className="text-xs font-bold text-slate-700 uppercase tracking-wider mb-3 block">
                    ODS
                  </span>
                  <select
                    value={odsSel}
                    onChange={(e) => setOdsSel(e.target.value)}
                    className="w-full py-3 border-2 border-slate-300 focus:border-blue-900 focus:ring-0 bg-white font-light"
                  >
                    {odsOptions.map((o) => (
                      <option key={o} value={o}>{o}</option>
                    ))}
                  </select>
                </label>

                {/* Director */}
                <label className="block lg:col-span-2">
                  <span className="text-xs font-bold text-slate-700 uppercase tracking-wider mb-3 block">
                    Director / Co-Director
                  </span>
                  <select
                    value={directorSel}
                    onChange={(e) => setDirectorSel(e.target.value)}
                    className="w-full py-3 border-2 border-slate-300 focus:border-blue-900 focus:ring-0 bg-white font-light"
                  >
                    {directoresOptions.map((o) => (
                      <option key={o} value={o}>{o}</option>
                    ))}
                  </select>
                </label>
              </div>

              {/* Filtros activos */}
              {activeFiltersCount > 0 && (
                <div className="mt-6 pt-6 border-t-2 border-slate-300">
                  <div className="flex flex-wrap items-center gap-3">
                    <span className="text-xs font-bold text-slate-700 uppercase tracking-wider">Filtros Activos:</span>
                    {q && (
                      <button
                        onClick={() => setQ("")}
                        className="inline-flex items-center gap-2 text-sm bg-white text-slate-700 px-4 py-2 border-2 border-slate-300 hover:border-blue-900 transition-all"
                      >
                        <FiSearch className="w-3.5 h-3.5" /> "{q}"
                        <FiX className="w-3.5 h-3.5" />
                      </button>
                    )}
                    {estadoSel !== "Todos" && (
                      <button
                        onClick={() => setEstadoSel("Todos")}
                        className="inline-flex items-center gap-2 text-sm bg-white text-slate-700 px-4 py-2 border-2 border-slate-300 hover:border-blue-900 transition-all"
                      >
                        Estado: {estadoSel}
                        <FiX className="w-3.5 h-3.5" />
                      </button>
                    )}
                    {lineaSel !== "Todos" && (
                      <button
                        onClick={() => setLineaSel("Todos")}
                        className="inline-flex items-center gap-2 text-sm bg-white text-slate-700 px-4 py-2 border-2 border-slate-300 hover:border-blue-900 transition-all"
                      >
                        Línea: {lineaSel}
                        <FiX className="w-3.5 h-3.5" />
                      </button>
                    )}
                    {odsSel !== "Todos" && (
                      <button
                        onClick={() => setOdsSel("Todos")}
                        className="inline-flex items-center gap-2 text-sm bg-white text-slate-700 px-4 py-2 border-2 border-slate-300 hover:border-blue-900 transition-all"
                      >
                        ODS: {odsSel}
                        <FiX className="w-3.5 h-3.5" />
                      </button>
                    )}
                    {directorSel !== "Todos" && (
                      <button
                        onClick={() => setDirectorSel("Todos")}
                        className="inline-flex items-center gap-2 text-sm bg-white text-slate-700 px-4 py-2 border-2 border-slate-300 hover:border-blue-900 transition-all"
                      >
                        Director: {directorSel}
                        <FiX className="w-3.5 h-3.5" />
                      </button>
                    )}

                    <button
                      onClick={() => {
                        setQ("");
                        setOdsSel("Todos");
                        setDirectorSel("Todos");
                        setLineaSel("Todos");
                        setEstadoSel("Todos");
                      }}
                      className="ml-auto inline-flex items-center gap-2 text-sm text-slate-700 hover:text-blue-900 font-bold uppercase tracking-wider"
                    >
                      <FiX className="w-4 h-4" /> Limpiar Todos
                    </button>
                  </div>
                </div>
              )}
            </div>
          </motion.section>
        )}
      </AnimatePresence>

      {/* Lista de proyectos */}
      <section className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12 sm:py-16">
        <AnimatePresence mode="wait">
          {slice.length === 0 ? (
            <motion.div
              key="empty"
              initial={{ opacity: 0, scale: 0.95 }}
              animate={{ opacity: 1, scale: 1 }}
              exit={{ opacity: 0, scale: 0.95 }}
              className="border-2 border-slate-300 bg-slate-50 p-16 text-center"
            >
              <div className="max-w-md mx-auto">
                <div className="w-20 h-20 mx-auto mb-6 bg-slate-200 flex items-center justify-center">
                  <FiSearch className="w-10 h-10 text-slate-400" />
                </div>
                <h3 className="text-2xl font-serif font-bold text-slate-900 mb-3">No se Encontraron Proyectos</h3>
                <p className="text-slate-600 mb-8 font-light">
                  Los criterios de búsqueda actuales no coinciden con ningún proyecto.
                  Por favor, ajuste los filtros e intente nuevamente.
                </p>
                <button
                  onClick={() => {
                    setQ("");
                    setOdsSel("Todos");
                    setDirectorSel("Todos");
                    setLineaSel("Todos");
                    setEstadoSel("Todos");
                  }}
                  className="inline-flex items-center gap-2 px-6 py-3 bg-blue-900 text-white font-medium hover:bg-blue-800 transition-colors"
                >
                  <FiX className="w-4 h-4" /> Restablecer Filtros
                </button>
              </div>
            </motion.div>
          ) : (
            <motion.div
              key={`${viewMode}-${pageSafe}`}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              exit={{ opacity: 0, y: -20 }}
              transition={{ duration: 0.3 }}
            >
              {viewMode === "grid" ? (
                <div className="grid md:grid-cols-2 xl:grid-cols-3 gap-6">
                  {slice.map((p, idx) => (
                    <ProyectoCard key={p.id} p={p} index={idx} />
                  ))}
                </div>
              ) : (
                <div className="space-y-4">
                  {slice.map((p, idx) => (
                    <ProyectoListItem key={p.id} p={p} index={idx} />
                  ))}
                </div>
              )}
            </motion.div>
          )}
        </AnimatePresence>

        {/* Paginación académica */}
        {totalPages > 1 && slice.length > 0 && (
          <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} className="mt-16 pt-8 border-t-2 border-slate-200">
            <div className="flex flex-col sm:flex-row items-center justify-between gap-6">
              <div className="text-sm text-slate-600 font-light">
                Mostrando <span className="font-bold text-slate-900">{(pageSafe - 1) * pageSize + 1}</span> a{" "}
                <span className="font-bold text-slate-900">{Math.min(pageSafe * pageSize, total)}</span> de{" "}
                <span className="font-bold text-slate-900">{total}</span> proyectos
              </div>

              <div className="flex items-center gap-2">
                <button
                  onClick={() => setPage((x) => Math.max(1, x - 1))}
                  disabled={pageSafe === 1}
                  className="px-5 py-2.5 border-2 border-slate-300 text-slate-700 font-medium disabled:opacity-40 disabled:cursor-not-allowed hover:border-blue-900 hover:bg-slate-50 transition-all"
                >
                  Anterior
                </button>

                <div className="hidden sm:flex items-center gap-2">
                  {Array.from({ length: Math.min(5, totalPages) }, (_, i) => {
                    let pageNum;
                    if (totalPages <= 5) {
                      pageNum = i + 1;
                    } else if (pageSafe <= 3) {
                      pageNum = i + 1;
                    } else if (pageSafe >= totalPages - 2) {
                      pageNum = totalPages - 4 + i;
                    } else {
                      pageNum = pageSafe - 2 + i;
                    }

                    return (
                      <button
                        key={pageNum}
                        onClick={() => setPage(pageNum)}
                        className={`w-12 h-12 font-medium transition-all ${
                          pageSafe === pageNum ? "bg-blue-900 text-white" : "text-slate-700 border-2 border-slate-300 hover:border-blue-900 hover:bg-slate-50"
                        }`}
                      >
                        {pageNum}
                      </button>
                    );
                  })}
                </div>

                <div className="sm:hidden text-sm font-medium text-slate-700">
                  Página {pageSafe} de {totalPages}
                </div>

                <button
                  onClick={() => setPage((x) => Math.min(totalPages, x + 1))}
                  disabled={pageSafe === totalPages}
                  className="px-5 py-2.5 border-2 border-slate-300 text-slate-700 font-medium disabled:opacity-40 disabled:cursor-not-allowed hover:border-blue-900 hover:bg-slate-50 transition-all"
                >
                  Siguiente
                </button>
              </div>
            </div>
          </motion.div>
        )}
      </section>

      {/* CTA final académico */}
      <section className="bg-gradient-to-b from-white to-slate-50 border-t-2 border-slate-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16 sm:py-20">
          <div className="max-w-3xl mx-auto text-center">
            <motion.div initial={{ opacity: 0, y: 20 }} whileInView={{ opacity: 1, y: 0 }} viewport={{ once: true }} className="mb-8">
              <h2 className="text-2xl sm:text-3xl font-serif font-bold text-slate-900 mb-4">
                ¿Tiene un Proyecto de Investigación?
              </h2>
              <div className="w-24 h-1 bg-blue-900 mx-auto mb-6" />
              <p className="text-lg text-slate-600 font-light">
                Forme parte de nuestra comunidad académica y comparta su investigación con la institución.
              </p>
            </motion.div>

            <motion.div
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ delay: 0.1 }}
              className="flex flex-col sm:flex-row items-center justify-center gap-4"
            >
              <Link
                href="/login"
                className="inline-flex items-center gap-2 px-8 py-3.5 bg-blue-900 text-white font-medium hover:bg-blue-800 transition-all shadow-sm"
              >
                Acceso Institucional
                <FiArrowRight className="w-4 h-4" />
              </Link>
              <Link
                href="/contacto"
                className="inline-flex items-center gap-2 px-8 py-3.5 border-2 border-slate-300 text-slate-700 font-medium hover:border-blue-900 hover:bg-slate-50 transition-all"
              >
                Más Información
              </Link>
            </motion.div>
          </div>
        </div>
      </section>
    </PublicLayout>
  );
}
