import React, { useEffect, useState, useMemo } from "react";
import { Head, Link, usePage } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import { Menu, X, ChevronRight } from "lucide-react";

/* ======================================
   Enlace de navegación (NavLink)
   ====================================== */
function NavLink({ href, children, onClick }) {
  const { url } = usePage();
  const active = url === href || (href !== "/" && url.startsWith(href));

  return (
    <Link
      href={href}
      onClick={onClick}
      aria-current={active ? "page" : undefined}
      className={[
        "relative inline-flex items-center gap-2 px-3 py-2 rounded-md text-sm font-medium transition",
        "focus:outline-none focus-visible:ring-2 focus-visible:ring-primary-500/60",
        active
          ? "bg-primary-700 text-white shadow-sm"
          : "text-secondary-700 hover:text-secondary-700 hover:bg-secondary-50",
        "dark:text-gray-200 dark:hover:text-white dark:hover:bg-white/10",
      ].join(" ")}
    >
      {children}

      {/* Línea inferior animada si está activo */}
      <motion.span
        layoutId="nav-underline"
        className={[
          "absolute left-3 right-3 -bottom-0.5 h-[2px] rounded",
          active ? "bg-white/80 dark:bg-primary-300" : "bg-transparent",
        ].join(" ")}
      />
    </Link>
  );
}

/* ======================================
   Layout Público
   ====================================== */
export default function PublicLayout({ title = "Portal de Proyectos", children, nav }) {
  const year = new Date().getFullYear();

  const defaultNav = useMemo(
    () => [
      { label: "Inicio", href: "/" },
      { label: "Proyectos", href: "/proyectos" },
      { label: "Acceso", href: "/login" },
    ],
    []
  );

  const links = nav && nav.length ? nav : defaultNav;

  const [mobileOpen, setMobileOpen] = useState(false);
  const [scrolled, setScrolled] = useState(false);

  // Efecto de sombra al hacer scroll
  useEffect(() => {
    const onScroll = () => setScrolled(window.scrollY > 8);
    onScroll();
    window.addEventListener("scroll", onScroll, { passive: true });
    return () => window.removeEventListener("scroll", onScroll);
  }, []);

  // Cerrar menú móvil al navegar
  const handleItemClick = () => setMobileOpen(false);

  return (
    <>
      <Head title={title} />

      {/* Acceso rápido al contenido */}
      <a
        href="#main"
        className="sr-only focus:not-sr-only focus:fixed focus:top-3 focus:left-3 focus:z-[100] focus:px-3 focus:py-2 focus:rounded-md focus:bg-primary-700 focus:text-white"
      >
        Saltar al contenido
      </a>

      {/* HEADER */}
      <header
        className={[
          "fixed top-0 inset-x-0 z-40 transition-all",
          "backdrop-blur supports-[backdrop-filter]:bg-white/70 bg-white/90",
          scrolled ? "shadow-md" : "shadow-sm",
          "dark:bg-zinc-900/80 dark:supports-[backdrop-filter]:bg-zinc-900/60",
        ].join(" ")}
      >
        <div className="max-w-7xl mx-auto px-4 sm:px-6">
          <div className="h-16 flex items-center justify-between gap-4">
            {/* Logo y título */}
            <Link href="/" className="flex items-center gap-3 group">
              <div className="h-9 w-9 rounded-lg bg-primary-700 shadow-sm group-hover:scale-105 transition-transform" />
              <div className="leading-tight">
                <p className="text-base font-bold text-primary-700 dark:text-white">
                  Portal de Proyectos
                </p>
                <p className="text-[11px] text-secondary-700 dark:text-gray-400">
                  Investigación · Innovación · Impacto
                </p>
              </div>
            </Link>

            {/* Navegación escritorio */}
            <nav className="hidden md:flex items-center gap-1">
              {links.map((item) => (
                <NavLink key={item.href} href={item.href}>
                  {item.label}
                </NavLink>
              ))}
            </nav>

            {/* Botones de acción */}
            <div className="flex items-center gap-2">
              <Link
                href="/proyectos"
                className="hidden sm:inline-flex items-center gap-2 rounded-md px-3 py-2 text-sm font-semibold text-white bg-primary-700 hover:bg-primary-800 transition shadow-sm"
              >
                Ver proyectos <ChevronRight className="w-4 h-4" />
              </Link>

              <button
                type="button"
                className="md:hidden inline-flex items-center justify-center rounded-md p-2 text-secondary-700 hover:bg-secondary-50 hover:text-secondary-700 focus:outline-none focus-visible:ring-2 focus-visible:ring-primary-500 dark:text-gray-200 dark:hover:bg-white/10"
                aria-label={mobileOpen ? "Cerrar menú" : "Abrir menú"}
                onClick={() => setMobileOpen((v) => !v)}
              >
                {mobileOpen ? <X className="w-6 h-6" /> : <Menu className="w-6 h-6" />}
              </button>
            </div>
          </div>
        </div>

        {/* Menú móvil */}
        <AnimatePresence>
          {mobileOpen && (
            <motion.nav
              key="mobile-nav"
              initial={{ height: 0, opacity: 0 }}
              animate={{ height: "auto", opacity: 1 }}
              exit={{ height: 0, opacity: 0 }}
              transition={{ duration: 0.25 }}
              className="md:hidden border-t border-secondary-200 dark:border-white/10"
            >
              <div className="px-4 sm:px-6 py-2 space-y-1 bg-white/95 dark:bg-zinc-900/90 backdrop-blur">
                {links.map((item) => (
                  <NavLink key={item.href} href={item.href} onClick={handleItemClick}>
                    {item.label}
                  </NavLink>
                ))}
              </div>
            </motion.nav>
          )}
        </AnimatePresence>
      </header>

      {/* CONTENIDO PRINCIPAL */}
      <main id="main" className="pt-20 sm:pt-24  bg-white dark:bg-zinc-950">
        <motion.div initial={{ opacity: 0, y: 12 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.35 }}>
          {children}
        </motion.div>
      </main>

      {/* FOOTER */}
      <footer className="bg-primary-700 text-white mt-20">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 py-12 grid sm:grid-cols-2 lg:grid-cols-4 gap-8">
          <div>
            <h3 className="text-lg font-semibold mb-3">Portal de Proyectos</h3>
            <p className="text-sm text-white/80">
              Una vitrina académica para conocer iniciativas, objetivos e impactos de investigación.
            </p>
          </div>

          <div>
            <h3 className="text-lg font-semibold mb-3">Navegación</h3>
            <ul className="text-sm space-y-1">
              {links.map((l) => (
                <li key={`f-${l.href}`}>
                  <Link
                    href={l.href}
                    className="hover:opacity-90 inline-flex items-center gap-1 text-secondary-200 hover:text-secondary-100"
                  >
                    {l.label}
                  </Link>
                </li>
              ))}
            </ul>
          </div>

          <div>
            <h3 className="text-lg font-semibold mb-3">Contacto</h3>
            <p className="text-sm text-white/80">info@proyectos.edu.ec</p>
            <p className="text-sm text-white/80">Quito · Ecuador</p>
          </div>

          <div>
            <h3 className="text-lg font-semibold mb-3">Acerca de</h3>
            <p className="text-sm text-white/80">
              Transparencia, ciencia abierta y vinculación con la sociedad.
            </p>
          </div>
        </div>

        <div className="border-t border-secondary-300/30">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 py-6 text-center text-xs text-white/70">
            © {year} Portal de Proyectos. Todos los derechos reservados.
          </div>
        </div>
      </footer>
    </>
  );
}
