// resources/js/app.jsx
import React, { useState, useEffect } from 'react';
import './bootstrap';
import '../css/app.css';

import { createRoot } from 'react-dom/client';
import { createInertiaApp } from '@inertiajs/react';
import { resolvePageComponent } from 'laravel-vite-plugin/inertia-helpers';
import { Inertia } from '@inertiajs/inertia';
import { AnimatePresence, motion } from 'framer-motion';

const appName = import.meta.env.VITE_APP_NAME || 'Laravel';

/* =========================
   Loader PRO con Framer Motion
   ========================= */
function Loader() {
  return (
    <motion.div
      className="fixed inset-0 z-[9999] flex items-center justify-center
                 bg-gradient-to-br from-primary-50 via-secondary-50 to-primary-100
                 dark:from-ink/80 dark:via-ink/90 dark:to-ink"
      initial={{ opacity: 0 }}
      animate={{ opacity: 1 }}
      exit={{ opacity: 0 }}
      aria-busy="true"
      aria-live="polite"
      role="status"
    >
      {/* Tarjeta central */}
      <motion.div
        initial={{ scale: 0.98, y: 10, opacity: 0 }}
        animate={{ scale: 1, y: 0, opacity: 1 }}
        exit={{ scale: 0.98, y: -10, opacity: 0 }}
        transition={{ type: 'spring', stiffness: 120, damping: 18 }}
        className="w-[min(92vw,460px)] rounded-2xl bg-white/85 dark:bg-ink/70
                   backdrop-blur-xl shadow-xl border border-primary-100/60 dark:border-ink/60 p-6"
      >
        {/* Logo / Marca */}
        <div className="flex items-center gap-3 mb-5">
          <div className="relative h-12 w-12">
            {/* Spinner con brillo */}
            <motion.div
              className="absolute inset-0 rounded-full border-4
                         border-primary-300 dark:border-primary-500
                         border-t-transparent"
              animate={{ rotate: 360 }}
              transition={{ repeat: Infinity, duration: 1.2, ease: 'linear' }}
            />
            <motion.div
              className="absolute inset-2 rounded-full bg-gradient-to-br
                         from-primary-200 to-secondary-200
                         dark:from-primary-600 dark:to-secondary-600"
              animate={{ scale: [1, 1.06, 1] }}
              transition={{ repeat: Infinity, duration: 1.6, ease: 'easeInOut' }}
            />
          </div>
          <div>
            <p className="text-sm uppercase tracking-wider text-primary-600 dark:text-primary-300">
              Cargando
            </p>
            <h2 className="text-xl font-semibold text-ink dark:text-white">
              {appName}
            </h2>
          </div>
        </div>

        {/* Barra de progreso */}
        <div className="w-full h-2.5 rounded-full bg-primary-100/70 dark:bg-ink/60 overflow-hidden">
          <motion.div
            className="h-full rounded-full bg-gradient-to-r from-primary-500 via-secondary-500 to-primary-600"
            initial={{ x: '-100%' }}
            animate={{ x: ['-100%', '0%', '100%'] }}
            transition={{ duration: 1.8, repeat: Infinity, ease: 'easeInOut' }}
          />
        </div>

        {/* Texto de apoyo */}
        <div className="mt-3 flex items-center justify-between text-xs text-subInk dark:text-subInk/80">
          <span>Preparando interfaz</span>
          <span className="font-medium text-primary-700 dark:text-primary-300">Un momento…</span>
        </div>
      </motion.div>
    </motion.div>
  );
}

/* =========================
   App Wrapper con transiciones
   ========================= */
function AppWrapper({ App, props }) {
  const [loading, setLoading] = useState(true);

  // Splash inicial breve
  useEffect(() => {
    const timer = setTimeout(() => setLoading(false), 900);
    return () => clearTimeout(timer);
  }, []);

  // Loader en navegación de Inertia
  useEffect(() => {
    const start = () => setLoading(true);
    const finish = () => setTimeout(() => setLoading(false), 300);

    Inertia.on('start', start);
    Inertia.on('finish', finish);
    return () => {
      Inertia.off('start', start);
      Inertia.off('finish', finish);
    };
  }, []);

  return (
    <>
      <AnimatePresence mode="wait">{loading && <Loader />}</AnimatePresence>
      <App {...props} />
    </>
  );
}

createInertiaApp({
  title: (title) => `${title} - ${appName}`,
  resolve: (name) =>
    resolvePageComponent(`./Pages/${name}.jsx`, import.meta.glob('./Pages/**/*.jsx')),
  setup({ el, App, props }) {
    const root = createRoot(el);
    root.render(<AppWrapper App={App} props={props} />);
  },
  // Color de la barra de progreso de Inertia (usa tu primary)
  progress: {
    color: '#2563EB', // ej. primary-600 (ajústalo a tu paleta)
  },
});
