// resources/js/Pages/Landing/VidaEstudiantil/TIC/Projects/Index.jsx
import React, { useEffect, useMemo, useRef, useState } from "react";
import { router, usePage } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import {
  Network, Shield, Code, Laptop, Cpu, Wifi, Database, Globe,
  Video, Lock, Server, Radio, Search, X, Filter, ChevronDown,
  Calendar, User, BookOpen, Target, Sparkles, Loader2
} from "lucide-react";
import PublicLayout from "@/Layouts/PublicLayout";

/* ========= THEME HELPERS ========= */
const cx = (...xs) => xs.filter(Boolean).join(" ");
const GRADIENT_PS = "from-primary-600 to-secondary-600";
const GRADIENT_PS_SOFT = "from-primary-500 to-secondary-500";

/* Imagen del HERO */
const HERO_IMAGE = "/assets/images/fondo_tic.png";

/* Debounce */
const useDebounced = (value, ms = 350) => {
  const [v, setV] = useState(value);
  useEffect(() => {
    const t = setTimeout(() => setV(value), ms);
    return () => clearTimeout(t);
  }, [value, ms]);
  return v;
};

const areaIcons = {
  "Redes de Datos": <Network className="w-4 h-4" />,
  Ciberseguridad: <Shield className="w-4 h-4" />,
  "Protocolos de Comunicación": <Code className="w-4 h-4" />,
  "Infraestructura de Red": <Cpu className="w-4 h-4" />,
  "Comunicaciones Satelitales": <Laptop className="w-4 h-4" />,
  "Redes 5G": <Wifi className="w-4 h-4" />,
  "Edge Computing": <Server className="w-4 h-4" />,
  Blockchain: <Database className="w-4 h-4" />,
  "Internet de las Cosas": <Globe className="w-4 h-4" />,
  "Streaming de Video": <Video className="w-4 h-4" />,
  "Inteligencia Artificial": <Sparkles className="w-4 h-4" />,
  "Fibra Óptica": <Radio className="w-4 h-4" />,
  "Diseño de Antenas": <Radio className="w-4 h-4" />,
  "Análisis de Vulnerabilidades": <Lock className="w-4 h-4" />,
  "Computación en la Nube": <Server className="w-4 h-4" />,
  "Arquitectura Distribuida": <Network className="w-4 h-4" />,
  "Análisis de Rendimiento": <Target className="w-4 h-4" />,
  "Transmisión de Datos": <Radio className="w-4 h-4" />,
  "Redes Metropolitanas": <Globe className="w-4 h-4" />,
  "Redes de Distribución": <Network className="w-4 h-4" />,
  "Compresión de Datos": <Database className="w-4 h-4" />,
  "Monitoreo de Red": <Target className="w-4 h-4" />,
};

function ProjectCard({ project, index }) {
  return (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      exit={{ opacity: 0, scale: 0.9 }}
      transition={{ delay: index * 0.05 }}
      layout
      className="group bg-white rounded-2xl shadow-lg hover:shadow-2xl transition-all duration-300 overflow-hidden border border-gray-100"
    >
      {/* Header SIN imagen (solo gradiente primary/secondary) */}
      <div className={cx("relative h-20 bg-gradient-to-r", GRADIENT_PS)} />
      {/* Badges en header */}
      <div className="px-6 -mt-6 flex items-center justify-between">
        <div className="inline-flex items-center gap-2 bg-white/95 backdrop-blur-sm px-3 py-1 rounded-full text-xs font-semibold text-gray-700 shadow border border-gray-200">
          {project.carrera}
        </div>
        <div
          className="inline-flex items-center gap-1 text-white px-3 py-1 rounded-full text-xs font-bold shadow bg-gradient-to-r"
          style={{ backgroundImage: "linear-gradient(to right, var(--tw-gradient-from), var(--tw-gradient-to))" }}
        >
          <span className={cx("bg-gradient-to-r", GRADIENT_PS, "px-0 py-0 rounded-full hidden")} />
          <Calendar className="w-3 h-3" />
          {project.semestre}
        </div>
      </div>

      {/* Contenido */}
      <div className="p-6">
        <h3 className="text-lg sm:text-xl font-bold text-gray-900 mb-3 group-hover:text-primary-700 transition-colors line-clamp-2">
          {project.nombreProyecto}
        </h3>

        {/* Profesor */}
        <div className="flex items-center gap-2 text-sm text-gray-600 mb-4">
          <div className="p-1.5 bg-primary-50 rounded-lg border border-primary-100">
            <User className="w-4 h-4 text-primary-700" />
          </div>
          <span className="font-medium">{project.profesorCargo}</span>
        </div>

        {/* Áreas */}
        {Array.isArray(project.areasTrabajo) && project.areasTrabajo.length > 0 && (
          <div className="mb-4">
            <div className="flex items-center gap-2 text-sm font-semibold text-gray-700 mb-2">
              <Target className="w-4 h-4 text-secondary-700" />
              Áreas de Trabajo
            </div>
            <div className="flex flex-wrap gap-2">
              {project.areasTrabajo.slice(0, 3).map((area, idx) => (
                <div
                  key={idx}
                  className="flex items-center gap-1.5 bg-gradient-to-r from-gray-50 to-gray-100 px-3 py-1.5 rounded-lg text-xs font-medium text-gray-700 border border-gray-200"
                >
                  {areaIcons[area] || <Cpu className="w-3 h-3" />}
                  <span className="truncate max-w-[120px]">{area}</span>
                </div>
              ))}
              {project.areasTrabajo.length > 3 && (
                <div className="flex items-center px-3 py-1.5 bg-gray-100 rounded-lg text-xs font-bold text-gray-600">
                  +{project.areasTrabajo.length - 3}
                </div>
              )}
            </div>
          </div>
        )}

        {/* Resumen */}
        {project.resumen && (
          <p className="text-sm text-gray-600 mb-4 line-clamp-3 leading-relaxed">
            {project.resumen}
          </p>
        )}

        {/* Línea de investigación */}
        {project.lineaInvestigacion && (
          <div className="flex items-center gap-2 mb-4">
            <BookOpen className="w-4 h-4 text-gray-400" />
            <span className="text-xs font-semibold px-3 py-1 rounded-full bg-primary-50 text-primary-700 border border-primary-100">
              {project.lineaInvestigacion}
            </span>
          </div>
        )}

        {/* Asignaturas */}
        {Array.isArray(project.asignaturasRelacionadas) && project.asignaturasRelacionadas.length > 0 && (
          <div className="pt-4 border-t border-gray-100">
            <div className="flex flex-wrap gap-2">
              {project.asignaturasRelacionadas.map((asignatura, idx) => (
                <span
                  key={idx}
                  className="text-xs bg-secondary-50 text-secondary-700 px-2 py-1 rounded-md font-medium border border-secondary-100"
                >
                  {asignatura}
                </span>
              ))}
            </div>
          </div>
        )}
      </div>

      {/* Hover underline */}
      <div className={cx("h-1 bg-gradient-to-r transform scale-x-0 group-hover:scale-x-100 transition-transform duration-300 origin-left", GRADIENT_PS)} />
    </motion.div>
  );
}

const TicProjects = () => {
  const { props } = usePage();

  // Del backend:
  // - projects: paginado simple (simplePaginate) con .data, .current_page, .next_page_url
  // - filtros disponibles: carreras, profesores, semestres, asignaturas, lineasInvestigacion
  // - filters: estado actual de filtros (SSR)
  const {
    projects,
    carreras = [],
    profesores = [],
    semestres = [],
    asignaturas = [],
    lineasInvestigacion = [],
    filters = {},
    total_count, // opcional: total de proyectos publicados
  } = props;

  /* Estado de filtros (UI) */
  const [filtros, setFiltros] = useState({
    carrera: filters.carrera || "",
    profesorCargo: filters.profesorCargo || "",
    semestre: filters.semestre || "",
    asignaturaRelacionada: filters.asignaturaRelacionada || "",
    lineaInvestigacion: filters.lineaInvestigacion || "",
  });
  const [searchTerm, setSearchTerm] = useState(filters.q || "");
  const [showFilters, setShowFilters] = useState(true);

  /* Estado de lista y paginación */
  const [items, setItems] = useState(projects?.data || []);
  const [nextUrl, setNextUrl] = useState(projects?.next_page_url || null);
  const [loadingMore, setLoadingMore] = useState(false);
  const [loadingFilter, setLoadingFilter] = useState(false);

  /* Debounce */
  const dCarrera  = useDebounced(filtros.carrera);
  const dProfesor = useDebounced(filtros.profesorCargo);
  const dSemestre = useDebounced(filtros.semestre);
  const dAsig     = useDebounced(filtros.asignaturaRelacionada);
  const dLinea    = useDebounced(filtros.lineaInvestigacion);
  const dQ        = useDebounced(searchTerm);

  /* Evitar navegación en primer render */
  const firstRunRef = useRef(true);

  /* Aplicar data nueva cuando cambie la respuesta del backend */
  useEffect(() => {
    const current = projects?.current_page || 1;
    const data    = projects?.data || [];
    const next    = projects?.next_page_url || null;

    if (current === 1) setItems(data);
    else setItems(prev => [...prev, ...data]);

    setNextUrl(next);
    setLoadingMore(false);
    setLoadingFilter(false);
  }, [projects?.current_page, projects?.next_page_url]);

  /* Efecto: navegar cuando cambien filtros (comparando con props.filters para evitar loops) */
  useEffect(() => {
    if (firstRunRef.current) {
      firstRunRef.current = false;
      setLoadingFilter(false);
      return;
    }

    const same =
      (filters.carrera || "") === (dCarrera || "") &&
      (filters.profesorCargo || "") === (dProfesor || "") &&
      (filters.semestre || "") === (dSemestre || "") &&
      (filters.asignaturaRelacionada || "") === (dAsig || "") &&
      (filters.lineaInvestigacion || "") === (dLinea || "") &&
      (filters.q || "") === (dQ || "");

    if (same) {
      setLoadingFilter(false);
      return;
    }

    setLoadingFilter(true);
    router.get(
      route("tic.projects.index"),
      {
        carrera: dCarrera || undefined,
        profesorCargo: dProfesor || undefined,
        semestre: dSemestre || undefined,
        asignaturaRelacionada: dAsig || undefined,
        lineaInvestigacion: dLinea || undefined,
        q: dQ || undefined,
        page: 1,
      },
      {
        preserveState: true,
        preserveScroll: true,
        replace: true,
        only: ["projects", "filters"], // rehidrata solo lo necesario
      }
    );
  }, [dCarrera, dProfesor, dSemestre, dAsig, dLinea, dQ]);

  /* Handlers */
  const clearAllFilters = () => {
    setFiltros({
      carrera: "",
      profesorCargo: "",
      semestre: "",
      asignaturaRelacionada: "",
      lineaInvestigacion: "",
    });
    setSearchTerm("");
  };

  const hasActiveFilters =
    Object.values(filtros).some((v) => v !== "") || searchTerm !== "";

  const handleLoadMore = () => {
    if (!nextUrl || loadingMore) return;
    setLoadingMore(true);
    router.visit(nextUrl, {
      preserveState: true,
      preserveScroll: true,
      replace: true,
      only: ["projects"],
    });
  };

  const totalMostrados = items.length;
  const totalBackend   = typeof total_count === "number" ? total_count : (projects?.total ?? totalMostrados);

  return (
    <PublicLayout titulo={"TIC · Proyectos de Integración Curricular"}>
      <div className="min-h-screen bg-gradient-to-br from-gray-50 via-white to-gray-50">
        {/* HERO */}
        <section className="relative overflow-hidden">
          <img
            src={HERO_IMAGE}
            alt="TIC DETRI"
            className="absolute inset-0 h-full w-full object-cover"
            loading="eager"
          />
          <div className="absolute inset-0 bg-black/45" />
          <div className={cx("absolute inset-0 bg-gradient-to-br mix-blend-multiply opacity-60", GRADIENT_PS)} />
          <div className="relative max-w-7xl mx-auto px-6 py-20 sm:py-24">
            <motion.div initial={{ opacity: 0, y: 30 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.8 }} className="text-center">
              <motion.div initial={{ scale: 0 }} animate={{ scale: 1 }} transition={{ delay: 0.2, type: "spring" }}
                className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-white/15 backdrop-blur-sm text-white text-sm font-semibold mb-6 border border-white/25">
                <Sparkles className="w-4 h-4" />
                DETRI · EPN
              </motion.div>

              <h1 className="text-4xl sm:text-5xl lg:text-6xl font-black text-white mb-6">
                Proyectos de{" "}
                <span className="bg-clip-text text-transparent bg-gradient-to-r from-white to-white/70">
                  Integración Curricular
                </span>
              </h1>

              <p className="text-lg sm:text-xl text-white/85 max-w-3xl mx-auto">
                Explora los proyectos innovadores desarrollados por nuestros estudiantes en Telecomunicaciones y Redes de Información.
              </p>

              {/* Buscador */}
              <motion.div initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.4 }} className="mt-10 max-w-2xl mx-auto">
                <div className="relative">
                  <Search className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                  <input
                    type="text"
                    placeholder="Buscar proyectos, profesores o temas..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="w-full pl-12 pr-12 py-4 rounded-2xl bg-white shadow-2xl text-gray-900 placeholder-gray-400 focus:outline-none focus:ring-4 focus:ring-primary-300/40"
                  />
                  {searchTerm && (
                    <button
                      onClick={() => setSearchTerm("")}
                      className="absolute right-4 top-1/2 -translate-y-1/2 p-1 hover:bg-gray-100 rounded-full transition-colors"
                    >
                      <X className="w-5 h-5 text-gray-400" />
                    </button>
                  )}
                </div>
              </motion.div>
            </motion.div>
          </div>
        </section>

        <div className="max-w-7xl mx-auto px-6 py-10">
          {/* FILTROS */}
          <motion.div initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} className="bg-white rounded-2xl shadow-xl p-6 mb-8">
            <div className="flex items-center justify-between mb-4">
              <button
                onClick={() => setShowFilters(!showFilters)}
                className="flex items-center gap-2 text-lg font-bold text-gray-900 hover:text-primary-600 transition-colors"
              >
                <Filter className="w-5 h-5" />
                Filtros Avanzados
                <ChevronDown className={`w-5 h-5 transition-transform ${showFilters ? "rotate-180" : ""}`} />
              </button>

              {hasActiveFilters && (
                <motion.button
                  initial={{ scale: 0 }}
                  animate={{ scale: 1 }}
                  onClick={clearAllFilters}
                  className="flex items-center gap-2 px-4 py-2 bg-primary-50 text-primary-700 rounded-lg hover:bg-primary-100 transition-colors text-sm font-semibold border border-primary-200"
                >
                  <X className="w-4 h-4" />
                  Limpiar Filtros
                </motion.button>
              )}
            </div>

            <AnimatePresence>
              {showFilters && (
                <motion.div
                  initial={{ height: 0, opacity: 0 }}
                  animate={{ height: "auto", opacity: 1 }}
                  exit={{ height: 0, opacity: 0 }}
                  transition={{ duration: 0.3 }}
                  className="overflow-hidden"
                >
                  <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-4 pt-4">
                    <div>
                      <label className="block text-sm font-semibold text-gray-700 mb-2">Carrera</label>
                      <select
                        className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-primary-500 transition-colors bg-gray-50"
                        value={filtros.carrera}
                        onChange={(e) => setFiltros({ ...filtros, carrera: e.target.value })}
                      >
                        <option value="">Todas las Carreras</option>
                        {carreras.map((c) => (
                          <option key={c} value={c}>{c}</option>
                        ))}
                      </select>
                    </div>

                    <div>
                      <label className="block text-sm font-semibold text-gray-700 mb-2">Profesor</label>
                      <select
                        className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-primary-500 transition-colors bg-gray-50"
                        value={filtros.profesorCargo}
                        onChange={(e) => setFiltros({ ...filtros, profesorCargo: e.target.value })}
                      >
                        <option value="">Todos los Profesores</option>
                        {profesores.map((p) => (
                          <option key={p} value={p}>{p}</option>
                        ))}
                      </select>
                    </div>

                    <div>
                      <label className="block text-sm font-semibold text-gray-700 mb-2">Semestre</label>
                      <select
                        className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-primary-500 transition-colors bg-gray-50"
                        value={filtros.semestre}
                        onChange={(e) => setFiltros({ ...filtros, semestre: e.target.value })}
                      >
                        <option value="">Todos los Semestres</option>
                        {semestres.map((s) => (
                          <option key={s} value={s}>{s}</option>
                        ))}
                      </select>
                    </div>

                    <div>
                      <label className="block text-sm font-semibold text-gray-700 mb-2">Asignatura</label>
                      <select
                        className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-primary-500 transition-colors bg-gray-50"
                        value={filtros.asignaturaRelacionada}
                        onChange={(e) => setFiltros({ ...filtros, asignaturaRelacionada: e.target.value })}
                      >
                        <option value="">Todas las Asignaturas</option>
                        {asignaturas.map((a) => (
                          <option key={a} value={a}>{a}</option>
                        ))}
                      </select>
                    </div>

                    <div className="md:col-span-2">
                      <label className="block text-sm font-semibold text-gray-700 mb-2">Línea de Investigación</label>
                      <select
                        className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-primary-500 transition-colors bg-gray-50"
                        value={filtros.lineaInvestigacion}
                        onChange={(e) => setFiltros({ ...filtros, lineaInvestigacion: e.target.value })}
                      >
                        <option value="">Todas las Líneas</option>
                        {lineasInvestigacion.map((l) => (
                          <option key={l} value={l}>{l}</option>
                        ))}
                      </select>
                    </div>
                  </div>
                </motion.div>
              )}
            </AnimatePresence>
          </motion.div>

          {/* CONTADOR */}
          <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} className="mb-6 flex items-center justify-between">
            <p className="text-gray-700 font-medium">
              Mostrando{" "}
              <span className="text-primary-700 font-bold">{totalMostrados}</span>{" "}
              de <span className="font-bold">{totalBackend}</span> proyectos
            </p>
            {loadingFilter && (
              <span className="inline-flex items-center gap-2 text-sm text-primary-700">
                <Loader2 className="w-4 h-4 animate-spin" /> Aplicando filtros…
              </span>
            )}
          </motion.div>

          {/* GRID DE PROYECTOS */}
          <AnimatePresence mode="popLayout">
            {items.length > 0 ? (
              <>
                <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
                  {items.map((project, index) => (
                    <ProjectCard key={project.id} project={project} index={index} />
                  ))}
                </div>

                {/* Ver más */}
                <div className="mt-10 flex justify-center">
                  {nextUrl ? (
                    <button
                      onClick={handleLoadMore}
                      disabled={loadingMore}
                      className="inline-flex items-center gap-2 px-6 py-3 bg-gradient-to-r from-primary-600 to-secondary-600 text-white rounded-xl hover:brightness-110 transition-colors font-semibold shadow disabled:opacity-60"
                    >
                      {loadingMore && <Loader2 className="w-4 h-4 animate-spin" />}
                      {loadingMore ? "Cargando…" : "Ver más"}
                    </button>
                  ) : (
                    <span className="text-sm text-primary-600">No hay más resultados</span>
                  )}
                </div>
              </>
            ) : (
              <motion.div initial={{ opacity: 0, scale: 0.9 }} animate={{ opacity: 1, scale: 1 }} className="text-center py-20">
                <div className="inline-flex items-center justify-center w-20 h-20 bg-primary-50 rounded-full mb-4 border border-primary-100">
                  <Search className="w-10 h-10 text-primary-600" />
                </div>
                <h3 className="text-2xl font-bold text-gray-900 mb-2">No se encontraron proyectos</h3>
                <p className="text-gray-600 mb-6">Intenta ajustar tus filtros de búsqueda</p>
                <button
                  onClick={clearAllFilters}
                  className="px-6 py-3 bg-gradient-to-r from-primary-600 to-secondary-600 text-white rounded-xl hover:brightness-110 transition-colors font-semibold shadow"
                >
                  Limpiar todos los filtros
                </button>
              </motion.div>
            )}
          </AnimatePresence>
        </div>
      </div>
    </PublicLayout>
  );
};

export default TicProjects;
