import React, { useEffect, useMemo, useRef, useState } from "react";
import { router, usePage } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import {
  FiBriefcase, FiMapPin, FiClock, FiCalendar, FiUsers,
  FiDollarSign, FiCheckCircle, FiAlertCircle, FiFilter,
  FiAward, FiTrendingUp, FiLoader, FiSearch
} from "react-icons/fi";
import PublicLayout from "@/Layouts/PublicLayout";

/* ============== Helpers ============== */
const cls = (...xs) => xs.filter(Boolean).join(" ");
const useDebounced = (value, ms = 350) => {
  const [v, setV] = useState(value);
  useEffect(() => { const t = setTimeout(() => setV(value), ms); return () => clearTimeout(t); }, [value, ms]);
  return v;
};

/* ============== Card ============== */
function PasantiaCard({ p, index }) {
  const [expanded, setExpanded] = useState(false);

  return (
    <motion.article
      initial={{ opacity: 0, y: 30 }}
      whileInView={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.5, delay: index * 0.06 }}
      viewport={{ once: true, amount: 0.2 }}
      className="group relative"
    >
      <div className="relative h-full rounded-2xl border-2 border-primary-100 dark:border-primary-800 bg-white dark:bg-primary-950 shadow-lg hover:shadow-2xl transition-all duration-500 overflow-hidden hover:border-primary-200 dark:hover:border-primary-700">
        {/* Barra superior fija con tema primary */}
        <div className="h-1.5 bg-gradient-to-r from-primary-600 to-secondary-600" />

        <div className="p-6 pb-4">
          <div className="flex items-start justify-between gap-4 mb-4">
            <div className="flex-1">
              <div className="flex flex-wrap items-center gap-2 mb-3">
                {p.remunerado && (
                  <span className="inline-flex items-center gap-1.5 px-3 py-1 rounded-full bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-400 border border-green-200 dark:border-green-800 text-xs font-bold">
                    <FiDollarSign className="w-3 h-3" />
                    Remunerado
                  </span>
                )}
                <span className="px-3 py-1 rounded-full text-xs font-bold border bg-primary-100 text-primary-700 border-primary-200">
                  {p.tipo || "Pasantía"}
                </span>
              </div>

              <h3 className="text-xl font-bold text-primary-900 dark:text-white mb-2 group-hover:text-primary-700 dark:group-hover:text-primary-300 transition-colors duration-300">
                {p.titulo}
              </h3>

              <p className="text-lg font-semibold text-primary-700 dark:text-primary-300">
                {p.empresa}
              </p>
            </div>

            <div className="p-3 rounded-xl bg-gradient-to-br from-primary-600 to-secondary-600 shadow-lg group-hover:scale-110 transition-transform duration-300">
              <FiBriefcase className="w-6 h-6 text-white" />
            </div>
          </div>

          <div className="grid grid-cols-2 gap-3 mb-4">
            <div className="flex items-center gap-2 text-sm text-primary-700 dark:text-primary-300">
              <FiMapPin className="w-4 h-4 text-primary-600" />
              <span className="font-medium">{p.modalidad}</span>
            </div>
            <div className="flex items-center gap-2 text-sm text-primary-700 dark:text-primary-300">
              <FiClock className="w-4 h-4 text-primary-600" />
              <span className="font-medium">{p.duracion}</span>
            </div>
            <div className="flex items-center gap-2 text-sm text-primary-700 dark:text-primary-300">
              <FiUsers className="w-4 h-4 text-primary-600" />
              <span className="font-medium">{p.cupos} cupos</span>
            </div>
            <div className="flex items-center gap-2 text-sm text-primary-700 dark:text-primary-300">
              <FiCalendar className="w-4 h-4 text-primary-600" />
              <span className="font-medium">{p.fecha_inicio_legible}</span>
            </div>
          </div>

          {p.carrera && (
            <div className="inline-flex items-center gap-2 px-3 py-2 rounded-lg bg-primary-50 dark:bg-primary-900/30 border border-primary-200 dark:border-primary-800">
              <FiAward className="w-4 h-4 text-primary-600" />
              <span className="text-sm font-semibold text-primary-800 dark:text-primary-200">
                {p.carrera}
              </span>
            </div>
          )}
        </div>

        <AnimatePresence>
          {expanded && (
            <motion.div
              initial={{ height: 0, opacity: 0 }}
              animate={{ height: "auto", opacity: 1 }}
              exit={{ height: 0, opacity: 0 }}
              transition={{ duration: 0.3 }}
              className="overflow-hidden"
            >
              <div className="px-6 pb-4 space-y-4">
                {Array.isArray(p.requisitos) && p.requisitos.length > 0 && (
                  <div>
                    <h4 className="text-sm font-bold text-primary-900 dark:text-white mb-2 flex items-center gap-2">
                      <FiCheckCircle className="text-primary-600" />
                      Requisitos
                    </h4>
                    <ul className="space-y-2">
                      {p.requisitos.map((req, idx) => (
                        <motion.li
                          key={idx}
                          initial={{ opacity: 0, x: -10 }}
                          animate={{ opacity: 1, x: 0 }}
                          transition={{ delay: idx * 0.05 }}
                          className="flex items-start gap-2 text-sm text-primary-700 dark:text-primary-300"
                        >
                          <span className="mt-1 w-1.5 h-1.5 rounded-full bg-primary-600" />
                          <span>{req}</span>
                        </motion.li>
                      ))}
                    </ul>
                  </div>
                )}

                {Array.isArray(p.beneficios) && p.beneficios.length > 0 && (
                  <div>
                    <h4 className="text-sm font-bold text-primary-900 dark:text-white mb-2 flex items-center gap-2">
                      <FiTrendingUp className="text-primary-600" />
                      Beneficios
                    </h4>
                    <ul className="space-y-2">
                      {p.beneficios.map((ben, idx) => (
                        <motion.li
                          key={idx}
                          initial={{ opacity: 0, x: -10 }}
                          animate={{ opacity: 1, x: 0 }}
                          transition={{ delay: idx * 0.05 }}
                          className="flex items-start gap-2 text-sm text-primary-700 dark:text-primary-300"
                        >
                          <span className="mt-1 w-1.5 h-1.5 rounded-full bg-primary-600" />
                          <span>{ben}</span>
                        </motion.li>
                      ))}
                    </ul>
                  </div>
                )}

                {p.fecha_maxima_postulacion_legible && (
                  <div className="flex items-start gap-3 p-4 rounded-xl bg-amber-50 dark:bg-amber-900/20 border border-amber-200 dark:border-amber-800">
                    <FiAlertCircle className="w-5 h-5 text-amber-600 dark:text-amber-400 flex-shrink-0 mt-0.5" />
                    <div>
                      <p className="text-sm font-semibold text-amber-900 dark:text-amber-200">Fecha máxima de postulación</p>
                      <p className="text-sm text-amber-700 dark:text-amber-300 mt-1">
                        {p.fecha_maxima_postulacion_legible}
                      </p>
                    </div>
                  </div>
                )}
              </div>
            </motion.div>
          )}
        </AnimatePresence>

        <div className="px-6 pb-6 pt-2 flex">
          <button
            onClick={() => setExpanded(!expanded)}
            className="w-full px-4 py-3 rounded-xl font-semibold text-sm border-2 border-primary-200 dark:border-primary-800 text-primary-700 dark:text-primary-300 hover:bg-primary-50 dark:hover:bg-primary-900/30 transition-all duration-300 hover:scale-105 active:scale-95"
          >
            {expanded ? "Ver menos" : "Ver detalles"}
          </button>
        </div>
      </div>
    </motion.article>
  );
}

/* ============== Página ============== */
export default function Index() {
  const { props } = usePage();
  const { pasantias, modalidades = ["Todas","Remoto","Presencial","Híbrido"], carreras = ["Todas"], filters = {}, empresas_count } = props;

  // Estado de filtros (inicializa con props.filters)
  const [filtroModalidad, setFiltroModalidad] = useState(filters.modalidad || "Todas");
  const [filtroCarrera, setFiltroCarrera]     = useState(filters.carrera   || "Todas");
  const [q, setQ]                             = useState(filters.q || "");

  // Datos locales para la lista y paginación "Ver más"
  const [items, setItems] = useState(pasantias?.data || []);
  const [nextUrl, setNextUrl] = useState(pasantias?.next_page_url || null);
  const [loadingMore, setLoadingMore] = useState(false);
  const [loadingFilter, setLoadingFilter] = useState(false);

  // Debounce para evitar spam de requests
  const dModalidad = useDebounced(filtroModalidad);
  const dCarrera   = useDebounced(filtroCarrera);
  const dQ         = useDebounced(q);

  // Ref para saltar el primer render (ya tienes datos SSR)
  const firstRunRef = useRef(true);

  // Efecto: aplicar nueva data cuando cambie la paginación/consulta desde el backend
  useEffect(() => {
    const current = pasantias?.current_page || 1;
    const data    = pasantias?.data || [];
    const next    = pasantias?.next_page_url || null;

    if (current === 1) setItems(data);
    else setItems(prev => [...prev, ...data]);

    setNextUrl(next);
    setLoadingMore(false);
    setLoadingFilter(false);
    // Dependemos de page y next_url; Inertia los cambia en cada respuesta
  }, [pasantias?.current_page, pasantias?.next_page_url]);

  // Efecto: navegación cuando cambien filtros (evita disparar en primer render y cuando no hay cambios reales)
  useEffect(() => {
    if (firstRunRef.current) {
      firstRunRef.current = false;
      setLoadingFilter(false);
      return;
    }

    const sameModalidad = (filters.modalidad || "Todas") === (dModalidad || "Todas");
    const sameCarrera   = (filters.carrera   || "Todas") === (dCarrera   || "Todas");
    const sameQ         = (filters.q         || "")      === (dQ         || "");

    if (sameModalidad && sameCarrera && sameQ) {
      setLoadingFilter(false);
      return;
    }

    setLoadingFilter(true);
    router.get(
      route("vida-estudiantil.pasantias"),
      {
        modalidad: dModalidad !== "Todas" ? dModalidad : undefined,
        carrera:   dCarrera   !== "Todas" ? dCarrera   : undefined,
        q: dQ || undefined,
        page: 1,
      },
      {
        preserveState: true,
        preserveScroll: true,
        replace: true,
        only: ["pasantias", "filters"],
      }
    );
  }, [dModalidad, dCarrera, dQ]);

  // Handler "Ver más" estable
  const handleLoadMore = () => {
    if (!nextUrl || loadingMore) return;
    setLoadingMore(true);
    router.visit(nextUrl, {
      preserveState: true,
      preserveScroll: true,
      replace: true,
      only: ["pasantias"],
    });
  };

  const totalCupos = useMemo(() => (items || []).reduce((acc, p) => acc + (p.cupos || 0), 0), [items]);

  return (
    <PublicLayout title="Pasantías y Prácticas Preprofesionales">
      <div className="min-h-screen bg-gradient-to-b from-primary-50/30 via-white to-secondary-50/30 dark:from-primary-950 dark:via-primary-900/50 dark:to-secondary-950/30">
        {/* HERO */}
        <section className="relative py-20 overflow-hidden">
          <div className="absolute inset-0 overflow-hidden">
            <motion.div
              className="absolute -top-20 -right-20 w-96 h-96 rounded-full bg-gradient-to-br from-primary-400/20 to-secondary-400/20 blur-3xl"
              animate={{ scale: [1, 1.2, 1], opacity: [0.3, 0.5, 0.3] }}
              transition={{ duration: 8, repeat: Infinity, ease: "easeInOut" }}
            />
            <motion.div
              className="absolute -bottom-20 -left-20 w-96 h-96 rounded-full bg-gradient-to-tr from-secondary-400/20 to-primary-400/20 blur-3xl"
              animate={{ scale: [1, 1.3, 1], opacity: [0.3, 0.6, 0.3] }}
              transition={{ duration: 10, repeat: Infinity, ease: "easeInOut", delay: 1 }}
            />
          </div>

          <div className="relative max-w-7xl mx-auto px-6 text-center">
            <motion.div
              initial={{ opacity: 0, y: -20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.6 }}
              className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-gradient-to-r from-primary-100 to-secondary-100 dark:from-primary-900/50 dark:to-secondary-900/50 border border-primary-200/50 dark:border-primary-700/50 mb-6"
            >
              <span className="w-2 h-2 rounded-full bg-gradient-to-r from-primary-500 to-secondary-500 animate-pulse" />
              <span className="text-sm font-semibold text-primary-700 dark:text-primary-300">
                Oportunidades Disponibles
              </span>
            </motion.div>

            <motion.h1
              initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.7, delay: 0.1 }}
              className="text-4xl sm:text-5xl lg:text-6xl font-black tracking-tight text-primary-900 dark:text-white mb-6"
            >
              Pasantías y Prácticas{" "}
              <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                Preprofesionales
              </span>
            </motion.h1>

            <motion.p
              initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.7, delay: 0.2 }}
              className="text-lg text-primary-700 dark:text-primary-200 max-w-3xl mx-auto leading-relaxed"
            >
              Conecta tu talento con empresas líderes. Encuentra la oportunidad perfecta para desarrollar tu carrera profesional.
            </motion.p>

            <motion.div
              initial={{ opacity: 0, y: 30 }} animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, delay: 0.4 }}
              className="mt-12 grid grid-cols-1 sm:grid-cols-3 gap-6 max-w-3xl mx-auto"
            >
              {[
                { label: "Oportunidades Activas", value: pasantias?.total ?? items.length, icon: FiBriefcase },
                { label: "Empresas Aliadas", value: empresas_count ?? "—", icon: FiAward },
                { label: "Cupos Disponibles", value: totalCupos, icon: FiUsers },
              ].map((stat, idx) => (
                <motion.div
                  key={stat.label} initial={{ opacity: 0, scale: 0.8 }} animate={{ opacity: 1, scale: 1 }}
                  transition={{ duration: 0.5, delay: 0.5 + idx * 0.1 }}
                  className="px-6 py-4 rounded-2xl backdrop-blur-md bg-white/70 dark:bg-primary-900/70 border border-primary-200/50 dark:border-primary-700/50 shadow-lg hover:shadow-xl hover:scale-105 transition-all duration-300"
                >
                  <stat.icon className="w-8 h-8 mx-auto mb-2 text-primary-600 dark:text-primary-400" />
                  <div className="text-3xl font-bold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                    {stat.value}
                  </div>
                  <div className="text-sm text-primary-700 dark:text-primary-300 font-medium mt-1">{stat.label}</div>
                </motion.div>
              ))}
            </motion.div>
          </div>
        </section>

        {/* FILTROS */}
        <section className="max-w-7xl mx-auto px-6 mb-8">
          <motion.div
            initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.5 }}
            className="bg-white dark:bg-primary-950 rounded-2xl shadow-lg border-2 border-primary-100 dark:border-primary-800 p-6"
          >
            <div className="grid grid-cols-1 lg:grid-cols-3 gap-4 items-end">
              {/* Modalidad */}
              <div>
                <label className="block text-sm font-semibold text-primary-700 dark:text-primary-300 mb-2">Modalidad</label>
                <div className="flex flex-wrap gap-2">
                  {modalidades.map(m => (
                    <button
                      key={m}
                      onClick={() => setFiltroModalidad(m)}
                      className={cls(
                        "px-4 py-2 rounded-xl font-medium text-sm transition-all duration-300",
                        filtroModalidad === m
                          ? "bg-gradient-to-r from-primary-600 to-secondary-600 text-white shadow-lg scale-105"
                          : "bg-primary-100 dark:bg-primary-900 text-primary-700 dark:text-primary-300 hover:bg-primary-200 dark:hover:bg-primary-800"
                      )}
                    >
                      {m}
                    </button>
                  ))}
                </div>
              </div>

              {/* Carrera */}
              <div>
                <label className="block text-sm font-semibold text-primary-700 dark:text-primary-300 mb-2">Carrera</label>
                <div className="flex flex-wrap gap-2">
                  {carreras.map(c => (
                    <button
                      key={c}
                      onClick={() => setFiltroCarrera(c)}
                      className={cls(
                        "px-4 py-2 rounded-xl font-medium text-sm transition-all duration-300",
                        filtroCarrera === c
                          ? "bg-gradient-to-r from-primary-600 to-secondary-600 text-white shadow-lg scale-105"
                          : "bg-primary-100 dark:bg-primary-900 text-primary-700 dark:text-primary-300 hover:bg-primary-200 dark:hover:bg-primary-800"
                      )}
                    >
                      {c}
                    </button>
                  ))}
                </div>
              </div>

              {/* Búsqueda */}
              <div className="w-full">
                <label className="block text-sm font-semibold text-primary-700 dark:text-primary-300 mb-2">Buscar</label>
                <div className="flex items-center gap-2">
                  <div className="flex-1 relative">
                    <FiSearch className="w-4 h-4 text-primary-400 absolute left-3 top-1/2 -translate-y-1/2" />
                    <input
                      type="text"
                      value={q}
                      onChange={(e) => setQ(e.target.value)}
                      placeholder="Título, empresa o carrera…"
                      className="w-full pl-9 pr-3 py-2 rounded-lg border border-primary-200 dark:border-primary-800 bg-white dark:bg-primary-900/40 text-sm"
                    />
                  </div>
                  {loadingFilter && <FiLoader className="animate-spin text-primary-600" />}
                </div>
              </div>
            </div>

            <div className="mt-4 pt-4 border-t border-primary-200 dark:border-primary-800">
              <p className="text-sm text-primary-700 dark:text-primary-300">
                Mostrando <span className="font-bold text-primary-900 dark:text-white">{items.length}</span> {items.length === 1 ? "oportunidad" : "oportunidades"}
              </p>
            </div>
          </motion.div>
        </section>

        {/* LISTA */}
        <section className="max-w-7xl mx-auto px-6 pb-20">
          <AnimatePresence mode="wait">
            {items.length > 0 ? (
              <>
                <motion.div key="grid" initial={{ opacity: 0 }} animate={{ opacity: 1 }} exit={{ opacity: 0 }}
                  className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                  {items.map((p, idx) => <PasantiaCard key={p.id} p={p} index={idx} />)}
                </motion.div>

                <div className="mt-10 flex justify-center">
                  {nextUrl ? (
                    <button
                      onClick={handleLoadMore}
                      disabled={loadingMore}
                      className="inline-flex items-center gap-2 px-6 py-3 rounded-xl bg-gradient-to-r from-primary-600 to-secondary-600 text-white font-semibold hover:shadow-lg transition-all duration-300 disabled:opacity-60"
                    >
                      {loadingMore && <FiLoader className="animate-spin" />}
                      {loadingMore ? "Cargando…" : "Ver más"}
                    </button>
                  ) : (
                    <span className="text-sm text-primary-600 dark:text-primary-300">No hay más resultados</span>
                  )}
                </div>
              </>
            ) : (
              <motion.div key="empty" initial={{ opacity: 0, scale: 0.95 }} animate={{ opacity: 1, scale: 1 }} exit={{ opacity: 0, scale: 0.95 }} className="text-center py-20">
                <FiAlertCircle className="w-16 h-16 mx-auto text-primary-400 dark:text-primary-600 mb-4" />
                <h3 className="text-2xl font-bold text-primary-900 dark:text-white mb-2">No se encontraron oportunidades</h3>
                <p className="text-primary-700 dark:text-primary-300">Ajusta los filtros para ver más resultados.</p>
              </motion.div>
            )}
          </AnimatePresence>
        </section>
      </div>
    </PublicLayout>
  );
}
