// resources/js/Pages/Landing/Oferta/Pregrado/Index.jsx
import React, { useRef } from "react";
import { Head, Link } from "@inertiajs/react";
import { motion, useMotionValue, useTransform, useSpring } from "framer-motion";
import { FiExternalLink, FiBook, FiAward, FiTrendingUp } from "react-icons/fi";
import PublicLayout from "@/Layouts/PublicLayout";

/* =========================
   STATS (puedes dejarlos estáticos)
   ========================= */
const STATS = [
  { label: "Años de Experiencia", value: "50+", icon: FiAward },
  { label: "Graduados", value: "3000+", icon: FiTrendingUp },
  { label: "Tasa de Empleabilidad", value: "95%", icon: FiBook },
];

/* =========================
   Tilt Card Hook (3D efecto)
   ========================= */
function useTilt(maxRotate = 6) {
  const ref = useRef(null);
  const x = useMotionValue(0.5);
  const y = useMotionValue(0.5);
  const rsX = useTransform(x, [0, 1], [maxRotate, -maxRotate]);
  const rsY = useTransform(y, [0, 1], [-maxRotate, maxRotate]);
  const rX = useSpring(rsX, { stiffness: 150, damping: 15 });
  const rY = useSpring(rsY, { stiffness: 150, damping: 15 });

  const onMouseMove = (e) => {
    const rect = ref.current?.getBoundingClientRect();
    if (!rect) return;
    x.set((e.clientX - rect.left) / rect.width);
    y.set((e.clientY - rect.top) / rect.height);
  };

  const onMouseLeave = () => {
    x.set(0.5);
    y.set(0.5);
  };

  return { ref, rX, rY, onMouseMove, onMouseLeave };
}

/* =========================
   Helpers
   ========================= */
const colorFrom = "from-primary-500";
const colorTo = "to-secondary-600";
const iconColor = "text-primary-600";
const bgGradient = "from-primary-50 to-secondary-50";

const storageOrHttp = (url) => {
  if (!url) return "/assets/images/placeholder_card.jpg";
  return url.startsWith("http") ? url : `/storage/${url}`;
};



/* =========================
   Card mejorada
   ========================= */
function ProgramaCard({ p, i }) {
  const { ref, rX, rY, onMouseMove, onMouseLeave } = useTilt(6);
  // Normalización desde backend
  const img = storageOrHttp(p.fotoTarjeta || p.imagenHero);
  const resumen = p.titulo || p.resumen || "";
  const descripcion = p.descripcion || "";
  const chips = [p.titulo, p.modalidad, p.duracion].filter(Boolean);

  return (
    <motion.article
      initial={{ opacity: 0, y: 40, scale: 0.95 }}
      whileInView={{ opacity: 1, y: 0, scale: 1 }}
      transition={{ duration: 0.7, delay: i * 0.15, ease: [0.23, 1, 0.32, 1] }}
      viewport={{ once: true, amount: 0.15 }}
      className="relative h-full"
    >
      <motion.div
        ref={ref}
        style={{ rotateX: rX, rotateY: rY, transformStyle: "preserve-3d" }}
        onMouseMove={onMouseMove}
        onMouseLeave={onMouseLeave}
        className="group h-full rounded-3xl border-2 border-primary-100/50 dark:border-primary-800/50
                   bg-white dark:bg-primary-950
                   shadow-xl hover:shadow-2xl hover:shadow-primary-500/20
                   transition-all duration-500 will-change-transform overflow-hidden
                   hover:border-primary-200/70 dark:hover:border-primary-700/70"
      >
        {/* Imagen cabecera con overlay gradiente */}
        <div className="relative overflow-hidden" style={{ aspectRatio: "1280 / 371" }}>
          <img
            src={p.fotoTarjeta}
            alt={p.nombre}
            className="h-full w-full object-cover transition-transform duration-700
                     group-hover:scale-110 group-hover:brightness-110"
            loading="lazy"
          />
          {/* Overlay gradiente animado */}
          <div
            className={`absolute inset-0 bg-gradient-to-br ${colorFrom} ${colorTo}
                       opacity-20 group-hover:opacity-40 transition-all duration-500`}
          />
          {/* Patrón decorativo */}
          <div
            className="absolute inset-0 opacity-0 group-hover:opacity-10 transition-opacity duration-500"
            style={{
              backgroundImage:
                "radial-gradient(circle, rgba(255,255,255,0.8) 1px, transparent 1px)",
              backgroundSize: "20px 20px",
            }}
          />

          {/* Badge flotante */}
          <div
            className={`absolute top-4 right-4 px-4 py-2 rounded-xl
                         backdrop-blur-md bg-white/90 dark:bg-primary-900/90
                         border border-white/50 dark:border-primary-700/50
                         shadow-lg transform transition-all duration-500
                         group-hover:scale-110 group-hover:-translate-y-1`}
          >
            <span className={`text-xs font-bold ${iconColor} uppercase tracking-wider`}>
              Pregrado
            </span>
          </div>
        </div>

        {/* Contenido */}
        <div className="p-6 space-y-4">
          {/* Título con gradiente */}
          <div>
            <h3
              className={`text-xl font-bold bg-gradient-to-r ${colorFrom} ${colorTo}
                          bg-clip-text text-transparent group-hover:scale-105
                          transition-transform duration-300 inline-block`}
            >
              {p.nombre}
            </h3>
            {resumen && (
              <p className="mt-2 text-sm text-primary-700 dark:text-primary-300 font-medium leading-relaxed">
                {resumen}
              </p>
            )}
            {descripcion && (
              <p className="mt-2 text-sm text-primary-900/60 dark:text-primary-100/60 leading-relaxed">
                {descripcion}
              </p>
            )}
          </div>

          {/* Chips (áreas/datos cortos) */}
          {chips.length > 0 && (
            <div className="space-y-2 pt-2">
              <p className="text-xs font-semibold text-primary-900/50 dark:text-primary-100/50 uppercase tracking-wider">
                Información
              </p>
              <div className="flex flex-wrap gap-2">
                {chips.map((item, idx) => (
                  <motion.span
                    key={`${item}-${idx}`}
                    initial={{ opacity: 0, scale: 0.8 }}
                    whileInView={{ opacity: 1, scale: 1 }}
                    transition={{ delay: i * 0.15 + idx * 0.08 }}
                    className={`px-3 py-1.5 rounded-lg text-xs font-medium
                            bg-gradient-to-r ${bgGradient} dark:from-primary-900/30 dark:to-primary-800/30
                            ${iconColor} border border-current/20
                            hover:scale-105 transition-transform duration-300 cursor-default`}
                  >
                    {item}
                  </motion.span>
                ))}
              </div>
            </div>
          )}

          {/* Botón */}
         <div className="flex flex-wrap gap-3 pt-4">
  <Link
    href={route("oferta.pregrado.detalle", p.slug)}
    className={`group/btn flex-1 inline-flex items-center justify-center gap-2
                rounded-xl px-5 py-3 text-sm font-bold text-white
                bg-gradient-to-r ${colorFrom} ${colorTo}
                hover:shadow-lg hover:shadow-current/30
                hover:scale-105 active:scale-95
                transition-all duration-300 relative overflow-hidden`}
  >
    <span className="relative z-10">Ver programa</span>
    <FiExternalLink className="relative z-10 group-hover/btn:translate-x-1 transition-transform" />
    <div
      className="absolute inset-0 -translate-x-full group-hover/btn:translate-x-full
                  bg-gradient-to-r from-transparent via-white/30 to-transparent
                  transition-transform duration-1000"
    />
  </Link>
</div>

        </div>

        {/* Borde glow */}
        <div
          className="absolute inset-0 rounded-3xl opacity-0 group-hover:opacity-100
                      transition-opacity duration-500 pointer-events-none"
        >
          <div className={`absolute inset-x-0 top-0 h-px bg-gradient-to-r ${colorFrom} ${colorTo}`} />
          <div className={`absolute inset-y-0 right-0 w-px bg-gradient-to-b ${colorFrom} ${colorTo}`} />
          <div className={`absolute inset-x-0 bottom-0 h-px bg-gradient-to-l ${colorFrom} ${colorTo}`} />
          <div className={`absolute inset-y-0 left-0 w-px bg-gradient-to-t ${colorFrom} ${colorTo}`} />
        </div>
      </motion.div>

      {/* Glow exterior */}
      <div
        aria-hidden
        className={`absolute -inset-1 rounded-3xl blur-3xl opacity-0 group-hover:opacity-30
                    transition-all duration-700 -z-10
                    bg-gradient-to-br ${colorFrom} ${colorTo}`}
      />
    </motion.article>
  );
}

/* =========================
   Página principal
   ========================= */
export default function Oferta({ pregrados = [] }) {
  const count = pregrados.length;

  return (
    <PublicLayout titulo={"Oferta Académica"}>
      <Head title="Pregrado - DETRI" />

      {/* Hero */}
      <section className="relative min-h-[65vh] flex items-center justify-center overflow-hidden">
        {/* Fondo */}
        <div className="absolute inset-0 bg-gradient-to-br from-primary-50 via-white to-secondary-50
                      dark:from-primary-950 dark:via-primary-900 dark:to-secondary-950" />

        {/* Patrón de puntos animado */}
        <motion.div
          className="pointer-events-none absolute inset-0 opacity-20"
          style={{
            backgroundImage:
              "radial-gradient(circle, rgba(99,102,241,0.4) 1px, transparent 1px)",
            backgroundSize: "30px 30px",
          }}
          animate={{ backgroundPosition: ["0px 0px", "30px 30px"] }}
          transition={{ duration: 20, repeat: Infinity, ease: "linear" }}
        />

        {/* Círculos decorativos */}
        <div className="absolute inset-0 overflow-hidden">
          <motion.div
            className="absolute top-20 right-20 w-72 h-72 rounded-full
                     bg-gradient-to-br from-primary-400/20 to-secondary-400/20 blur-3xl"
            animate={{ scale: [1, 1.2, 1], opacity: [0.3, 0.5, 0.3] }}
            transition={{ duration: 8, repeat: Infinity, ease: "easeInOut" }}
          />
          <motion.div
            className="absolute bottom-20 left-20 w-96 h-96 rounded-full
                     bg-gradient-to-tr from-secondary-400/20 to-primary-400/20 blur-3xl"
            animate={{ scale: [1, 1.3, 1], opacity: [0.3, 0.6, 0.3] }}
            transition={{ duration: 10, repeat: Infinity, ease: "easeInOut", delay: 1 }}
          />
        </div>

        {/* Contenido hero */}
        <div className="relative z-10 text-center px-6 max-w-5xl mx-auto">
          <motion.div initial={{ opacity: 0, scale: 0.9 }} animate={{ opacity: 1, scale: 1 }} transition={{ duration: 0.8, ease: [0.23, 1, 0.32, 1] }}>
            <motion.div
              initial={{ opacity: 0, y: -20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.6, delay: 0.2 }}
              className="inline-flex items-center gap-2 px-4 py-2 rounded-full
                       bg-gradient-to-r from-primary-100 to-secondary-100
                       dark:from-primary-900/50 dark:to-secondary-900/50
                       border border-primary-200/50 dark:border-primary-700/50
                       mb-6"
            >
              <span className="w-2 h-2 rounded-full bg-gradient-to-r from-primary-500 to-secondary-500 animate-pulse" />
              <span className="text-sm font-semibold text-primary-700 dark:text-primary-300">
                Programas de Pregrado
              </span>
            </motion.div>

            <h1 className="text-4xl sm:text-5xl lg:text-6xl font-black tracking-tight
                         text-primary-900 dark:text-white mb-6">
              Oferta Académica{" "}
              <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                DETRI
              </span>
            </h1>

            <motion.p
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.7, delay: 0.3 }}
              className="text-lg text-primary-700 dark:text-primary-200 max-w-3xl mx-auto leading-relaxed"
            >
              {count > 0
                ? `Explora ${count} ${count === 1 ? "programa disponible" : "programas disponibles"} en nuestro departamento.`
                : "Pronto encontrarás aquí todos nuestros programas de pregrado."}
            </motion.p>
          </motion.div>

          {/* Stats */}
          <motion.div
            initial={{ opacity: 0, y: 30 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, delay: 0.5 }}
            className="mt-12 grid grid-cols-1 sm:grid-cols-3 gap-6 max-w-3xl mx-auto"
          >
            {STATS.map((stat, idx) => (
              <motion.div
                key={stat.label}
                initial={{ opacity: 0, scale: 0.8 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ duration: 0.5, delay: 0.6 + idx * 0.1 }}
                className="relative group"
              >
                <div className="px-6 py-4 rounded-2xl backdrop-blur-md
                              bg-white/70 dark:bg-primary-900/70
                              border border-primary-200/50 dark:border-primary-700/50
                              shadow-lg hover:shadow-xl
                              transition-all duration-300 hover:scale-105">
                  <stat.icon className="w-8 h-8 mx-auto mb-2 text-primary-600 dark:text-primary-400 group-hover:scale-110 transition-transform" />
                  <div className="text-3xl font-bold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                    {stat.value}
                  </div>
                  <div className="text-sm text-primary-700 dark:text-primary-300 font-medium mt-1">
                    {stat.label}
                  </div>
                </div>
              </motion.div>
            ))}
          </motion.div>
        </div>
      </section>

      {/* Grid de programas */}
      <section className="py-20 bg-gradient-to-b from-white via-primary-50/30 to-white
                        dark:from-primary-950 dark:via-primary-900/30 dark:to-primary-950">
        <div className="mx-auto max-w-7xl px-6 lg:px-8">
          {/* Título de sección */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.6 }}
            className="text-center mb-16"
          >
            <h2 className="text-3xl font-bold text-primary-900 dark:text-white mb-4">
              Nuestros Programas
            </h2>
            <p className="text-primary-700 dark:text-primary-300 max-w-2xl mx-auto">
              Descubre la carrera que transformará tu futuro profesional
            </p>
          </motion.div>

          {/* Grid responsivo */}
          {pregrados.length === 0 ? (
            <p className="text-center text-gray-500">No hay carreras disponibles por el momento.</p>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 lg:gap-10">
              {pregrados.map((p, i) => (
                <ProgramaCard key={p.id ?? p.slug} p={p} i={i} />
              ))}
            </div>
          )}
        </div>
      </section>
    </PublicLayout>
  );
}
