import React, { useState } from "react";
import { Head } from "@inertiajs/react";
import { motion } from "framer-motion";
import PublicLayout from "@/Layouts/PublicLayout";
import {
  FiFileText, FiMapPin, FiClock, FiAward, FiBookOpen,
  FiDownload, FiMail, FiUser, FiCheckCircle, FiExternalLink,
  FiBook, FiCalendar, FiTarget
} from "react-icons/fi";

export default function Postgrado({ programa, titulo }) {
  const {
    nombre,
    resumen,
    banner,
    duracion,
    modalidad,
    titulo: tituloAcademico,
    resolucion,
    enlaces = {},
    objetivos = [],
    asignaturas = [],
    contacto = {},
    colorFrom = "from-primary-600",
    colorTo = "to-secondary-600",
  } = programa || {};

  const [selectedTab, setSelectedTab] = useState('objetivos');

  const infoCards = [
    { icon: FiClock, label: "Duración", value: duracion || "—", color: "from-primary-500 to-primary-600" },
    { icon: FiMapPin, label: "Modalidad", value: modalidad || "—", color: "from-secondary-500 to-secondary-600" },
    { icon: FiAward, label: "Título", value: tituloAcademico || "—", color: "from-blue-500 to-cyan-600" },
    { icon: FiFileText, label: "Resolución CES", value: resolucion || "—", color: "from-emerald-500 to-teal-600" },
  ];

  const tabs = [
    { id: 'objetivos', label: 'Objetivos', icon: FiTarget },
    { id: 'plan', label: 'Plan de Estudios', icon: FiBookOpen },
  ];

  return (
    <PublicLayout titulo={nombre || titulo || "Posgrado"}>
      <Head title={nombre || "Posgrado"} />

      {/* HERO MEJORADO */}
      <section className="relative h-[55vh] min-h-[420px] w-full overflow-hidden">
        {/* Imagen de fondo */}
        <div className="absolute inset-0">
          <img
            src={banner}
            alt={nombre}
            className="h-full w-full object-cover"
            loading="lazy"
          />
          {/* Overlays graduales */}
          <div className="absolute inset-0 bg-gradient-to-b from-primary-950/80 via-primary-900/70 to-primary-950/90" />
          <div className={`absolute inset-0 bg-gradient-to-tr ${colorFrom} ${colorTo} opacity-30 mix-blend-overlay`} />
        </div>

        {/* Patrón decorativo animado */}
        <motion.div
          className="absolute inset-0 opacity-10"
          style={{
            backgroundImage: 'radial-gradient(circle, rgba(255,255,255,0.4) 1px, transparent 1px)',
            backgroundSize: '30px 30px'
          }}
          animate={{
            backgroundPosition: ['0px 0px', '30px 30px'],
          }}
          transition={{
            duration: 20,
            repeat: Infinity,
            ease: "linear"
          }}
        />

        {/* Círculos decorativos */}
        <div className="absolute inset-0 overflow-hidden">
          <motion.div
            className={`absolute top-20 right-20 w-72 h-72 rounded-full
                     bg-gradient-to-br ${colorFrom} ${colorTo} opacity-20 blur-3xl`}
            animate={{
              scale: [1, 1.2, 1],
              opacity: [0.2, 0.4, 0.2],
            }}
            transition={{
              duration: 8,
              repeat: Infinity,
              ease: "easeInOut"
            }}
          />
          <motion.div
            className={`absolute bottom-20 left-20 w-96 h-96 rounded-full
                     bg-gradient-to-tr ${colorFrom} ${colorTo} opacity-20 blur-3xl`}
            animate={{
              scale: [1, 1.3, 1],
              opacity: [0.2, 0.5, 0.2],
            }}
            transition={{
              duration: 10,
              repeat: Infinity,
              ease: "easeInOut",
              delay: 1
            }}
          />
        </div>

        {/* Contenido del hero */}
        <div className="relative z-10 h-full flex items-center">
          <div className="mx-auto max-w-7xl w-full px-6 lg:px-8">
            <motion.div
              initial={{ opacity: 0, y: -20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.6 }}
              className="inline-flex items-center gap-2 px-4 py-2 rounded-full
                       bg-white/10 backdrop-blur-md
                       border border-white/20 mb-6"
            >
              <FiAward className="text-white w-4 h-4" />
              <span className="text-sm font-semibold text-white">
                Programa de Posgrado
              </span>
            </motion.div>

            <motion.h1
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.7, delay: 0.1 }}
              className="text-4xl md:text-5xl lg:text-6xl font-black text-white
                       drop-shadow-2xl mb-6 leading-tight max-w-4xl"
            >
              {nombre}
            </motion.h1>

            {resumen && (
              <motion.p
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: 0.2, duration: 0.6 }}
                className="text-lg md:text-xl text-white/90 max-w-3xl leading-relaxed
                         backdrop-blur-sm bg-white/5 rounded-2xl p-6 border border-white/10"
              >
                {resumen}
              </motion.p>
            )}
          </div>
        </div>

        {/* Ola decorativa inferior */}
        <div className="absolute bottom-0 left-0 right-0">
          <svg className="w-full h-16 md:h-24" viewBox="0 0 1200 120" preserveAspectRatio="none">
            <path
              d="M0,0 C300,80 600,80 900,40 L1200,0 L1200,120 L0,120 Z"
              className="fill-primary-50 dark:fill-primary-950"
            />
          </svg>
        </div>
      </section>

      {/* INFO CARDS MEJORADAS */}
      <section className="bg-gradient-to-b from-primary-50 to-white dark:from-primary-950 dark:to-primary-900
                        -mt-1 pt-16 pb-8">
        <div className="mx-auto max-w-7xl px-6 lg:px-8">
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
            {infoCards.map((card, idx) => (
              <motion.div
                key={card.label}
                initial={{ opacity: 0, y: 30, scale: 0.95 }}
                animate={{ opacity: 1, y: 0, scale: 1 }}
                transition={{ duration: 0.5, delay: idx * 0.1 }}
                className="group relative"
              >
                <div className="relative rounded-2xl bg-white dark:bg-primary-950
                              shadow-xl hover:shadow-2xl
                              border-2 border-primary-100 dark:border-primary-800
                              p-6 transition-all duration-500
                              hover:scale-105 hover:-translate-y-2
                              overflow-hidden">

                  {/* Gradiente de fondo en hover */}
                  <div className={`absolute inset-0 bg-gradient-to-br ${card.color}
                                opacity-0 group-hover:opacity-10 transition-opacity duration-500`} />

                  {/* Icono con gradiente */}
                  <div className={`relative inline-flex p-3 rounded-xl mb-4
                                bg-gradient-to-br ${card.color}
                                shadow-lg group-hover:scale-110 transition-transform duration-300`}>
                    <card.icon className="w-6 h-6 text-white" />
                  </div>

                  <div className="relative">
                    <p className="text-sm font-medium text-primary-600 dark:text-primary-400 mb-2">
                      {card.label}
                    </p>
                    <p className="text-lg font-bold text-primary-900 dark:text-white">
                      {card.value}
                    </p>
                  </div>

                  {/* Borde brillante */}
                  <div className="absolute inset-0 rounded-2xl opacity-0 group-hover:opacity-100
                                transition-opacity duration-500 pointer-events-none">
                    <div className={`absolute inset-x-0 top-0 h-px bg-gradient-to-r ${card.color}`} />
                    <div className={`absolute inset-y-0 right-0 w-px bg-gradient-to-b ${card.color}`} />
                    <div className={`absolute inset-x-0 bottom-0 h-px bg-gradient-to-l ${card.color}`} />
                    <div className={`absolute inset-y-0 left-0 w-px bg-gradient-to-t ${card.color}`} />
                  </div>
                </div>

                {/* Glow exterior */}
                <div className={`absolute -inset-1 rounded-2xl blur-2xl opacity-0
                              group-hover:opacity-30 transition-opacity duration-700 -z-10
                              bg-gradient-to-br ${card.color}`} />
              </motion.div>
            ))}
          </div>
        </div>
      </section>

      {/* CONTENIDO PRINCIPAL */}
      <section className="bg-white dark:bg-primary-900 py-16">
        <div className="mx-auto max-w-7xl px-6 lg:px-8">
          <div className="grid lg:grid-cols-3 gap-10">

            {/* Columna principal - Con tabs */}
            <div className="lg:col-span-2">
              <motion.div
                initial={{ opacity: 0, y: 20 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ duration: 0.6 }}
                className="rounded-3xl bg-gradient-to-br from-primary-50 to-secondary-50
                         dark:from-primary-950 dark:to-secondary-950
                         border-2 border-primary-100 dark:border-primary-800
                         shadow-2xl overflow-hidden"
              >
                {/* Tabs Header */}
                <div className="bg-white/50 dark:bg-primary-900/50 backdrop-blur-md
                              border-b-2 border-primary-100 dark:border-primary-800 p-2">
                  <div className="flex gap-2">
                    {tabs.map((tab) => (
                      <button
                        key={tab.id}
                        onClick={() => setSelectedTab(tab.id)}
                        className={`flex-1 flex items-center justify-center gap-2 px-6 py-3 rounded-xl
                                  font-semibold text-sm transition-all duration-300
                                  ${selectedTab === tab.id
                                    ? `bg-gradient-to-r ${colorFrom} ${colorTo} text-white shadow-lg scale-105`
                                    : 'bg-transparent text-primary-700 dark:text-primary-300 hover:bg-white/50 dark:hover:bg-primary-800/50'
                                  }`}
                      >
                        <tab.icon className="w-5 h-5" />
                        <span>{tab.label}</span>
                      </button>
                    ))}
                  </div>
                </div>

                {/* Tabs Content */}
                <div className="p-8">
                  {selectedTab === 'objetivos' && objetivos?.length > 0 && (
                    <motion.div
                      initial={{ opacity: 0, x: -20 }}
                      animate={{ opacity: 1, x: 0 }}
                      transition={{ duration: 0.4 }}
                    >
                      <div className="flex items-center gap-3 mb-6">
                        <div className={`p-3 rounded-xl bg-gradient-to-br ${colorFrom} ${colorTo}`}>
                          <FiTarget className="w-6 h-6 text-white" />
                        </div>
                        <h3 className="text-2xl font-bold text-primary-900 dark:text-white">
                          Objetivos del Programa
                        </h3>
                      </div>

                      <div className="space-y-4">
                        {objetivos.map((objetivo, idx) => (
                          <motion.div
                            key={idx}
                            initial={{ opacity: 0, x: -10 }}
                            animate={{ opacity: 1, x: 0 }}
                            transition={{ delay: idx * 0.1 }}
                            className="flex items-start gap-4 p-4 rounded-xl
                                     bg-white dark:bg-primary-900/50
                                     border border-primary-200 dark:border-primary-800
                                     hover:shadow-lg hover:scale-102 transition-all duration-300"
                          >
                            <div className={`flex-shrink-0 w-8 h-8 rounded-full
                                          bg-gradient-to-br ${colorFrom} ${colorTo}
                                          flex items-center justify-center text-white font-bold text-sm`}>
                              {idx + 1}
                            </div>
                            <p className="text-primary-800 dark:text-primary-200 leading-relaxed">
                              {objetivo}
                            </p>
                          </motion.div>
                        ))}
                      </div>
                    </motion.div>
                  )}

                  {selectedTab === 'plan' && asignaturas?.length > 0 && (
                    <motion.div
                      initial={{ opacity: 0, x: 20 }}
                      animate={{ opacity: 1, x: 0 }}
                      transition={{ duration: 0.4 }}
                    >
                      <div className="flex items-center gap-3 mb-6">
                        <div className={`p-3 rounded-xl bg-gradient-to-br ${colorFrom} ${colorTo}`}>
                          <FiBookOpen className="w-6 h-6 text-white" />
                        </div>
                        <h3 className="text-2xl font-bold text-primary-900 dark:text-white">
                          Plan de Estudios
                        </h3>
                      </div>

                      <div className="grid sm:grid-cols-2 gap-3">
                        {asignaturas.map((asignatura, idx) => (
                          <motion.div
                            key={idx}
                            initial={{ opacity: 0, scale: 0.9 }}
                            animate={{ opacity: 1, scale: 1 }}
                            transition={{ delay: idx * 0.05 }}
                            className="group relative flex items-center gap-3 p-4 rounded-xl
                                     bg-white dark:bg-primary-900/50
                                     border border-primary-200 dark:border-primary-800
                                     hover:shadow-lg hover:scale-105 transition-all duration-300"
                          >
                            <FiCheckCircle className={`w-5 h-5 flex-shrink-0
                                                      text-primary-600 dark:text-primary-400
                                                      group-hover:scale-110 transition-transform`} />
                            <span className="text-sm font-medium text-primary-800 dark:text-primary-200">
                              {asignatura}
                            </span>
                          </motion.div>
                        ))}
                      </div>
                    </motion.div>
                  )}
                </div>
              </motion.div>
            </div>

            {/* Columna lateral - Sticky */}
            <aside className="space-y-6 lg:sticky lg:top-24 lg:self-start">

              {/* Documentos */}
              {(enlaces?.brochure || enlaces?.plan) && (
                <motion.div
                  initial={{ opacity: 0, x: 20 }}
                  whileInView={{ opacity: 1, x: 0 }}
                  viewport={{ once: true }}
                  transition={{ duration: 0.6 }}
                  className="rounded-2xl bg-white dark:bg-primary-950
                           shadow-xl border-2 border-primary-100 dark:border-primary-800
                           p-6 overflow-hidden relative"
                >
                  {/* Decoración */}
                  <div className={`absolute top-0 right-0 w-32 h-32 bg-gradient-to-br ${colorFrom} ${colorTo}
                                opacity-10 blur-3xl rounded-full`} />

                  <div className="relative">
                    <div className="flex items-center gap-3 mb-5">
                      <div className={`p-2 rounded-lg bg-gradient-to-br ${colorFrom} ${colorTo}`}>
                        <FiDownload className="w-5 h-5 text-white" />
                      </div>
                      <h4 className="text-lg font-bold text-primary-900 dark:text-white">
                        Documentos
                      </h4>
                    </div>

                    <div className="space-y-3">
                      {enlaces?.brochure && (
                        <a
                          href={enlaces.brochure}
                          target="_blank"
                          rel="noreferrer"
                          className="group flex items-center justify-between p-4 rounded-xl
                                   border-2 border-primary-200 dark:border-primary-800
                                   text-primary-700 dark:text-primary-300
                                   hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50
                                   dark:hover:from-primary-900/50 dark:hover:to-secondary-900/50
                                   transition-all duration-300 hover:scale-105"
                        >
                          <div className="flex items-center gap-3">
                            <FiBook className="w-5 h-5" />
                            <span className="font-semibold">Brochure</span>
                          </div>
                          <FiExternalLink className="w-4 h-4 group-hover:translate-x-1 transition-transform" />
                        </a>
                      )}

                      {enlaces?.plan && (
                        <a
                          href={enlaces.plan}
                          target="_blank"
                          rel="noreferrer"
                          className="group flex items-center justify-between p-4 rounded-xl
                                   border-2 border-primary-200 dark:border-primary-800
                                   text-primary-700 dark:text-primary-300
                                   hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50
                                   dark:hover:from-primary-900/50 dark:hover:to-secondary-900/50
                                   transition-all duration-300 hover:scale-105"
                        >
                          <div className="flex items-center gap-3">
                            <FiFileText className="w-5 h-5" />
                            <span className="font-semibold">Plan de Estudios</span>
                          </div>
                          <FiExternalLink className="w-4 h-4 group-hover:translate-x-1 transition-transform" />
                        </a>
                      )}
                    </div>
                  </div>
                </motion.div>
              )}

              {/* Contacto */}
              {(contacto?.coordinador || contacto?.email) && (
                <motion.div
                  initial={{ opacity: 0, x: 20 }}
                  whileInView={{ opacity: 1, x: 0 }}
                  viewport={{ once: true }}
                  transition={{ duration: 0.6, delay: 0.1 }}
                  className="rounded-2xl bg-gradient-to-br from-primary-600 to-secondary-600
                           shadow-xl border-2 border-primary-500
                           p-6 text-white relative overflow-hidden"
                >
                  {/* Patrón decorativo */}
                  <div className="absolute inset-0 opacity-10"
                       style={{
                         backgroundImage: 'radial-gradient(circle, rgba(255,255,255,0.4) 1px, transparent 1px)',
                         backgroundSize: '20px 20px'
                       }} />

                  <div className="relative">
                    <div className="flex items-center gap-3 mb-5">
                      <div className="p-2 rounded-lg bg-white/20 backdrop-blur-sm">
                        <FiMail className="w-5 h-5" />
                      </div>
                      <h4 className="text-lg font-bold">
                        Información de Contacto
                      </h4>
                    </div>

                    <div className="space-y-4">
                      {contacto?.coordinador && (
                        <div className="flex items-start gap-3 p-3 rounded-xl bg-white/10 backdrop-blur-sm">
                          <FiUser className="w-5 h-5 flex-shrink-0 mt-0.5" />
                          <div>
                            <p className="text-sm font-medium text-white/80 mb-1">Coordinador/a</p>
                            <p className="font-semibold">{contacto.coordinador}</p>
                          </div>
                        </div>
                      )}

                      {contacto?.email && (
                        <a
                          href={`mailto:${contacto.email}`}
                          className="group flex items-start gap-3 p-3 rounded-xl
                                   bg-white/10 backdrop-blur-sm
                                   hover:bg-white/20 transition-all duration-300"
                        >
                          <FiMail className="w-5 h-5 flex-shrink-0 mt-0.5" />
                          <div>
                            <p className="text-sm font-medium text-white/80 mb-1">Email</p>
                            <p className="font-semibold group-hover:underline break-all">
                              {contacto.email}
                            </p>
                          </div>
                        </a>
                      )}
                    </div>
                  </div>
                </motion.div>
              )}

              {/* CTA adicional */}
              <motion.div
                initial={{ opacity: 0, y: 20 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ duration: 0.6, delay: 0.2 }}
              >
                <button
                  className={`w-full px-6 py-4 rounded-2xl font-bold text-white
                           bg-gradient-to-r ${colorFrom} ${colorTo}
                           hover:shadow-2xl hover:scale-105 active:scale-95
                           transition-all duration-300 relative overflow-hidden group`}
                >
                  <span className="relative z-10 flex items-center justify-center gap-2">
                    <FiCalendar />
                    Solicitar Información
                  </span>
                  {/* Shimmer effect */}
                  <div className="absolute inset-0 -translate-x-full group-hover:translate-x-full
                                bg-gradient-to-r from-transparent via-white/30 to-transparent
                                transition-transform duration-1000" />
                </button>
              </motion.div>
            </aside>
          </div>
        </div>
      </section>
    </PublicLayout>
  );
}
