import React from "react";
import { Head, Link, usePage } from "@inertiajs/react";
import { motion } from "framer-motion";
import {
  FiAward,
  FiClock,
  FiBookOpen,
  FiFileText,
  FiExternalLink,
  FiArrowRight,
} from "react-icons/fi";
import PublicLayout from "@/Layouts/PublicLayout";

/* =========================
   Helper animación
   ========================= */
const fadeUp = (d = 0) => ({
  initial: { opacity: 0, y: 26 },
  whileInView: { opacity: 1, y: 0 },
  transition: { duration: 0.6, ease: "easeOut", delay: d },
  viewport: { once: true, amount: 0.2 },
});

/* =========================
   Card Posgrado
   ========================= */
function PosgradoCard({ p, i }) {
  const banner = p.foto_tarjeta || p.imagen_hero || "/assets/images/postgrado/fallback.jpg";
  const colorFrom = p.colorFrom || "from-primary-500";
  const colorTo = p.colorTo || "to-secondary-600";

  return (
    <motion.article
      {...fadeUp(i * 0.06)}
      className="relative overflow-hidden rounded-3xl border border-primary-100/70 bg-white/70 backdrop-blur-lg shadow-lg hover:shadow-xl transition"
    >
      {/* Cintillo POSGRADO */}
      <div className="absolute left-0 top-0 h-full w-2 bg-gradient-to-b from-primary-400 to-primary-600" aria-hidden />
      <div className="absolute left-4 top-4 z-10">
        <span className="inline-flex items-center rounded-full bg-primary-600/90 text-white text-[10px] uppercase tracking-widest px-3 py-1">
          POSGRADO
        </span>
      </div>

      {/* Header banner */}
      <div className="relative overflow-hidden" style={{ aspectRatio: "16 / 6" }}>
        <img
          src={banner}
          alt={p.nombre}
          className="object-cover w-full h-full transition-transform duration-700 hover:scale-[1.03]"
          loading="lazy"
        />
        <div className={`absolute inset-0 bg-gradient-to-tr ${colorFrom} ${colorTo} opacity-15`} />
      </div>

      {/* Cuerpo */}
      <div className="p-6 md:p-7">
        <div className="flex items-center justify-between gap-3">
          <h3 className="text-xl md:text-2xl font-bold text-ink">{p.nombre}</h3>
          {p.tipo && (
            <span className={`shrink-0 inline-flex items-center rounded-full px-3 py-1 text-xs font-semibold text-white bg-gradient-to-r ${colorFrom} ${colorTo}`}>
              {p.tipo}
            </span>
          )}
        </div>

        {p.resumen && (
          <p className="mt-3 text-sm md:text-[15px] leading-relaxed text-ink/80">{p.resumen}</p>
        )}

        {/* Highlights */}
        <div className="mt-5 grid grid-cols-1 sm:grid-cols-3 gap-3 text-[13px]">
          <div className="flex items-center gap-2 text-ink/70">
            <FiClock /> {p.duracion || "—"}
          </div>
          <div className="flex items-center gap-2 text-ink/70">
            <FiBookOpen /> {p.modalidad || "—"}
          </div>
          <div className="flex items-center gap-2 text-ink/70">
            <FiAward /> {p.titulo || "—"}
          </div>
        </div>

        {(p.resolucion || p.resolucion_senescyt) && (
          <div className="mt-2 text-[12px] text-ink/60">
            Resolución CES: {p.resolucion || p.resolucion_senescyt}
          </div>
        )}

        {/* CTAs */}
        <div className="mt-6 flex flex-wrap gap-3">
          {p.brochure && (
            <a
              href={p.brochure}
              target="_blank"
              rel="noreferrer"
              className="inline-flex items-center gap-2 rounded-xl bg-primary-600 text-white px-4 py-2 text-sm font-semibold hover:bg-primary-700"
            >
              Brochure <FiFileText />
            </a>
          )}
          {p.malla && (
            <a
              href={p.malla}
              target="_blank"
              rel="noreferrer"
              className="inline-flex items-center gap-2 rounded-xl border border-primary-300 text-ink px-4 py-2 text-sm font-semibold hover:bg-primary-50"
            >
              Plan de estudios <FiExternalLink />
            </a>
          )}
          <Link
            href={`/oferta/posgrado/${p.slug}`}
            className="inline-flex items-center gap-2 rounded-xl bg-gradient-to-r from-primary-600 to-primary-500 text-white px-4 py-2 text-sm font-semibold hover:brightness-110"
          >
            Conocer más <FiArrowRight />
          </Link>
        </div>
      </div>
    </motion.article>
  );
}

/* =========================
   Página Posgrado
   ========================= */
export default function PosgradoIndex({postgrados, titulo}) {
    console.log(postgrados);

  return (
    <PublicLayout titulo={titulo}>
      <Head title={titulo} />

      {/* HERO */}
      <section className="relative overflow-hidden bg-gradient-to-br from-primary-50 via-white to-blue-50">
        <div className="relative min-h-[50vh] flex flex-col items-center justify-center text-center px-6">
          <span className="inline-flex items-center rounded-full bg-primary-100 text-primary-800 text-[10px] uppercase tracking-widest px-3 py-1">
            POSGRADO DETRI
          </span>
          <h1 className="mt-4 text-4xl md:text-5xl font-black tracking-tight text-ink">
            Maestrías DETRI — Excelencia que impacta
          </h1>
          <p className="mt-4 max-w-3xl mx-auto text-ink/70">
            Programas de alto rigor académico y profesional para liderar la innovación y la investigación en el país.
          </p>
          <div className="mt-6">
            <Link
              href="/admision"
              className="inline-flex items-center gap-2 rounded-xl bg-primary-600 text-white px-5 py-2.5 text-sm font-semibold hover:bg-primary-700"
            >
              Admisiones posgrado
            </Link>
          </div>
        </div>
      </section>

      {/* GRID */}
      <section className="py-16 bg-gradient-to-b from-white to-primary-50/50">
        <div className="mx-auto max-w-7xl px-6 lg:px-8">
          {postgrados.length === 0 ? (
            <div className="text-center text-ink/60">No hay programas de posgrado disponibles por el momento.</div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 gap-10">
              {postgrados.map((p, i) => (
                <PosgradoCard key={p.slug || p.id || i} p={p} i={i} />
              ))}
            </div>
          )}
        </div>
      </section>

      {/* CTA final */}
      <section className="pb-20 bg-white">
        <div className="mx-auto max-w-5xl px-6 lg:px-8">
          <motion.div
            {...fadeUp(0.1)}
            className="rounded-3xl p-8 md:p-10 bg-gradient-to-tr from-primary-600 to-blue-700 text-white text-center shadow-xl"
          >
            <h2 className="text-2xl md:text-3xl font-extrabold">
              ¿Listo para dar el siguiente paso académico?
            </h2>
            <p className="mt-3 text-white/90">
              Descubre los beneficios, becas y oportunidades de formación avanzada que ofrecen nuestras maestrías.
            </p>
            <div className="mt-6 flex flex-wrap items-center justify-center gap-3">
              <Link
                href="/admision"
                className="rounded-xl bg-white text-ink px-5 py-2.5 text-sm font-semibold hover:bg-white/90"
              >
                Ir a Admisiones
              </Link>
              <Link
                href="/contacto"
                className="rounded-xl border border-white/60 px-5 py-2.5 text-sm font-semibold hover:bg-white/10"
              >
                Solicitar información
              </Link>
            </div>
          </motion.div>
        </div>
      </section>
    </PublicLayout>
  );
}
