import React, { useMemo, useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { FiUser, FiClock, FiBookOpen, FiMapPin, FiFileText, FiLayers } from "react-icons/fi";
import PublicLayout from "@/Layouts/PublicLayout";

const cx = (...xs) => xs.filter(Boolean).join(" ");
const GRADIENT_PS = "from-primary-600 to-secondary-600";

export default function CarreraDetalleNueva({ career }) {
  const [active, setActive] = useState("informacion");

  const tabs = useMemo(() => ([
    { id: "informacion", label: "Información" },
    { id: "perfil",      label: "Perfil de estudiante" },
    { id: "malla",       label: "Malla curricular" },
    { id: "requisitos",  label: "Requisitos de graduación" },
  ]), []);

  const tituloObtener = useMemo(() => `Ingeniería en ${career?.nombre ?? ""}`, [career]);

  return (
    <PublicLayout titulo={career?.nombre || "Carrera"}>
      <div className="min-h-screen bg-gradient-to-b from-gray-50 to-white">
        {/* HERO (solo aquí hay imagen) */}
        <section className="relative overflow-hidden">
          {career?.imagenHero && (
            <img
              src={career.imagenHero}
              alt={career.nombre}
              className="absolute inset-0 h-full w-full object-cover"
            />
          )}
          <div className="absolute inset-0 bg-black/45" />
          <div className={cx("absolute inset-0 bg-gradient-to-br mix-blend-multiply opacity-60", GRADIENT_PS)} />
          <div className="relative max-w-7xl mx-auto px-6 py-16 sm:py-24">
            <motion.div
              initial={{ opacity: 0, y: 24 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.6 }}
              className="text-white"
            >
              <div className="inline-flex items-center gap-2 px-4 py-1.5 rounded-full bg-white/10 border border-white/20 text-xs font-semibold mb-4">
                <FiLayers className="opacity-90" /> Pregrado · DETRI
              </div>
              <h1 className="text-4xl sm:text-5xl lg:text-6xl font-black tracking-tight">
                {career?.nombre}
              </h1>
              <p className="mt-3 text-white/85 max-w-2xl">
                Conoce la información clave, el perfil estudiantil, la malla curricular y requisitos de graduación.
              </p>
            </motion.div>
          </div>
        </section>

        {/* Métricas principales */}
        <section className="max-w-7xl mx-auto px-6 -mt-10 relative z-10">
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
            <div className="rounded-2xl bg-white shadow border border-gray-100 p-5">
              <div className="text-xs text-gray-500 font-semibold">Título a obtener</div>
              <div className="mt-1 text-gray-900 font-bold">{tituloObtener}</div>
            </div>
            <div className="rounded-2xl bg-white shadow border border-gray-100 p-5">
              <div className="text-xs text-gray-500 font-semibold">Duración</div>
              <div className="mt-1 text-gray-900 font-bold">
                {career?.duracion ?? "-"} semestres
              </div>
            </div>
            <div className="rounded-2xl bg-white shadow border border-gray-100 p-5">
              <div className="text-xs text-gray-500 font-semibold">Modalidad</div>
              <div className="mt-1 text-gray-900 font-bold">{career?.modalidad ?? "-"}</div>
            </div>
            <div className="rounded-2xl bg-white shadow border border-gray-100 p-5">
              <div className="text-xs text-gray-500 font-semibold">Coordinación</div>
              <div className="mt-1 text-gray-900 font-bold">{career?.coordinador?.nombre ?? "-"}</div>
              {career?.coordinador?.email && (
                <a
                  href={`mailto:${career.coordinador.email}`}
                  className="text-sm text-primary-700 hover:text-primary-800"
                >
                  {career.coordinador.email}
                </a>
              )}
            </div>
          </div>
        </section>

        {/* Tabs */}
        <section className="max-w-7xl mx-auto px-6 mt-8">
          <div className="bg-white rounded-2xl shadow border border-gray-100">
            {/* Pestañas */}
            <div className="flex flex-wrap gap-2 border-b border-gray-100 p-3">
              {tabs.map(t => {
                const activeTab = active === t.id;
                return (
                  <button
                    key={t.id}
                    onClick={() => setActive(t.id)}
                    className={cx(
                      "relative px-4 py-2 rounded-lg text-sm font-semibold transition-colors",
                      activeTab
                        ? "text-white bg-gradient-to-r from-primary-600 to-secondary-600 shadow"
                        : "text-gray-700 hover:bg-gray-100"
                    )}
                    aria-selected={activeTab}
                  >
                    {t.label}
                  </button>
                );
              })}
            </div>

            {/* Contenido */}
            <div className="p-6">
              <AnimatePresence mode="wait">
                {active === "informacion" && (
                  <motion.div
                    key="informacion"
                    initial={{ opacity: 0, y: 8 }}
                    animate={{ opacity: 1, y: 0 }}
                    exit={{ opacity: 0, y: -8 }}
                    transition={{ duration: 0.25 }}
                  >
                    <div className="flex items-center gap-2 text-gray-800 font-bold">
                      <FiFileText /> Información sobre la carrera
                    </div>
                    <div className="mt-3 whitespace-pre-line leading-relaxed text-gray-700">
                      {career?.informacion || "—"}
                    </div>
                  </motion.div>
                )}

                {active === "perfil" && (
                  <motion.div
                    key="perfil"
                    initial={{ opacity: 0, y: 8 }}
                    animate={{ opacity: 1, y: 0 }}
                    exit={{ opacity: 0, y: -8 }}
                    transition={{ duration: 0.25 }}
                  >
                    <div className="flex items-center gap-2 text-gray-800 font-bold">
                      <FiUser /> Perfil de estudiante
                    </div>
                    <div className="mt-3 whitespace-pre-line leading-relaxed text-gray-700">
                      {career?.perfil || "—"}
                    </div>
                  </motion.div>
                )}

                {active === "malla" && (
                  <motion.div
                    key="malla"
                    initial={{ opacity: 0, y: 8 }}
                    animate={{ opacity: 1, y: 0 }}
                    exit={{ opacity: 0, y: -8 }}
                    transition={{ duration: 0.25 }}
                  >
                    <div className="flex items-center gap-2 text-gray-800 font-bold">
                      <FiBookOpen /> Malla curricular
                    </div>
                    {career?.mallaUrl ? (
                      <div className="mt-4 rounded-xl overflow-hidden border border-gray-200">
                        <iframe
                          src={`${career.mallaUrl}#zoom=80`}
                          title="Malla Curricular"
                          width="100%"
                          height="600"
                        />
                      </div>
                    ) : (
                      <p className="mt-3 text-gray-600">No disponible.</p>
                    )}
                  </motion.div>
                )}

                {active === "requisitos" && (
                  <motion.div
                    key="requisitos"
                    initial={{ opacity: 0, y: 8 }}
                    animate={{ opacity: 1, y: 0 }}
                    exit={{ opacity: 0, y: -8 }}
                    transition={{ duration: 0.25 }}
                  >
                    <div className="flex items-center gap-2 text-gray-800 font-bold">
                      <FiMapPin /> Requisitos de graduación
                    </div>
                    <div className="mt-3 space-y-2">
                      {(career?.requisitos || []).length ? (
                        career.requisitos.map((req, i) => (
                          <div key={i} className="flex items-start gap-2">
                            <span className="mt-1 h-2 w-2 rounded-full bg-primary-600" />
                            <p className="text-gray-700">{req}</p>
                          </div>
                        ))
                      ) : (
                        <p className="text-gray-600">No disponible.</p>
                      )}
                    </div>
                  </motion.div>
                )}
              </AnimatePresence>
            </div>
          </div>
        </section>

        {/* Footer info coordinador opcional */}
        {career?.coordinador?.foto && (
          <section className="max-w-7xl mx-auto px-6 mt-10 mb-16">
            <div className="rounded-2xl bg-white border border-gray-100 shadow p-6 flex items-center gap-4">
              <img
                src={career.coordinador.foto}
                alt={career.coordinador.nombre}
                className="h-16 w-16 rounded-xl object-cover"
              />
              <div>
                <div className="text-sm text-gray-500 font-semibold">Coordinador/a</div>
                <div className="text-gray-900 font-bold">{career.coordinador.nombre}</div>
                {career.coordinador.email && (
                  <a href={`mailto:${career.coordinador.email}`} className="text-primary-700 hover:text-primary-800 text-sm">
                    {career.coordinador.email}
                  </a>
                )}
              </div>
            </div>
          </section>
        )}
      </div>
    </PublicLayout>
  );
}
