import React, { useEffect, useMemo, useRef, useState } from "react";
import { router } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import {
  FiSearch, FiFilter, FiX, FiChevronDown, FiChevronUp, FiUsers,
  FiBookOpen, FiCalendar, FiExternalLink, FiAward, FiFileText,
  FiTrendingUp, FiLoader, FiPlusCircle, FiZap,
} from "react-icons/fi";
import PublicLayout from "@/Layouts/PublicLayout";

/* =========================
   Helpers
   ========================= */
const fadeUp = (d = 0) => ({
  initial: { opacity: 0, y: 20 },
  whileInView: { opacity: 1, y: 0 },
  transition: { duration: 0.4, ease: [0.22, 1, 0.36, 1], delay: d },
  viewport: { once: true, amount: 0.15 },
});
const estadoBadge = (estado) =>
  estado === "publicado"
    ? "bg-secondary-50 text-secondary-700 border-secondary-200"
    : "bg-primary-50 text-primary-700 border-primary-200";
const formatNumber = (n) => new Intl.NumberFormat("es-ES").format(n);

/* =========================
   Tarjeta
   ========================= */
function PublicacionCard({ pub }) {
  const [open, setOpen] = useState(false);
  return (
    <motion.article {...fadeUp(0.05)} className="group relative rounded-2xl border border-gray-200 bg-white shadow-sm hover:shadow-md transition-all duration-300 overflow-hidden">
      <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-secondary-500 via-primary-500 to-secondary-500" />
      <div className="flex items-center justify-between px-6 py-4 border-b border-gray-100 bg-gradient-to-br from-gray-50 to-white">
        <div className="flex items-center gap-3">
          <span className={`inline-flex items-center gap-1.5 rounded-lg px-3 py-1.5 text-xs font-semibold border ${estadoBadge(pub.estado)}`}>
            <div className={`w-1.5 h-1.5 rounded-full ${pub.estado === "publicado" ? "bg-secondary-500" : "bg-primary-500"}`} />
            {pub.estado === "publicado" ? "Publicado" : "En Prensa"}
          </span>
          <div className="inline-flex items-center gap-1.5 text-xs font-medium text-gray-600">
            <FiCalendar className="w-3.5 h-3.5" />
            <span>{pub.anio}</span>
          </div>
        </div>
        {pub.citadoPor > 0 && (
          <div className="flex items-center gap-1.5 text-xs font-semibold text-secondary-700">
            <FiAward className="w-3.5 h-3.5" />
            <span>{formatNumber(pub.citadoPor)} citas</span>
          </div>
        )}
      </div>

      <div className="p-6">
        <h3 className="text-lg font-bold text-gray-900 leading-tight mb-4 group-hover:text-secondary-700 transition-colors">{pub.titulo}</h3>
        <div className="mb-4 flex items-start gap-2">
          <FiUsers className="w-4 h-4 mt-0.5 text-primary-600 flex-shrink-0" />
          <p className="text-sm text-gray-700 font-medium">{pub.autores}</p>
        </div>
        <div className="mb-4 pb-4 border-b border-gray-100 space-y-2">
          <div className="flex items-start gap-2 text-sm">
            <FiBookOpen className="w-4 h-4 mt-0.5 text-secondary-600 flex-shrink-0" />
            <span className="text-gray-700">{pub.revista}</span>
          </div>
          <div className="flex items-center gap-2 text-xs text-gray-600 ml-6">
            <span className="px-2 py-1 rounded-md bg-gray-100 font-medium">{pub.editorial}</span>
          </div>
        </div>
        <p className="text-sm text-gray-600 leading-relaxed line-clamp-3 mb-4">{pub.extracto}</p>
        {pub.versiones > 0 && (
          <div className="mb-4 inline-flex items-center gap-1.5 text-xs text-gray-600 bg-gray-50 px-3 py-1.5 rounded-lg border border-gray-200">
            <FiFileText className="w-3.5 h-3.5" />
            <span>{pub.versiones} {pub.versiones === 1 ? "versión" : "versiones"}</span>
          </div>
        )}
        <div className="flex items-center gap-3 pt-4 border-t border-gray-100">
          <button onClick={() => setOpen((v) => !v)} className="inline-flex items-center gap-1.5 text-sm font-semibold text-primary-700 hover:text-primary-800 transition-colors group/btn">
            <FiFileText className="w-4 h-4" />
            <span>{open ? "Ver menos" : "Ver extracto completo"}</span>
            {open ? <FiChevronUp className="w-4 h-4" /> : <FiChevronDown className="w-4 h-4" />}
          </button>
          {pub.url && (
            <a href={pub.url} target="_blank" rel="noopener noreferrer" className="ml-auto inline-flex items-center gap-1.5 text-sm font-semibold text-secondary-700 hover:text-secondary-800 transition-colors group/link">
              <span>Ver publicación</span>
              <FiExternalLink className="w-4 h-4" />
            </a>
          )}
        </div>
        <AnimatePresence initial={false}>
          {open && (
            <motion.div initial={{ height: 0, opacity: 0 }} animate={{ height: "auto", opacity: 1 }} exit={{ height: 0, opacity: 0 }} transition={{ duration: 0.3 }} className="overflow-hidden">
              <div className="mt-4 rounded-xl border border-secondary-100 bg-secondary-50/30 p-4">
                <h4 className="text-sm font-semibold text-gray-900 mb-2 flex items-center gap-2">
                  <FiFileText className="w-4 h-4" /> Extracto completo
                </h4>
                <p className="text-sm text-gray-700 leading-relaxed">{pub.extracto}</p>
              </div>
            </motion.div>
          )}
        </AnimatePresence>
      </div>
    </motion.article>
  );
}

/* =========================
   Página principal con filtros SSR + "Mostrar más"
   ========================= */
export default function Publicaciones({ publicaciones, filters, catalogs, metrics }) {
  // Estado local para "append" de páginas
  const [items, setItems] = useState(publicaciones.data || []);
  const [nextUrl, setNextUrl] = useState(publicaciones.next_page_url);
  const [loadingMore, setLoadingMore] = useState(false);
  const [autoLoad, setAutoLoad] = useState(true); // 👈 activar/desactivar auto-carga al llegar al final

  // Filtros UI (reflect current filters)
  const [q, setQ] = useState(filters?.q ?? "");
  const [fEstado, setFEstado] = useState(filters?.estado ?? "");
  const [fAnio, setFAnio] = useState(filters?.anio ?? "");
  const [fDesde, setFDesde] = useState(filters?.anio_from ?? "");
  const [fHasta, setFHasta] = useState(filters?.anio_to ?? "");
  const [sort, setSort] = useState(filters?.sort ?? "anio");
  const [dir, setDir] = useState(filters?.dir ?? "desc");

  // Si el servidor devuelve nueva primera página (por filtros), reemplazamos lista
  useEffect(() => {
    setItems(publicaciones.data || []);
    setNextUrl(publicaciones.next_page_url);
  }, [publicaciones]);

  // Aplicar filtros (SSR)
  const apply = () => {
    router.get(
      route("landing.investigacion.publicaciones"),
      {
        q,
        estado: fEstado,
        anio: fAnio,
        anio_from: fDesde,
        anio_to: fHasta,
        sort,
        dir,
        page: 1,
      },
      { preserveScroll: true, replace: true }
    );
  };

  const resetFiltros = () => {
    setQ(""); setFEstado(""); setFAnio(""); setFDesde(""); setFHasta("");
    setSort("anio"); setDir("desc");
    router.get(route("landing.investigacion.publicaciones"), {}, { preserveScroll: true, replace: true });
  };

  // Cargar más (usa next_page_url del paginador de Laravel)
  const loadMore = () => {
    if (!nextUrl || loadingMore) return;
    setLoadingMore(true);
    router.visit(nextUrl, {
      preserveScroll: true,
      preserveState: true,
      replace: true,
      only: ["publicaciones"],
      onSuccess: (page) => {
        const nextPage = page.props.publicaciones;
        setItems((prev) => [...prev, ...(nextPage.data || [])]);
        setNextUrl(nextPage.next_page_url);
      },
      onFinish: () => setLoadingMore(false),
    });
  };

  // Auto-carga con IntersectionObserver
  const sentinelRef = useRef(null);
  useEffect(() => {
    if (!autoLoad) return;
    const el = sentinelRef.current;
    if (!el) return;
    const obs = new IntersectionObserver((entries) => {
      entries.forEach((e) => {
        if (e.isIntersecting) loadMore();
      });
    }, { rootMargin: "200px" });
    obs.observe(el);
    return () => obs.disconnect();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [autoLoad, nextUrl]);

  // Catálogos
  const years   = catalogs?.years ?? [];
  const estados = catalogs?.estados ?? [];

  // Métricas (hero)
  const totalPubs    = metrics?.total ?? items.length;
  const totalCitas   = metrics?.totalCitas ?? 0;
  const recientes5an = metrics?.recientes5an ?? 0;

  const hayFiltrosActivos = q || fEstado || fAnio || fDesde || fHasta || sort !== "anio" || dir !== "desc";

  return (
    <PublicLayout titulo={"Publicaciones Científicas"}>
      <div className="min-h-screen bg-gray-50">
        {/* HERO */}
        <section className="relative bg-gradient-to-br from-secondary-600 via-secondary-700 to-primary-700 overflow-hidden">
          <div className="absolute inset-0 opacity-10">
            <div className="absolute top-0 right-1/4 w-96 h-96 bg-white rounded-full blur-3xl" />
            <div className="absolute bottom-0 left-1/4 w-96 h-96 bg-primary-300 rounded-full blur-3xl" />
          </div>

          <div className="relative max-w-7xl mx-auto px-6 lg:px-8 py-20 sm:py-28">
            <motion.div initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.6 }} className="text-center">
              <div className="inline-flex items-center justify-center w-16 h-16 rounded-2xl bg-white/10 backdrop-blur-sm mb-6">
                <FiBookOpen className="w-8 h-8 text-white" />
              </div>
              <h1 className="text-4xl sm:text-5xl lg:text-6xl font-black tracking-tight text-white mb-6">Publicaciones Científicas</h1>
              <p className="text-lg sm:text-xl text-secondary-100 max-w-3xl mx-auto leading-relaxed">
                Contribuciones académicas que impulsan el conocimiento y la innovación tecnológica
              </p>

              {/* Métricas destacadas */}
              <div className="mt-10 grid grid-cols-1 sm:grid-cols-3 gap-6 max-w-3xl mx-auto">
                <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20">
                  <div className="flex items-center justify-center gap-2 text-white/90 text-sm font-medium mb-2">
                    <FiFileText className="w-4 h-4" /><span>Total Publicaciones</span>
                  </div>
                  <div className="text-3xl font-black text-white">{totalPubs}</div>
                </div>
                <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20">
                  <div className="flex items-center justify-center gap-2 text-white/90 text-sm font-medium mb-2">
                    <FiAward className="w-4 h-4" /><span>Total Citas</span>
                  </div>
                  <div className="text-3xl font-black text-white">{formatNumber(totalCitas)}</div>
                </div>
                <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20">
                  <div className="flex items-center justify-center gap-2 text-white/90 text-sm font-medium mb-2">
                    <FiTrendingUp className="w-4 h-4" /><span>Últimos 5 años</span>
                  </div>
                  <div className="text-3xl font-black text-white">{recientes5an}</div>
                </div>
              </div>
            </motion.div>
          </div>
        </section>

        {/* FILTROS (SSR) */}
        <section className="bg-white border-b border-gray-200 sticky top-0 z-10 shadow-sm">
          <div className="max-w-7xl mx-auto px-6 lg:px-8 py-6">
            <div className="flex items-center gap-3 mb-4">
              <FiFilter className="w-5 h-5 text-gray-600" />
              <h2 className="text-lg font-semibold text-gray-900">Filtros de búsqueda</h2>
              {hayFiltrosActivos && (
                <span className="inline-flex items-center rounded-full bg-secondary-100 px-2.5 py-0.5 text-xs font-medium text-secondary-800">
                  Filtros activos
                </span>
              )}
            </div>

            <div className="grid gap-3 md:grid-cols-2 lg:grid-cols-6">
              {/* Búsqueda */}
              <div className="md:col-span-2 relative">
                <FiSearch className="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 w-4 h-4" />
                <input
                  value={q}
                  onChange={(e) => setQ(e.target.value)}
                  onKeyDown={(e) => e.key === "Enter" && apply()}
                  placeholder="Buscar por título, autores, revista, editorial…"
                  className="w-full rounded-lg border border-gray-300 bg-white pl-11 pr-4 py-2.5 text-sm text-gray-900 placeholder:text-gray-500 focus:outline-none focus:ring-2 focus:ring-secondary-500 focus:border-transparent transition-shadow"
                />
              </div>

              {/* Estado */}
              <div className="relative">
                <select
                  value={fEstado}
                  onChange={(e) => setFEstado(e.target.value)}
                  className="w-full appearance-none rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm text-gray-900 focus:outline-none focus:ring-2 focus:ring-secondary-500 focus:border-transparent cursor-pointer transition-shadow"
                >
                  <option value="">Estado: Todos</option>
                  {estados.map((e) => <option key={e} value={e}>{e === 'publicado' ? 'Publicado' : 'En Prensa'}</option>)}
                </select>
                <FiChevronDown className="absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 w-4 h-4 pointer-events-none" />
              </div>

              {/* Año exacto */}
              <div className="relative">
                <select
                  value={fAnio}
                  onChange={(e) => setFAnio(e.target.value)}
                  className="w-full appearance-none rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm text-gray-900 focus:outline-none focus:ring-2 focus:ring-secondary-500 focus:border-transparent cursor-pointer transition-shadow"
                >
                  <option value="">Año: Todos</option>
                  {years.map((y) => <option key={y} value={y}>{y}</option>)}
                </select>
                <FiChevronDown className="absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 w-4 h-4 pointer-events-none" />
              </div>

              {/* Rango de años */}
              <div className="flex gap-2">
                <select
                  value={fDesde}
                  onChange={(e) => setFDesde(e.target.value)}
                  className="w-full rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm focus:outline-none focus:ring-2 focus:ring-secondary-500"
                >
                  <option value="">Desde</option>
                  {years.map((y) => <option key={y} value={y}>{y}</option>)}
                </select>
                <select
                  value={fHasta}
                  onChange={(e) => setFHasta(e.target.value)}
                  className="w-full rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm focus:outline-none focus:ring-2 focus:ring-secondary-500"
                >
                  <option value="">Hasta</option>
                  {years.map((y) => <option key={y} value={y}>{y}</option>)}
                </select>
              </div>

              {/* Orden */}
              <div className="flex gap-2">
                <select
                  value={sort}
                  onChange={(e) => setSort(e.target.value)}
                  className="w-full rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm focus:outline-none focus:ring-2 focus:ring-secondary-500"
                >
                  <option value="anio">Orden: Año</option>
                  <option value="citado_por">Orden: Citado</option>
                  <option value="titulo">Orden: Título</option>
                </select>
                <select
                  value={dir}
                  onChange={(e) => setDir(e.target.value)}
                  className="w-full rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm focus:outline-none focus:ring-2 focus:ring-secondary-500"
                >
                  <option value="desc">Desc</option>
                  <option value="asc">Asc</option>
                </select>
              </div>

              {/* Botones */}
              <div className="flex gap-2 md:col-span-2 lg:col-span-6">
                <button
                  onClick={apply}
                  className="inline-flex items-center gap-2 rounded-lg bg-gradient-to-r from-secondary-600 to-primary-600 px-4 py-2.5 text-sm font-semibold text-white hover:opacity-90 transition"
                >
                  <FiZap /> Aplicar filtros
                </button>
                <button
                  onClick={resetFiltros}
                  disabled={!hayFiltrosActivos}
                  className="inline-flex items-center gap-2 rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm font-semibold text-gray-700 hover:bg-gray-50 disabled:opacity-50"
                >
                  <FiX /> Limpiar
                </button>
                <label className="ml-auto inline-flex items-center gap-2 text-sm text-gray-600 cursor-pointer">
                  <input type="checkbox" checked={autoLoad} onChange={(e)=>setAutoLoad(e.target.checked)} />
                  Auto-cargar al final
                </label>
              </div>
            </div>
          </div>
        </section>

        {/* LISTADO */}
        <section className="py-12 sm:py-16">
          <div className="max-w-7xl mx-auto px-6 lg:px-8">
            {items.length === 0 ? (
              <motion.div initial={{ opacity: 0, scale: 0.95 }} animate={{ opacity: 1, scale: 1 }} className="text-center py-20">
                <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-gray-100 mb-4">
                  <FiSearch className="w-8 h-8 text-gray-400" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 mb-2">No se encontraron publicaciones</h3>
                <p className="text-sm text-gray-600 mb-6">Ajusta los filtros y vuelve a intentar.</p>
                <button onClick={resetFiltros} className="inline-flex items-center gap-2 rounded-lg bg-secondary-600 px-4 py-2.5 text-sm font-semibold text-white hover:bg-secondary-700 transition">
                  <FiX /> Limpiar filtros
                </button>
              </motion.div>
            ) : (
              <>
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 lg:gap-8">
                  {items.map((pub) => <PublicacionCard key={pub.id} pub={pub} />)}
                </div>

                {/* Paginación tipo Landing */}
                <div className="mt-10 flex flex-col items-center">
                  {nextUrl ? (
                    <button
                      onClick={loadMore}
                      disabled={loadingMore}
                      className="inline-flex items-center gap-2 rounded-full bg-gradient-to-r from-primary-600 to-secondary-600 px-6 py-3 text-sm font-semibold text-white shadow hover:opacity-95 disabled:opacity-60"
                    >
                      {loadingMore ? <FiLoader className="animate-spin" /> : <FiPlusCircle />}
                      {loadingMore ? "Cargando..." : "Mostrar más"}
                    </button>
                  ) : (
                    <div className="text-sm text-gray-500 mt-2">Has llegado al final</div>
                  )}

                  {/* Sentinela para auto-carga */}
                  <div ref={sentinelRef} className="h-2 w-full" />
                </div>
              </>
            )}
          </div>
        </section>
      </div>
    </PublicLayout>
  );
}
