import React, { useMemo, useState } from "react";
import { router } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import {
  FiSearch,
  FiFilter,
  FiX,
  FiChevronDown,
  FiChevronUp,
  FiUser,
  FiTag,
  FiCalendar,
  FiBookOpen,
} from "react-icons/fi";
import PublicLayout from "@/Layouts/PublicLayout";

/* =========================
   Helpers / Animations
   ========================= */
const fadeUp = (d = 0) => ({
  initial: { opacity: 0, y: 20 },
  whileInView: { opacity: 1, y: 0 },
  transition: { duration: 0.4, ease: [0.22, 1, 0.36, 1], delay: d },
  viewport: { once: true, amount: 0.15 },
});
const estadoBadge = (estado) =>
  estado === "Completado"
    ? "bg-secondary-50 text-secondary-700 border-secondary-200"
    : "bg-primary-50 text-primary-700 border-primary-200";

/* =========================
   Tarjeta de proyecto
   ========================= */
function ProyectoCard({ p }) {
  // Mapear nombres del backend a los usados en la tarjeta original
  const lineaInvestigacion = p.linea_investigacion ?? p.lineaInvestigacion;
  const profesorACargo = p.profesor_a_cargo ?? p.profesorACargo;
  const [open, setOpen] = useState(false);

  return (
    <motion.article
      {...fadeUp(0.05)}
      className="group relative rounded-2xl border border-gray-200 bg-white shadow-sm hover:shadow-md transition-all duration-300 overflow-hidden"
    >
      {/* Decoración superior */}
      <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-primary-500 via-secondary-500 to-primary-500" />

      {/* Header */}
      <div className="flex items-center justify-between px-6 py-4 border-b border-gray-100 bg-gradient-to-br from-gray-50 to-white">
        <span
          className={`inline-flex items-center gap-1.5 rounded-lg px-3 py-1.5 text-xs font-semibold border ${estadoBadge(
            p.estado
          )}`}
        >
          <div
            className={`w-1.5 h-1.5 rounded-full ${
              p.estado === "Completado" ? "bg-secondary-500" : "bg-primary-500"
            }`}
          />
          {p.estado}
        </span>
        <div className="inline-flex items-center gap-1.5 text-xs font-medium text-gray-600">
          <FiCalendar className="w-3.5 h-3.5" />
          <span>{p.ano}</span>
        </div>
      </div>

      <div className="p-6">
        {/* Título */}
        <h3 className="text-xl font-bold text-gray-900 leading-tight mb-4 group-hover:text-primary-700 transition-colors">
          {p.titulo}
        </h3>

        {/* Línea de investigación */}
        <div className="mb-4">
          <div className="inline-flex items-center gap-2 bg-primary-50 text-primary-800 px-3 py-1.5 rounded-lg text-xs font-medium border border-primary-100">
            <FiTag className="w-3.5 h-3.5" />
            <span>{lineaInvestigacion}</span>
          </div>
        </div>

        {/* Profesor */}
        <div className="mb-4 pb-4 border-b border-gray-100">
          <div className="flex items-start gap-2 text-sm text-gray-700">
            <FiUser className="w-4 h-4 mt-0.5 text-secondary-600 flex-shrink-0" />
            <span className="font-medium">{profesorACargo}</span>
          </div>
        </div>

        {/* Descripción breve (el controlador ya puede acortar si quieres) */}
        <p className="text-sm text-gray-600 leading-relaxed line-clamp-3 mb-4">
          {p.descripcion}
        </p>

        {/* Keywords */}
        {Array.isArray(p.keywords) && p.keywords.length > 0 && (
          <div className="flex flex-wrap gap-2 mb-4">
            {p.keywords.map((k) => (
              <span
                key={k}
                className="inline-flex items-center rounded-md bg-gray-50 px-2.5 py-1 text-xs font-medium text-gray-700 border border-gray-200"
              >
                {k}
              </span>
            ))}
          </div>
        )}

        {/* Botón ver detalles */}
        <button
          onClick={() => setOpen((v) => !v)}
          className="inline-flex items-center gap-1.5 text-sm font-semibold text-primary-700 hover:text-primary-800 transition-colors group/btn"
        >
          <FiBookOpen className="w-4 h-4" />
          <span>{open ? "Ver menos" : "Ver más detalles"}</span>
          {open ? (
            <FiChevronUp className="w-4 h-4 transition-transform group-hover/btn:-translate-y-0.5" />
          ) : (
            <FiChevronDown className="w-4 h-4 transition-transform group-hover/btn:translate-y-0.5" />
          )}
        </button>

        {/* Detalles expandibles */}
        <AnimatePresence initial={false}>
          {open && (
            <motion.div
              initial={{ height: 0, opacity: 0 }}
              animate={{ height: "auto", opacity: 1 }}
              exit={{ height: 0, opacity: 0 }}
              transition={{ duration: 0.3, ease: [0.22, 1, 0.36, 1] }}
              className="overflow-hidden"
            >
              <div className="mt-4 rounded-xl border border-primary-100 bg-primary-50/30 p-4">
                <h4 className="text-sm font-semibold text-gray-900 mb-2">
                  Descripción completa
                </h4>
                <p className="text-sm text-gray-700 leading-relaxed">
                  {p.descripcion}
                </p>
              </div>
            </motion.div>
          )}
        </AnimatePresence>
      </div>
    </motion.article>
  );
}

/* =========================
   Página principal (con datos del backend)
   ========================= */
export default function ProyectosLanding({ auth, proyectos, lineas, estados, anos, filters }) {
  // Opciones de filtros (from backend)
  const lineasOpts = useMemo(
    () => ["Todas", ...(Array.isArray(lineas) ? lineas : [])],
    [lineas]
  );
  const aniosOpts = useMemo(
    () => ["Todos", ...(Array.isArray(anos) ? anos : [])],
    [anos]
  );
  const estadosOpts = useMemo(
    () => ["Todos", ...(Array.isArray(estados) ? estados : [])],
    [estados]
  );

  // Estado local sincronizado con filtros iniciales
  const [q, setQ] = useState(filters?.q ?? "");
  const [fLinea, setFLinea] = useState(filters?.linea || "Todas");
  const [fAnio, setFAnio] = useState(
    typeof filters?.ano === "number" || filters?.ano ? String(filters.ano) : "Todos"
  );
  const [fEstado, setFEstado] = useState(filters?.estado || "Todos");

  const apply = () => {
    router.get(
      route("investigacion.proyectos-investigacion"),
      {
        q,
        linea: fLinea === "Todas" ? "" : fLinea,
        ano: fAnio === "Todos" ? "" : fAnio,
        estado: fEstado === "Todos" ? "" : fEstado,
        page: 1,
      },
      { preserveScroll: true, replace: true }
    );
  };

  const resetFiltros = () => {
    setQ("");
    setFLinea("Todas");
    setFAnio("Todos");
    setFEstado("Todos");
    router.get(route("landing.investigacion.proyectos"), {}, { preserveScroll: true, replace: true });
  };

  const hayFiltrosActivos = q || fLinea !== "Todas" || fAnio !== "Todos" || fEstado !== "Todos";

  const goto = (url) => url && router.visit(url, { preserveScroll: true, replace: true });

  return (
    <PublicLayout auth={auth} titulo="Proyectos de Investigación">
      <div className="min-h-screen bg-gray-50">
        {/* HERO */}
        <section className="relative bg-gradient-to-br from-primary-600 via-primary-700 to-secondary-700 overflow-hidden">
          {/* Patrón decorativo */}
          <div className="absolute inset-0 opacity-10">
            <div className="absolute top-0 left-1/4 w-96 h-96 bg-white rounded-full blur-3xl" />
            <div className="absolute bottom-0 right-1/4 w-96 h-96 bg-secondary-300 rounded-full blur-3xl" />
          </div>

          <div className="relative max-w-7xl mx-auto px-6 lg:px-8 py-20 sm:py-28">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.6 }}
              className="text-center"
            >
              <h1 className="text-4xl sm:text-5xl lg:text-6xl font-black tracking-tight text-white mb-6">
                Proyectos de Investigación
              </h1>
              <p className="text-lg sm:text-xl text-primary-100 max-w-3xl mx-auto leading-relaxed">
                Exploramos, innovamos y conectamos ciencia con sociedad a través de investigación de vanguardia
              </p>
              <div className="mt-8 flex items-center justify-center gap-6 text-sm text-primary-100">
                <div className="flex items-center gap-2">
                  <div className="w-2 h-2 rounded-full bg-secondary-400" />
                  <span>{proyectos?.total ?? 0} Proyectos</span>
                </div>
                <div className="flex items-center gap-2">
                  <div className="w-2 h-2 rounded-full bg-secondary-400" />
                  <span>{(lineas || []).length} Líneas de Investigación</span>
                </div>
              </div>
            </motion.div>
          </div>
        </section>

        {/* FILTROS */}
        <section className="bg-white border-b border-gray-200 sticky top-0 z-10 shadow-sm">
          <div className="max-w-7xl mx-auto px-6 lg:px-8 py-6">
            <div className="flex items-center gap-3 mb-4">
              <FiFilter className="w-5 h-5 text-gray-600" />
              <h2 className="text-lg font-semibold text-gray-900">Filtros de búsqueda</h2>
              {hayFiltrosActivos && (
                <span className="inline-flex items-center rounded-full bg-primary-100 px-2.5 py-0.5 text-xs font-medium text-primary-800">
                  {proyectos?.total ?? 0} {proyectos?.total === 1 ? "resultado" : "resultados"}
                </span>
              )}
            </div>

            <div className="grid gap-3 sm:grid-cols-2 lg:grid-cols-6">
              {/* Búsqueda */}
              <div className="sm:col-span-2 relative">
                <FiSearch className="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 w-4 h-4" />
                <input
                  value={q}
                  onChange={(e) => setQ(e.target.value)}
                  onKeyDown={(e) => e.key === "Enter" && apply()}
                  placeholder="Buscar proyectos, keywords..."
                  className="w-full rounded-lg border border-gray-300 bg-white pl-11 pr-4 py-2.5 text-sm text-gray-900 placeholder:text-gray-500 focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent transition-shadow"
                />
              </div>

              {/* Línea */}
              <div className="relative">
                <select
                  value={fLinea}
                  onChange={(e) => setFLinea(e.target.value)}
                  className="w-full appearance-none rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm text-gray-900 focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent cursor-pointer transition-shadow"
                >
                  {lineasOpts.map((l) => (
                    <option key={l}>{l}</option>
                  ))}
                </select>
                <FiChevronDown className="absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 w-4 h-4 pointer-events-none" />
              </div>

              {/* Año */}
              <div className="relative">
                <select
                  value={fAnio}
                  onChange={(e) => setFAnio(e.target.value)}
                  className="w-full appearance-none rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm text-gray-900 focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent cursor-pointer transition-shadow"
                >
                  {aniosOpts.map((a) => (
                    <option key={a}>{a}</option>
                  ))}
                </select>
                <FiChevronDown className="absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 w-4 h-4 pointer-events-none" />
              </div>

              {/* Estado */}
              <div className="relative">
                <select
                  value={fEstado}
                  onChange={(e) => setFEstado(e.target.value)}
                  className="w-full appearance-none rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm text-gray-900 focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent cursor-pointer transition-shadow"
                >
                  {estadosOpts.map((e) => (
                    <option key={e}>{e}</option>
                  ))}
                </select>
                <FiChevronDown className="absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 w-4 h-4 pointer-events-none" />
              </div>

              {/* Acciones */}
              <div className="flex gap-2">
                <button
                  onClick={resetFiltros}
                  className="w-full inline-flex items-center justify-center gap-2 rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm font-semibold text-gray-700 hover:bg-gray-50 transition-colors"
                >
                  <FiX className="w-4 h-4" />
                  <span>Limpiar</span>
                </button>
                <button
                  onClick={apply}
                  className="w-full inline-flex items-center justify-center gap-2 rounded-lg bg-primary-600 px-4 py-2.5 text-sm font-semibold text-white hover:bg-primary-700 transition-colors"
                >
                  Aplicar
                </button>
              </div>
            </div>
          </div>
        </section>

        {/* LISTADO */}
        <section className="py-12 sm:py-16">
          <div className="max-w-7xl mx-auto px-6 lg:px-8">
            {proyectos?.data?.length === 0 ? (
              <motion.div
                initial={{ opacity: 0, scale: 0.95 }}
                animate={{ opacity: 1, scale: 1 }}
                className="text-center py-20"
              >
                <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-gray-100 mb-4">
                  <FiSearch className="w-8 h-8 text-gray-400" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 mb-2">
                  No se encontraron proyectos
                </h3>
                <p className="text-sm text-gray-600 mb-6">
                  Intenta ajustar los filtros de búsqueda para obtener más resultados
                </p>
                <button
                  onClick={resetFiltros}
                  className="inline-flex items-center gap-2 rounded-lg bg-primary-600 px-4 py-2.5 text-sm font-semibold text-white hover:bg-primary-700 transition-colors"
                >
                  <FiX className="w-4 h-4" />
                  Limpiar filtros
                </button>
              </motion.div>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-6 lg:gap-8">
                {proyectos.data.map((p) => (
                  <ProyectoCard key={p.id} p={p} />
                ))}
              </div>
            )}
          </div>
        </section>

        {/* Paginación */}
        <div className="mt-6 mb-12 max-w-7xl mx-auto px-6 lg:px-8 flex items-center justify-between text-sm">
          <div className="text-primary-700/80 dark:text-primary-200/80">
            Mostrando {proyectos?.from ?? 0}–{proyectos?.to ?? 0} de {proyectos?.total ?? 0}
          </div>
          <div className="flex gap-2">
            <button
              disabled={!proyectos?.prev_page_url}
              onClick={() => goto(proyectos.prev_page_url)}
              className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
            >
              Anterior
            </button>
            <button
              disabled={!proyectos?.next_page_url}
              onClick={() => goto(proyectos.next_page_url)}
              className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
            >
              Siguiente
            </button>
          </div>
        </div>
      </div>
    </PublicLayout>
  );
}
