import React, { useMemo, useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import {
  FiSearch,
  FiFilter,
  FiX,
  FiUser,
  FiLayers,
  FiCalendar,
  FiCode,
  FiStar,
  FiInfo,
  FiChevronRight,
} from "react-icons/fi";
import PublicLayout from "@/Layouts/PublicLayout";

/* =========================
   DATA DEMO
   ========================= */


/* =========================
   Helpers
   ========================= */
const fadeIn = (d = 0) => ({
  initial: { opacity: 0, scale: 0.95 },
  whileInView: { opacity: 1, scale: 1 },
  transition: { duration: 0.5, ease: [0.22, 1, 0.36, 1], delay: d },
  viewport: { once: true, amount: 0.1 },
});

const slideIn = (direction = 'left', d = 0) => ({
  initial: { opacity: 0, x: direction === 'left' ? -40 : 40 },
  whileInView: { opacity: 1, x: 0 },
  transition: { duration: 0.6, ease: [0.22, 1, 0.36, 1], delay: d },
  viewport: { once: true, amount: 0.2 },
});

const uniq = (arr) => Array.from(new Set(arr));

// Colores degradados únicos por línea
const gradientColors = [
  'from-primary-500 to-primary-700',
  'from-secondary-500 to-secondary-700',
  'from-primary-600 to-secondary-600',
  'from-secondary-600 to-primary-600',
  'from-primary-700 to-secondary-500',
  'from-secondary-700 to-primary-500',
  'from-primary-500 to-secondary-700',
];

/* =========================
   Tarjeta de línea de investigación
   ========================= */
function LineaCard({ linea, index }) {
  const [isHovered, setIsHovered] = useState(false);
  const gradient = gradientColors[index % gradientColors.length];

  return (
    <motion.article
      {...fadeIn(index * 0.1)}
      onHoverStart={() => setIsHovered(true)}
      onHoverEnd={() => setIsHovered(false)}
      className="group relative rounded-3xl overflow-hidden bg-white shadow-lg hover:shadow-2xl transition-all duration-500 h-full flex flex-col"
    >
      {/* Fondo degradado animado */}
      <div className={`absolute inset-0 bg-gradient-to-br ${gradient} opacity-0 group-hover:opacity-5 transition-opacity duration-500`} />

      {/* Header con gradiente - altura consistente */}
      <div className={`relative bg-gradient-to-br ${gradient} p-8 pb-24 min-h-[220px] sm:min-h-[240px]`}>
        <div className="relative z-10">
          {/* Badge de tipo */}
          <div className="flex items-center justify-between mb-4">
            <span className={`inline-flex items-center gap-2 rounded-full px-4 py-1.5 text-xs font-bold ${
              linea.tipo === 'Prioritaria'
                ? 'bg-white/20 text-white backdrop-blur-sm border border-white/30'
                : 'bg-white/10 text-white/90 backdrop-blur-sm border border-white/20'
            }`}>
              {linea.tipo === 'Prioritaria' ? <FiStar className="w-3.5 h-3.5" /> : <FiLayers className="w-3.5 h-3.5" />}
              {linea.tipo === 'Prioritaria' ? 'Línea Prioritaria' : 'Línea de Investigación'}
            </span>

            <div className="text-xs font-semibold text-white/90 bg-white/10 backdrop-blur-sm rounded-full px-3 py-1.5 border border-white/20">
              {linea.fecha}
            </div>
          </div>

          {/* Título */}
          <h3 className="text-2xl font-black text-white leading-tight mb-3">
            {linea.nombre}
          </h3>

          {/* Código */}
          <div className="inline-flex items-center gap-2 text-sm font-mono text-white/90 bg-white/10 backdrop-blur-sm rounded-lg px-3 py-1.5 border border-white/20">
            <FiCode className="w-3.5 h-3.5" />
            {linea.codigo}
          </div>
        </div>

        {/* Decoración circular */}
        <div className="absolute -bottom-20 -right-20 w-40 h-40 bg-white/10 rounded-full blur-2xl" />
        <div className="absolute -bottom-10 -right-10 w-32 h-32 bg-white/5 rounded-full blur-xl" />
      </div>

      {/* Contenido */}
      <div className="relative -mt-16 px-8 pb-8 flex-1 flex flex-col">
        {/* Card de información blanco que crece */}
        <div className="relative bg-white rounded-2xl shadow-xl border border-gray-100 p-6 mb-6 flex-1 flex flex-col">
          {/* Profesor */}
          <div className="flex items-start gap-3 mb-4">
            <div className={`flex-shrink-0 w-10 h-10 rounded-full bg-gradient-to-br ${gradient} flex items-center justify-center`}>
              <FiUser className="w-5 h-5 text-white" />
            </div>
            <div className="flex-1 min-w-0">
              <div className="text-xs font-semibold text-gray-500 uppercase tracking-wider mb-1">
                Investigador Principal
              </div>
              <div className="text-sm font-bold text-gray-900 leading-snug">
                {linea.profesor}
              </div>
            </div>
          </div>

          {/* Descripción (clamp a 4 líneas sin plugin) */}
          <p
            className="text-sm text-gray-700 leading-relaxed"
            style={{
              display: '-webkit-box',
              WebkitLineClamp: 4,
              WebkitBoxOrient: 'vertical',
              overflow: 'hidden'
            }}
          >
            {linea.descripcion}
          </p>

          {/* Separador flexible para empujar el botón hacia abajo si hay poco texto */}
          <div className="mt-auto" />
        </div>

        {/* Botón de acción al fondo */}

      </div>

      {/* Indicador hover */}
      <motion.div
        initial={{ scaleX: 0 }}
        animate={{ scaleX: isHovered ? 1 : 0 }}
        transition={{ duration: 0.3 }}
        className={`absolute bottom-0 left-0 right-0 h-1 bg-gradient-to-r ${gradient} origin-left`}
      />
    </motion.article>
  );
}


/* =========================
   Vista de Lista
   ========================= */
function LineaListItem({ linea, index }) {
  const gradient = gradientColors[index % gradientColors.length];

  return (
    <motion.article
      {...slideIn('left', index * 0.05)}
      className="group relative rounded-2xl overflow-hidden bg-white shadow-md hover:shadow-xl transition-all duration-300 border border-gray-200"
    >
      <div className="flex flex-col md:flex-row">
        {/* Sidebar con gradiente */}
        <div className={`relative bg-gradient-to-br ${gradient} p-6 md:w-64 flex flex-col justify-between`}>
          <div>
            <span className={`inline-flex items-center gap-1.5 rounded-full px-3 py-1.5 text-xs font-bold mb-4 ${
              linea.tipo === 'Prioritaria'
                ? 'bg-white/20 text-white backdrop-blur-sm border border-white/30'
                : 'bg-white/10 text-white/90 backdrop-blur-sm border border-white/20'
            }`}>
              {linea.tipo === 'Prioritaria' ? <FiStar className="w-3 h-3" /> : <FiLayers className="w-3 h-3" />}
              {linea.tipo === 'Prioritaria' ? 'Prioritaria' : 'Línea'}
            </span>

            <h3 className="text-xl font-black text-white leading-tight mb-3">
              {linea.nombre}
            </h3>

            <div className="space-y-2 text-sm text-white/90">
              <div className="flex items-center gap-2">
                <FiCode className="w-3.5 h-3.5" />
                <span className="font-mono text-xs">{linea.codigo}</span>
              </div>
              <div className="flex items-center gap-2">
                <FiCalendar className="w-3.5 h-3.5" />
                <span className="text-xs">{linea.fecha}</span>
              </div>
            </div>
          </div>

          {/* Decoración */}
          <div className="absolute -bottom-10 -right-10 w-32 h-32 bg-white/10 rounded-full blur-2xl" />
        </div>

        {/* Contenido principal */}
        <div className="flex-1 p-6 flex flex-col justify-between">
          <div>
            {/* Profesor */}
            <div className="flex items-start gap-3 mb-4 pb-4 border-b border-gray-100">
              <div className={`flex-shrink-0 w-10 h-10 rounded-full bg-gradient-to-br ${gradient} flex items-center justify-center`}>
                <FiUser className="w-5 h-5 text-white" />
              </div>
              <div className="flex-1">
                <div className="text-xs font-semibold text-gray-500 uppercase tracking-wider mb-1">
                  Investigador Principal
                </div>
                <div className="text-sm font-bold text-gray-900">
                  {linea.profesor}
                </div>
              </div>
            </div>

            {/* Descripción */}
            <p className="text-sm text-gray-700 leading-relaxed mb-4">
              {linea.descripcion}
            </p>
          </div>

          {/* Botón */}
          <motion.button
            whileHover={{ x: 5 }}
            className={`self-start inline-flex items-center gap-2 rounded-lg bg-gradient-to-r ${gradient} px-5 py-2.5 text-sm font-semibold text-white shadow-md hover:shadow-lg transition-shadow group/btn`}
          >
            <span>Ver más</span>
            <FiChevronRight className="w-4 h-4 transition-transform group-hover/btn:translate-x-1" />
          </motion.button>
        </div>
      </div>
    </motion.article>
  );
}

/* =========================
   Página principal
   ========================= */
export default function LineasInvestigacion({lineas}) {
    const lineasInvestigacion = useMemo(() => Array.isArray(lineas) ? lineas : [], [lineas]);
  const tipos = ["Todos", "Prioritaria", "Linea"];

  const [q, setQ] = useState("");
  const [fTipo, setFTipo] = useState("Todos");
  const [vista, setVista] = useState("grid"); // 'grid' o 'list'

  const data = useMemo(() => {
    const text = q.trim().toLowerCase();
    return lineasInvestigacion.filter((linea) => {
      const matchText =
        !text ||
        linea.nombre.toLowerCase().includes(text) ||
        linea.descripcion.toLowerCase().includes(text) ||
        linea.profesor.toLowerCase().includes(text);
      const matchTipo = fTipo === "Todos" || linea.tipo === fTipo;
      return matchText && matchTipo;
    });
  }, [q, fTipo]);

  const resetFiltros = () => {
    setQ("");
    setFTipo("Todos");
  };

  const hayFiltrosActivos = q || fTipo !== "Todos";
  const lineasPrioritarias = lineasInvestigacion.filter(l => l.tipo === 'Prioritaria').length;

  return (
    <PublicLayout titulo={"Líneas de Investigación"}>
    <div className="min-h-screen bg-gradient-to-br from-gray-50 via-white to-gray-50">
      {/* HERO */}
      <section className="relative bg-gradient-to-br from-primary-900 via-secondary-900 to-primary-800 overflow-hidden">
        {/* Patrón de cuadrícula */}
        <div className="absolute inset-0 opacity-10">
          <div className="absolute inset-0" style={{
            backgroundImage: 'radial-gradient(circle, white 1px, transparent 1px)',
            backgroundSize: '40px 40px'
          }} />
        </div>

        {/* Elementos decorativos */}
        <div className="absolute top-20 right-20 w-72 h-72 bg-primary-400 rounded-full blur-3xl opacity-20" />
        <div className="absolute bottom-20 left-20 w-96 h-96 bg-secondary-400 rounded-full blur-3xl opacity-20" />

        <div className="relative max-w-7xl mx-auto px-6 lg:px-8 py-24 sm:py-32">
          <motion.div
            initial={{ opacity: 0, y: 30 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8 }}
            className="text-center"
          >
            <motion.div
              initial={{ scale: 0 }}
              animate={{ scale: 1 }}
              transition={{ duration: 0.5, delay: 0.2 }}
              className="inline-flex items-center justify-center w-20 h-20 rounded-3xl bg-white/10 backdrop-blur-lg mb-8 border border-white/20"
            >
              <FiLayers className="w-10 h-10 text-white" />
            </motion.div>

            <h1 className="text-4xl sm:text-5xl lg:text-6xl font-black tracking-tight text-white mb-6">
              Líneas de Investigación
            </h1>
            <p className="text-lg sm:text-xl text-primary-100 max-w-3xl mx-auto leading-relaxed mb-12">
              Áreas estratégicas de conocimiento que impulsan la innovación y el desarrollo tecnológico
            </p>

            {/* Métricas */}
            <div className="flex flex-wrap items-center justify-center gap-8 text-white/90">
              <div className="flex items-center gap-3">
                <div className="w-12 h-12 rounded-2xl bg-white/10 backdrop-blur-sm flex items-center justify-center border border-white/20">
                  <FiLayers className="w-6 h-6" />
                </div>
                <div className="text-left">
                  <div className="text-3xl font-black">{lineasInvestigacion.length}</div>
                  <div className="text-sm text-white/70">Líneas Activas</div>
                </div>
              </div>

              <div className="flex items-center gap-3">
                <div className="w-12 h-12 rounded-2xl bg-white/10 backdrop-blur-sm flex items-center justify-center border border-white/20">
                  <FiStar className="w-6 h-6" />
                </div>
                <div className="text-left">
                  <div className="text-3xl font-black">{lineasPrioritarias}</div>
                  <div className="text-sm text-white/70">Prioritarias</div>
                </div>
              </div>
            </div>
          </motion.div>
        </div>
      </section>

      {/* FILTROS */}
      <section className="sticky top-0 z-20 bg-white/80 backdrop-blur-xl border-b border-gray-200 shadow-sm">
        <div className="max-w-7xl mx-auto px-6 lg:px-8 py-6">
          <div className="flex items-center justify-between gap-4 mb-4">
            <div className="flex items-center gap-3">
              <FiFilter className="w-5 h-5 text-gray-600" />
              <h2 className="text-lg font-semibold text-gray-900">Explorar líneas</h2>
              {hayFiltrosActivos && (
                <span className="inline-flex items-center rounded-full bg-primary-100 px-2.5 py-0.5 text-xs font-medium text-primary-800">
                  {data.length} {data.length === 1 ? 'resultado' : 'resultados'}
                </span>
              )}
            </div>

            {/* Toggle vista */}
            <div className="flex items-center gap-2 bg-gray-100 rounded-lg p-1">
              <button
                onClick={() => setVista('grid')}
                className={`px-4 py-2 rounded-md text-sm font-semibold transition-colors ${
                  vista === 'grid'
                    ? 'bg-white text-gray-900 shadow-sm'
                    : 'text-gray-600 hover:text-gray-900'
                }`}
              >
                Cuadrícula
              </button>
              <button
                onClick={() => setVista('list')}
                className={`px-4 py-2 rounded-md text-sm font-semibold transition-colors ${
                  vista === 'list'
                    ? 'bg-white text-gray-900 shadow-sm'
                    : 'text-gray-600 hover:text-gray-900'
                }`}
              >
                Lista
              </button>
            </div>
          </div>

          <div className="grid gap-3 sm:grid-cols-4">
            {/* Búsqueda */}
            <div className="sm:col-span-2 relative">
              <FiSearch className="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 w-4 h-4" />
              <input
                value={q}
                onChange={(e) => setQ(e.target.value)}
                placeholder="Buscar por nombre, descripción, profesor..."
                className="w-full rounded-lg border border-gray-300 bg-white pl-11 pr-4 py-2.5 text-sm text-gray-900 placeholder:text-gray-500 focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent transition-shadow"
              />
            </div>

            {/* Tipo */}
            <div className="relative">
              <select
                value={fTipo}
                onChange={(e) => setFTipo(e.target.value)}
                className="w-full appearance-none rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm text-gray-900 focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent cursor-pointer transition-shadow"
              >
                {tipos.map((t) => (
                  <option key={t}>{t}</option>
                ))}
              </select>
              <FiChevronRight className="absolute right-3 top-1/2 -translate-y-1/2 rotate-90 text-gray-400 w-4 h-4 pointer-events-none" />
            </div>

            {/* Limpiar */}
            <button
              onClick={resetFiltros}
              disabled={!hayFiltrosActivos}
              className="inline-flex items-center justify-center gap-2 rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm font-semibold text-gray-700 hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
            >
              <FiX className="w-4 h-4" />
              Limpiar
            </button>
          </div>
        </div>
      </section>

      {/* LISTADO */}
      <section className="py-16">
        <div className="max-w-7xl mx-auto px-6 lg:px-8">
          {data.length === 0 ? (
            <motion.div
              initial={{ opacity: 0, scale: 0.95 }}
              animate={{ opacity: 1, scale: 1 }}
              className="text-center py-20"
            >
              <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-gray-100 mb-4">
                <FiSearch className="w-8 h-8 text-gray-400" />
              </div>
              <h3 className="text-lg font-semibold text-gray-900 mb-2">
                No se encontraron líneas de investigación
              </h3>
              <p className="text-sm text-gray-600 mb-6">
                Intenta ajustar los filtros para obtener más resultados
              </p>
              <button
                onClick={resetFiltros}
                className="inline-flex items-center gap-2 rounded-lg bg-primary-600 px-4 py-2.5 text-sm font-semibold text-white hover:bg-primary-700 transition-colors"
              >
                <FiX className="w-4 h-4" />
                Limpiar filtros
              </button>
            </motion.div>
          ) : vista === 'grid' ? (
            <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-8">
              {data.map((linea, index) => (
                <LineaCard key={linea.id} linea={linea} index={index} />
              ))}
            </div>
          ) : (
            <div className="space-y-6">
              {data.map((linea, index) => (
                <LineaListItem key={linea.id} linea={linea} index={index} />
              ))}
            </div>
          )}
        </div>
      </section>
    </div>
    </PublicLayout>
  );
}
