import React, { useEffect, useMemo, useRef, useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import {
  FiMapPin, FiUser, FiBookOpen, FiChevronLeft, FiChevronRight,
} from "react-icons/fi";
import PublicLayout from "@/Layouts/PublicLayout";

/* =========================
   THEME
   ========================= */
function classNames(...xs) { return xs.filter(Boolean).join(" "); }
const GRADIENT_PS = "from-primary-600 to-secondary-600";
const GRADIENT_PS_SOFT = "from-primary-500 to-secondary-500";

/* ⛰️ Imagen de fondo del HERO */
const HERO_IMAGE = "/assets/images/fondo_labo.png"; // ajusta a tu ruta real
const PLACEHOLDER = "/assets/images/placeholder_16x9.jpg"; // opcional

/* =========================
   Slider
   ========================= */
function ImageSlider({ images = [], gradient = GRADIENT_PS }) {
  const total = images.length || 1;
  const [idx, setIdx] = useState(0);
  const startX = useRef(0);
  const [direction, setDirection] = useState(0);

  const go = (n) => {
    if (!total) return;
    setDirection(n);
    setIdx((p) => (p + n + total) % total);
  };
  const goto = (i) => { setDirection(i > idx ? 1 : -1); setIdx(i); };
  const onTouchStart = (e) => (startX.current = e.touches[0].clientX);
  const onTouchEnd = (e) => {
    const dx = e.changedTouches[0].clientX - startX.current;
    if (Math.abs(dx) > 40) go(dx < 0 ? 1 : -1);
  };

  const slideVariants = {
    enter: (d) => ({ x: d > 0 ? 1000 : -1000, opacity: 0, scale: 0.95 }),
    center: { x: 0, opacity: 1, scale: 1 },
    exit:  (d) => ({ x: d < 0 ? 1000 : -1000, opacity: 0, scale: 0.95 }),
  };

  const srcs = total ? images : [PLACEHOLDER];

  return (
    <div className="relative group">
      <div
        className="relative overflow-hidden rounded-2xl"
        style={{ aspectRatio: "16 / 9" }}
        onTouchStart={onTouchStart}
        onTouchEnd={onTouchEnd}
      >
        <div className={classNames("absolute inset-0 bg-gradient-to-br opacity-20", gradient)} />

        <AnimatePresence initial={false} custom={direction} mode="wait">
          <motion.img
            key={idx}
            src={srcs[idx] || PLACEHOLDER}
            alt={`Imagen ${idx + 1}`}
            custom={direction}
            variants={slideVariants}
            initial="enter"
            animate="center"
            exit="exit"
            transition={{
              x: { type: "spring", stiffness: 300, damping: 30 },
              opacity: { duration: 0.25 },
              scale: { duration: 0.25 },
            }}
            className="absolute inset-0 h-full w-full object-cover"
            loading="lazy"
          />
        </AnimatePresence>

        <div className="absolute inset-0 bg-gradient-to-t from-black/50 via-transparent to-transparent pointer-events-none" />

        {srcs.length > 1 && (
          <>
            <motion.button
              whileHover={{ scale: 1.1, x: -5 }}
              whileTap={{ scale: 0.9 }}
              aria-label="Anterior"
              onClick={() => go(-1)}
              className="absolute left-4 top-1/2 -translate-y-1/2 grid place-items-center h-12 w-12 rounded-full
                         bg-white/90 backdrop-blur-sm shadow-xl opacity-0 group-hover:opacity-100
                         transition-opacity duration-300 hover:bg-white z-10"
            >
              <FiChevronLeft className="text-xl text-gray-800" />
            </motion.button>

            <motion.button
              whileHover={{ scale: 1.1, x: 5 }}
              whileTap={{ scale: 0.9 }}
              aria-label="Siguiente"
              onClick={() => go(1)}
              className="absolute right-4 top-1/2 -translate-y-1/2 grid place-items-center h-12 w-12 rounded-full
                         bg-white/90 backdrop-blur-sm shadow-xl opacity-0 group-hover:opacity-100
                         transition-opacity duration-300 hover:bg-white z-10"
            >
              <FiChevronRight className="text-xl text-gray-800" />
            </motion.button>

            <div className="absolute bottom-4 left-0 right-0 flex items-center justify-center gap-2">
              {srcs.map((_, i) => (
                <motion.button
                  key={i}
                  whileHover={{ scale: 1.3 }}
                  whileTap={{ scale: 0.9 }}
                  aria-label={`Ir a la imagen ${i + 1}`}
                  onClick={() => goto(i)}
                  className={classNames(
                    "h-2 rounded-full transition-all duration-300",
                    i === idx ? "w-8 bg-white shadow-lg" : "w-2 bg-white/60 hover:bg-white/80"
                  )}
                />
              ))}
            </div>
          </>
        )}
      </div>
    </div>
  );
}

/* =========================
   Tarjeta
   ========================= */
function LabBigCard({ lab }) {
  const nombre = lab?.nombre ?? "Laboratorio";
  const responsable = lab?.responsable ?? "—";
  const u = lab?.ubicacion ?? {};
  const asignaturas = Array.isArray(lab?.asignaturas) ? lab.asignaturas : [];
  const descripcion = lab?.descripcion ?? "";
  const imagenes = Array.isArray(lab?.imagenes) && lab.imagenes.length ? lab.imagenes : [PLACEHOLDER];

  return (
    <motion.section
      initial={{ opacity: 0, y: 40, scale: 0.98 }}
      animate={{ opacity: 1, y: 0, scale: 1 }}
      exit={{ opacity: 0, y: -40, scale: 0.98 }}
      transition={{ duration: 0.45, ease: [0.23, 1, 0.32, 1] }}
      className="rounded-3xl bg-white shadow-2xl overflow-hidden border border-gray-100"
    >
      <div className="grid grid-cols-1 lg:grid-cols-5 gap-0">
        {/* Slider */}
        <div className="lg:col-span-3 p-6 lg:p-8">
          <ImageSlider images={imagenes} gradient={GRADIENT_PS} />
        </div>

        {/* Detalles */}
        <div className="lg:col-span-2 p-8 lg:p-10 bg-gradient-to-br from-gray-50 to-white">
          <motion.div initial={{ opacity: 0, x: 20 }} animate={{ opacity: 1, x: 0 }} transition={{ delay: 0.15 }}>
            <div className={classNames("inline-block px-4 py-1 rounded-full bg-gradient-to-r text-white text-xs font-semibold mb-4", GRADIENT_PS)}>
              Laboratorio DETRI
            </div>
            <h2 className="text-3xl font-bold text-gray-900 leading-tight">{nombre}</h2>
          </motion.div>

          <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} transition={{ delay: 0.25 }} className="mt-8 space-y-5">
            <div className="flex items-start gap-3 group">
              <div className={classNames("mt-1 p-2 rounded-lg bg-gradient-to-br text-white group-hover:scale-110 transition-transform", GRADIENT_PS_SOFT)}>
                <FiUser className="text-lg" />
              </div>
              <div>
                <div className="text-xs font-semibold text-gray-500 uppercase tracking-wide">Responsable</div>
                <div className="text-gray-900 font-medium">{responsable || <span className="italic text-gray-400">No asignado</span>}</div>
              </div>
            </div>

            <div className="flex items-start gap-3 group">
              <div className={classNames("mt-1 p-2 rounded-lg bg-gradient-to-br text-white group-hover:scale-110 transition-transform", GRADIENT_PS_SOFT)}>
                <FiMapPin className="text-lg" />
              </div>
              <div>
                <div className="text-xs font-semibold text-gray-500 uppercase tracking-wide">Ubicación</div>
                <div className="text-gray-900 font-medium">
                  {u.edificio || u.piso || u.oficina ? (
                    <>Edificio {u.edificio ?? "—"} · Piso {u.piso ?? "—"} · Oficina {u.oficina ?? "—"}</>
                  ) : (
                    <span className="italic text-gray-400">No registrada</span>
                  )}
                </div>
              </div>
            </div>
          </motion.div>

          {asignaturas.length ? (
            <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} transition={{ delay: 0.35 }} className="mt-8">
              <div className="flex items-center gap-2 text-sm font-semibold text-gray-700 mb-3">
                <span className={classNames("p-1.5 rounded-lg text-white", "bg-gradient-to-br", GRADIENT_PS_SOFT)}>
                  <FiBookOpen />
                </span>
                Asignaturas
              </div>
              <div className="flex flex-wrap gap-2">
                {asignaturas.map((a, i) => (
                  <motion.span
                    key={`${a}-${i}`}
                    initial={{ opacity: 0, scale: 0.85 }}
                    animate={{ opacity: 1, scale: 1 }}
                    transition={{ delay: 0.45 + i * 0.06 }}
                    whileHover={{ scale: 1.05, y: -2 }}
                    className="rounded-full bg-white border border-gray-200 px-4 py-2 text-sm text-gray-700
                               shadow-sm hover:shadow-md hover:border-gray-300 transition-all cursor-default"
                  >
                    {a}
                  </motion.span>
                ))}
              </div>
            </motion.div>
          ) : null}

          {descripcion ? (
            <motion.p
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              transition={{ delay: 0.45 }}
              className="mt-8 text-base leading-relaxed text-gray-600 border-l-4 border-primary-200 pl-4"
            >
              {descripcion}
            </motion.p>
          ) : null}
        </div>
      </div>
    </motion.section>
  );
}

/* =========================
   Tabs
   ========================= */
function Tabs({ labs, current, onChange }) {
  if (!labs.length) return null;
  return (
    <div className="sticky top-0 z-20 bg-white/80 backdrop-blur-lg border-b border-gray-200 shadow-sm">
      <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8">
        <div className="flex gap-3 overflow-x-auto no-scrollbar py-4" role="tablist">
          {labs.map((l, i) => {
            const active = i === current;
            return (
              <motion.button
                key={l.id ?? l.slug ?? i}
                whileHover={{ scale: active ? 1 : 1.05, y: -2 }}
                whileTap={{ scale: 0.95 }}
                role="tab"
                aria-selected={active}
                onClick={() => onChange(i)}
                className={classNames(
                  "relative shrink-0 rounded-xl px-6 py-3 text-sm font-semibold transition-all duration-300",
                  active
                    ? classNames("bg-gradient-to-r text-white shadow-lg", GRADIENT_PS)
                    : "bg-gray-100 text-gray-600 hover:bg-gray-200 hover:text-gray-900"
                )}
              >
                {active && (
                  <motion.div
                    layoutId="activeTab"
                    className="absolute inset-0 rounded-xl"
                    style={{ background: `linear-gradient(to right, var(--tw-gradient-stops))` }}
                    transition={{ type: "spring", stiffness: 300, damping: 30 }}
                  />
                )}
                <span className="relative z-10">{l.nombre ?? "Laboratorio"}</span>
              </motion.button>
            );
          })}
        </div>
      </div>
    </div>
  );
}

/* =========================
   Página Principal
   ========================= */
export default function CentrosLaboratoriosIndex({ laboratorios = [] }) {
  // Usa los que vienen del backend
  const labs = useMemo(() => Array.isArray(laboratorios) ? laboratorios : [], [laboratorios]);
  const [tab, setTab] = useState(0);

  // Sin datos: mostrar mensaje agradable
  const isEmpty = labs.length === 0;

  // Sincroniza con query ?lab= (puede ser id o slug)
  useEffect(() => {
    if (isEmpty) return;
    const params = new URLSearchParams(window.location.search);
    const labParam = params.get("lab");
    if (labParam) {
      const idx = labs.findIndex((l) => String(l.id) === labParam || l.slug === labParam);
      if (idx >= 0) setTab(idx);
    }
  }, [labs, isEmpty]);

  // Navegación con flechas
  useEffect(() => {
    if (isEmpty) return;
    const onKey = (e) => {
      if (e.key === "ArrowRight") setTab((t) => (t + 1) % labs.length);
      if (e.key === "ArrowLeft") setTab((t) => (t - 1 + labs.length) % labs.length);
    };
    window.addEventListener("keydown", onKey);
    return () => window.removeEventListener("keydown", onKey);
  }, [labs.length, isEmpty]);

  const lab = labs[tab];

  return (
    <PublicLayout titulo={'Centros y Laboratorios'}>
      <div className="min-h-screen bg-gradient-to-br from-gray-50 via-white to-gray-50">
        {/* HERO */}
        <section className="relative overflow-hidden">
          <img
            src={HERO_IMAGE}
            alt="Centros y Laboratorios DETRI"
            className="absolute inset-0 h-full w-full object-cover"
            loading="eager"
          />
          <div className="absolute inset-0 bg-black/45" />
          <div className={classNames("absolute inset-0 bg-gradient-to-br mix-blend-multiply opacity-60", GRADIENT_PS)} />
          <div className="absolute inset-0 bg-grid-pattern opacity-[0.06]" />

          <div className="relative min-h-[40vh] lg:min-h-[50vh] flex items-center justify-center text-center px-6 py-16">
            <motion.div
              initial={{ opacity: 0, y: 30 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: [0.23, 1, 0.32, 1] }}
            >
              <motion.div
                initial={{ scale: 0.9, opacity: 0 }}
                animate={{ scale: 1, opacity: 1 }}
                transition={{ delay: 0.2 }}
                className={classNames("inline-block mb-4 px-4 py-2 rounded-full bg-white/10 text-white text-sm font-semibold border border-white/20")}
              >
                DETRI · Universidad EPN
              </motion.div>

              <h1 className="text-5xl sm:text-6xl lg:text-7xl font-black tracking-tight text-white mb-4">
                Centros y{" "}
                <span className={classNames("bg-clip-text text-transparent bg-gradient-to-r", GRADIENT_PS)}>
                  Laboratorios
                </span>
              </h1>

              <p className="mt-2 max-w-2xl mx-auto text-lg text-white/85">
                Explora nuestros espacios de innovación y tecnología
              </p>
            </motion.div>
          </div>
        </section>

        {/* Si no hay laboratorios */}
        {isEmpty ? (
          <section className="py-20">
            <div className="mx-auto max-w-3xl px-6 lg:px-8 text-center">
              <div className="inline-block rounded-2xl border border-gray-200 bg-white px-6 py-10 shadow-sm">
                <h3 className="text-2xl font-bold text-gray-900">Aún no hay laboratorios agregados</h3>
                <p className="mt-2 text-gray-600">
                  Cuando se registren laboratorios desde el panel, aparecerán aquí automáticamente.
                </p>
              </div>
            </div>
          </section>
        ) : (
          <>
            {/* PESTAÑAS */}
            <Tabs labs={labs} current={tab} onChange={setTab} />

            {/* TARJETA */}
            <section className="py-16">
              <div className="mx-auto max-w-7xl px-6 lg:px-8">
                <AnimatePresence mode="wait">
                  <LabBigCard key={lab?.id ?? lab?.slug ?? tab} lab={lab} />
                </AnimatePresence>

                {/* Navegación */}
                <motion.div
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: 0.2 }}
                  className="mt-10 flex items-center justify-between gap-4"
                >
                  <motion.button
                    whileHover={{ scale: 1.05, x: -5 }}
                    whileTap={{ scale: 0.95 }}
                    onClick={() => setTab((t) => (t - 1 + labs.length) % labs.length)}
                    className="inline-flex items-center gap-2 rounded-xl border-2 border-gray-200 px-6 py-3 text-sm font-semibold
                               text-gray-700 hover:border-gray-300 hover:bg-gray-50 transition-all shadow-sm hover:shadow-md"
                  >
                    <FiChevronLeft className="text-lg" /> Anterior
                  </motion.button>

                  <div className="text-sm text-gray-500 font-medium">
                    {tab + 1} / {labs.length}
                  </div>

                  <motion.button
                    whileHover={{ scale: 1.05, x: 5 }}
                    whileTap={{ scale: 0.95 }}
                    onClick={() => setTab((t) => (t + 1) % labs.length)}
                    className={classNames(
                      "inline-flex items-center gap-2 rounded-xl px-6 py-3 text-sm font-semibold text-white",
                      "bg-gradient-to-r shadow-md hover:shadow-lg transition-all",
                      GRADIENT_PS
                    )}
                  >
                    Siguiente <FiChevronRight className="text-lg" />
                  </motion.button>
                </motion.div>
              </div>
            </section>
          </>
        )}
      </div>
    </PublicLayout>
  );
}
