// resources/js/Pages/Detri/Ubicacion.jsx
import React from "react";
import { Head } from "@inertiajs/react";
import { motion, useReducedMotion } from "framer-motion";
import { FiMapPin, FiPhone, FiMail, FiExternalLink, FiNavigation } from "react-icons/fi";
import PublicLayout from "@/Layouts/PublicLayout";

/* =========================================================
   Configuración editable (coordenadas y textos)
   ========================================================= */
const COORDS = { lat: -0.2104, lng: -78.4892 }; // 🔁 Ajusta con precisión
const ADDRESS = "Escuela Politécnica Nacional, Quito, Ecuador"; // 🔁 Ajusta
const PHONE = "(+593) 2 000 0000"; // 🔁 Ajusta
const EMAIL = "detri@epn.edu.ec"; // 🔁 Ajusta
const PLACE_QUERY = encodeURIComponent(ADDRESS);
const GMAPS_LINK = `https://www.google.com/maps/search/?api=1&query=${COORDS.lat},${COORDS.lng}`;
const WAZE_LINK = `https://waze.com/ul?ll=${COORDS.lat},${COORDS.lng}&navigate=yes`;
const APPLE_MAPS = `https://maps.apple.com/?ll=${COORDS.lat},${COORDS.lng}&q=${PLACE_QUERY}`;

/* =========================================================
   Helpers de animación
   ========================================================= */
const fadeUp = (d = 0) => ({
  initial: { opacity: 0, y: 24 },
  animate: { opacity: 1, y: 0 },
  transition: { duration: 0.7, ease: [0.22, 1, 0.36, 1], delay: d },
});

function SectionTitle({ eyebrow, title, subtitle }) {
  return (
    <div className="mb-8 text-left">
      {eyebrow && (
        <motion.div {...fadeUp(0)} className="inline-block mb-3">
          <span className="text-xs tracking-widest font-semibold px-3 py-1 rounded-full bg-gray-100 text-gray-700 dark:bg-white/10 dark:text-white/90">
            {eyebrow}
          </span>
        </motion.div>
      )}
      {title && (
        <motion.h2 {...fadeUp(0.05)} className="text-3xl sm:text-4xl font-black text-gray-900 dark:text-white">
          {title}
        </motion.h2>
      )}
      {subtitle && (
        <motion.p {...fadeUp(0.1)} className="mt-3 text-lg text-gray-600 dark:text-gray-300 max-w-2xl">
          {subtitle}
        </motion.p>
      )}
      <motion.div
        {...fadeUp(0.15)}
        className="mt-4 h-1.5 w-28 bg-gradient-to-r from-blue-600 via-indigo-600 to-purple-600 rounded-full"
      />
    </div>
  );
}

/* =========================================================
   Bloques de UI
   ========================================================= */
function HeroOverlay({
  height = "h-[42vh] sm:h-[50vh]",
  image = "/assets/images/fondo_epn.png",
  badge = "VISÍTANOS",
  title = "Ubicación",
  subtitle = "Encuéntranos en el campus de la Escuela Politécnica Nacional.",
  overlay = "from-blue-900/90 via-indigo-900/85 to-purple-900/90",
}) {
  const reduce = useReducedMotion();
  return (
    <section className={`relative ${height} flex items-center justify-center overflow-hidden bg-black`}>
      {/* Imagen de fondo */}
      <div
        className="absolute inset-0 bg-cover bg-center"
        style={{ backgroundImage: `url('${image}')` }}
        role="img"
        aria-label="Imagen aérea del campus de la EPN"
      />
      {/* Overlay de color para contraste */}
      <div className={`absolute inset-0 bg-gradient-to-br ${overlay}`} />

      {/* Patrón decorativo sutil */}
      <div className="absolute inset-0 opacity-10 pointer-events-none">
        <div
          className="absolute inset-0"
          style={{
            backgroundImage: `radial-gradient(circle at 2px 2px, white 1px, transparent 0)`,
            backgroundSize: "42px 42px",
          }}
        />
      </div>

      {/* Blobs (desactivados si reduce motion) */}
      {!reduce && (
        <>
          <div className="absolute top-10 left-10 w-64 h-64 bg-blue-400 rounded-full blur-3xl opacity-20 animate-pulse" />
          <div
            className="absolute bottom-10 right-10 w-80 h-80 bg-purple-400 rounded-full blur-3xl opacity-20 animate-pulse"
            style={{ animationDelay: "0.6s" }}
          />
        </>
      )}

      {/* Contenido */}
      <div className="relative z-10 text-center text-white px-6 max-w-5xl mx-auto">
        <motion.div initial={{ opacity: 0, scale: 0.96 }} animate={{ opacity: 1, scale: 1 }} transition={{ duration: 0.8 }}>
          {badge && (
            <motion.div {...fadeUp(0.15)} className="inline-block mb-4">
              <div className="bg-white/10 backdrop-blur-sm rounded-full px-5 py-2 border border-white/20">
                <span className="text-sm font-semibold tracking-wide">{badge}</span>
              </div>
            </motion.div>
          )}
          <motion.h1 {...fadeUp(0.2)} className="text-4xl sm:text-5xl font-black tracking-tight mb-3">
            {title}
          </motion.h1>
          <motion.p {...fadeUp(0.25)} className="text-lg sm:text-xl text-blue-100 max-w-3xl mx-auto leading-relaxed">
            {subtitle}
          </motion.p>
          <motion.div {...fadeUp(0.35)} className="mt-6 flex items-center justify-center gap-3">
            <div className="h-1 w-20 rounded-full bg-gradient-to-r from-transparent via-white/60 to-white/60" />
            <div className="w-3 h-3 rounded-full bg-white/70" />
            <div className="h-1 w-20 rounded-full bg-gradient-to-r from-white/60 via-white/60 to-transparent" />
          </motion.div>
        </motion.div>
      </div>

      {/* Onda inferior */}
      <div className="absolute bottom-0 left-0 right-0">
        <svg className="w-full h-14 fill-white" viewBox="0 0 1200 120" preserveAspectRatio="none">
          <path d="M0,0V46.29c47.79,22.2,103.59,32.17,158,28,70.36-5.37,136.33-33.31,206.8-37.5C438.64,32.43,512.34,53.67,583,72.05c69.27,18,138.3,24.88,209.4,13.08,36.15-6,69.85-17.84,104.45-29.34C989.49,25,1113-14.29,1200,52.47V0Z" opacity=".25"></path>
          <path d="M0,0V15.81C13,36.92,27.64,56.86,47.69,72.05,99.41,111.27,165,111,224.58,91.58c31.15-10.15,60.09-26.07,89.67-39.8,40.92-19,84.73-46,130.83-49.67,36.26-2.85,70.9,9.42,98.6,31.56,31.77,25.39,62.32,62,103.63,73,40.44,10.79,81.35-6.69,119.13-24.28s75.16-39,116.92-43.05c59.73-5.85,113.28,22.88,168.9,38.84,30.2,8.66,59,6.17,87.09-7.5,22.43-10.89,48-26.93,60.65-49.24V0Z" opacity=".5"></path>
          <path d="M0,0V5.63C149.93,59,314.09,71.32,475.83,42.57c43-7.64,84.23-20.12,127.61-26.46,59-8.63,112.48,12.24,165.56,35.4C827.93,77.22,886,95.24,951.2,90c86.53-7,172.46-45.71,248.8-84.81V0Z"></path>
        </svg>
      </div>
    </section>
  );
}

function MapCard() {
  return (
    <div className="rounded-2xl overflow-hidden border border-primary-100/70 dark:border-white/10 bg-white dark:bg-zinc-900/60 shadow-md">
      <div className="aspect-video w-full">
        <iframe
          title="Mapa DETRI — EPN"
          loading="lazy"
          referrerPolicy="no-referrer-when-downgrade"
          className="w-full h-full"
          src={`https://www.google.com/maps?q=${COORDS.lat},${COORDS.lng}&z=16&output=embed`}
        />
      </div>
      <div className="p-5 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
        <div className="text-sm">
          <div className="flex items-center gap-2 font-medium text-ink dark:text-white">
            <FiMapPin /> {ADDRESS}
          </div>
          <div className="mt-1 text-ink/70 dark:text-white/70">
            Coordenadas: {COORDS.lat.toFixed(6)}, {COORDS.lng.toFixed(6)}
          </div>
        </div>
        <div className="flex flex-wrap gap-2">
          <a
            className="inline-flex items-center gap-2 rounded-xl bg-primary-600 text-white px-3 py-2 text-sm hover:bg-primary-700"
            href={GMAPS_LINK}
            target="_blank"
            rel="noreferrer"
          >
            Abrir en Google Maps <FiExternalLink />
          </a>
          <a
            className="inline-flex items-center gap-2 rounded-xl border px-3 py-2 text-sm text-ink dark:text-white border-primary-200/70 dark:border-white/10 hover:bg-primary-50 dark:hover:bg-white/5"
            href={WAZE_LINK}
            target="_blank"
            rel="noreferrer"
          >
            Waze
          </a>
          <a
            className="inline-flex items-center gap-2 rounded-xl border px-3 py-2 text-sm text-ink dark:text-white border-primary-200/70 dark:border-white/10 hover:bg-primary-50 dark:hover:bg-white/5"
            href={APPLE_MAPS}
            target="_blank"
            rel="noreferrer"
          >
            Apple Maps
          </a>
        </div>
      </div>
    </div>
  );
}

function ContactCard() {
  return (
    <motion.aside
      {...fadeUp(0.1)}
      className="rounded-2xl border border-primary-100/70 dark:border-white/10 bg-white dark:bg-zinc-900/60 p-6 shadow-md"
    >
      <h2 className="text-xl font-bold text-primary-700 dark:text-primary-300">Contacto</h2>
      <div className="mt-4 space-y-4 text-sm">
        <div className="flex items-start gap-2 text-ink dark:text-white">
          <FiMapPin className="mt-0.5" />
          <div>
            <div className="font-medium">Dirección</div>
            <div className="text-ink/70 dark:text-white/70">{ADDRESS}</div>
          </div>
        </div>
        <div className="flex items-start gap-2 text-ink dark:text-white">
          <FiPhone className="mt-0.5" />
          <div>
            <div className="font-medium">Teléfono</div>
            <div className="text-ink/70 dark:text-white/70">{PHONE}</div>
          </div>
        </div>
        <div className="flex items-start gap-2 text-ink dark:text-white">
          <FiMail className="mt-0.5" />
          <div>
            <div className="font-medium">Correo</div>
            <a className="text-primary-700 dark:text-primary-300 hover:underline" href={`mailto:${EMAIL}`}>
              {EMAIL}
            </a>
          </div>
        </div>
      </div>

      <div className="mt-6 grid grid-cols-1 gap-2">
        <a
          href={GMAPS_LINK}
          target="_blank"
          rel="noreferrer"
          className="inline-flex items-center justify-center gap-2 rounded-xl bg-primary-600 text-white px-4 py-2.5 text-sm font-semibold hover:bg-primary-700"
        >
          <FiNavigation /> Cómo llegar (Google Maps)
        </a>
        <a
          href="/docs/planos/campus-epn.pdf"
          className="inline-flex items-center justify-center gap-2 rounded-xl border px-4 py-2.5 text-sm text-ink dark:text-white border-primary-200/70 dark:border-white/10 hover:bg-primary-50 dark:hover:bg-white/5"
          target="_blank"
          rel="noreferrer"
        >
          Ver plano del campus (PDF)
        </a>
      </div>
    </motion.aside>
  );
}

/* =========================================================
   Página: Ubicación
   ========================================================= */
export default function Ubicacion() {
  return (
    <PublicLayout titulo={"Ubicación"}>
      <Head title="Ubicación" />

      {/* Hero con imagen de fondo */}
      <HeroOverlay />

      {/* Contenido principal */}
      <section className="py-12 bg-white dark:bg-ink">
        <div className="mx-auto max-w-7xl px-6 lg:px-8">
          <motion.div {...fadeUp(0)}>
            <SectionTitle
              eyebrow="INFORMACIÓN DE ACCESO"
              title="Cómo encontrarnos"
              subtitle="Estamos ubicados en el corazón del campus de la Escuela Politécnica Nacional. Planifica tu ruta y elige tu aplicación favorita para llegar con facilidad."
            />
          </motion.div>

          <div className="grid gap-8 lg:grid-cols-3">
            <div className="lg:col-span-2">
              <MapCard />
            </div>
            <ContactCard />
          </div>
        </div>
      </section>

      {/* Indicaciones */}
      <section className="pb-16 bg-white dark:bg-ink">
        <div className="mx-auto max-w-7xl px-6 lg:px-8">
          <motion.div
            {...fadeUp(0.05)}
            className="rounded-2xl border border-primary-100/70 dark:border-white/10 bg-white dark:bg-zinc-900/60 p-6 shadow-md"
          >
            <h2 className="text-xl font-bold text-primary-700 dark:text-primary-300">Indicaciones</h2>
            <div className="mt-4 grid gap-6 sm:grid-cols-2 lg:grid-cols-3 text-sm">
              <div>
                <h3 className="font-semibold">Transporte público</h3>
                <p className="mt-2 text-ink/70 dark:text-white/70">
                  Usa las líneas que pasan por La Marín / 12 de Octubre. Desciende en la parada más cercana al campus y sigue la señalética hacia DETRI. {/* 🔁 Ajusta con rutas reales si deseas */}
                </p>
              </div>
              <div>
                <h3 className="font-semibold">Vehículo particular</h3>
                <p className="mt-2 text-ink/70 dark:text-white/70">
                  Ingresa la dirección en tu navegador (Google Maps o Waze). Considera horarios de acceso y disponibilidad de parqueaderos.
                </p>
              </div>
              <div>
                <h3 className="font-semibold">Taxi / apps</h3>
                <p className="mt-2 text-ink/70 dark:text-white/70">
                  Indica al conductor “DETRI – Escuela Politécnica Nacional”. Verifica puerta de acceso habilitada según tu horario.
                </p>
              </div>
            </div>
          </motion.div>
        </div>
      </section>
    </PublicLayout>
  );
}
