import React, { useMemo, useState } from "react";
import { motion } from "framer-motion";
import {
  FiSearch, FiGlobe, FiExternalLink, FiBookOpen, FiFilter, FiMail,
} from "react-icons/fi";
import { FaGithub, FaLinkedin } from "react-icons/fa";
import { SiGooglescholar, SiOrcid, SiElsevier } from "react-icons/si";
import PublicLayout from "@/Layouts/PublicLayout";

/* =========================
   Helpers
   ========================= */
const fadeUp = (delay = 0) => ({
  initial: { opacity: 0, y: 20 },
  whileInView: { opacity: 1, y: 0 },
  transition: { duration: 0.5, ease: [0.22, 1, 0.36, 1], delay },
  viewport: { once: true, amount: 0.2 },
});

const unique = (arr) => Array.from(new Set(arr.filter(Boolean))).sort((a, b) => a.localeCompare(b));

/* =========================
   UI: Chips de asignaturas
   ========================= */
function SubjectChips({ items, onClick }) {
  if (!items || items.length === 0) return null;
  return (
    <div className="mt-3 flex flex-wrap gap-2">
      {items.map((s) => (
        <motion.button
          key={s}
          onClick={onClick ? () => onClick(s) : undefined}
          whileHover={{ scale: 1.05, y: -2 }}
          whileTap={{ scale: 0.95 }}
          className="rounded-full border border-blue-200 dark:border-blue-400/30 bg-gradient-to-r from-blue-50 to-indigo-50 dark:from-blue-900/20 dark:to-indigo-900/20 px-3.5 py-1.5 text-xs font-medium
                     text-blue-700 dark:text-blue-300 hover:shadow-md transition-shadow duration-200"
          title={`Filtrar por: ${s}`}
        >
          {s}
        </motion.button>
      ))}
    </div>
  );
}

/* =========================
   UI: Enlaces académicos
   ========================= */
function LinksRow({ links }) {
  const items = [
    { key: "web", icon: <FiGlobe />, label: "Sitio", color: "text-gray-600 dark:text-gray-300" },
    { key: "github", icon: <FaGithub />, label: "GitHub", color: "text-gray-800 dark:text-white" },
    { key: "scholar", icon: <SiGooglescholar />, label: "Scholar", color: "text-blue-600 dark:text-blue-400" },
    { key: "orcid", icon: <SiOrcid />, label: "ORCID", color: "text-green-600 dark:text-green-400" },
    { key: "scopus", icon: <SiElsevier />, label: "Scopus", color: "text-orange-600 dark:text-orange-400" },
    { key: "linkedin", icon: <FaLinkedin />, label: "LinkedIn", color: "text-blue-700 dark:text-blue-400" },
  ];

  if (!links || typeof links !== "object") return null;

  return (
    <div className="mt-5 pt-4 border-t border-gray-100 dark:border-gray-700">
      <div className="flex flex-wrap gap-3">
        {items.map(({ key, icon, label, color }) =>
          links?.[key] ? (
            <motion.a
              key={key}
              href={links[key]}
              target="_blank"
              rel="noreferrer"
              whileHover={{ scale: 1.08, y: -2 }}
              whileTap={{ scale: 0.95 }}
              className={`inline-flex items-center gap-1.5 ${color} hover:opacity-80 transition-all duration-200 text-sm font-medium`}
            >
              <span className="text-lg">{icon}</span>
              <span className="hidden sm:inline">{label}</span>
              <FiExternalLink className="text-xs opacity-60" />
            </motion.a>
          ) : null
        )}
      </div>
    </div>
  );
}

/* =========================
   UI: Card Docente
   ========================= */
function DocenteCard({ d, onSubjectClick, index }) {
  const nombre = d?.nombre ?? "Docente";
  const grado = d?.grado ?? "";
  const area = d?.area ?? "";
  const email = d?.email ?? "";
  const asignaturas = Array.isArray(d?.asignaturas) ? d.asignaturas : [];
  const enlaces = d?.enlaces && typeof d.enlaces === "object" ? d.enlaces : {};
  const foto = d?.foto || "/assets/docentes/placeholder.png";

  return (
    <motion.article
      {...fadeUp(index * 0.08)}
      whileHover={{ y: -8 }}
      className="group relative overflow-visible rounded-2xl bg-white dark:bg-gray-800 shadow-lg hover:shadow-2xl transition-all duration-300 border border-gray-100 dark:border-gray-700 mt-20"
    >
      {/* Gradiente decorativo superior */}
      <div className="absolute top-0 left-0 right-0 h-24 bg-gradient-to-br from-blue-500 via-indigo-500 to-purple-500 opacity-90 rounded-t-2xl" />

      {/* Patrón decorativo */}
      <div
        className="absolute top-0 left-0 right-0 h-28 opacity-10 rounded-t-2xl"
        style={{
          backgroundImage: `radial-gradient(circle at 2px 2px, white 1px, transparent 0)`,
          backgroundSize: "24px 24px",
        }}
      />

      {/* Contenedor de imagen */}
      <div className="relative -mt-20 mb-4 flex justify-center">
        <motion.div whileHover={{ scale: 1.05, y: -4 }} transition={{ duration: 0.3 }} className="relative">
          {/* Sombra */}
          <div className="absolute bottom-0 left-1/2 -translate-x-1/2 w-24 h-8 bg-black/10 dark:bg-black/20 blur-xl rounded-full" />
          {/* Imagen */}
          <img
            src={foto}
            alt={nombre}
            className="relative w-40 h-40 object-contain drop-shadow-2xl pointer-events-none"
          />
        </motion.div>
      </div>

      {/* Contenido */}
      <div className="relative px-6 pb-6 pt-16">
        {/* Información principal */}
        <div className="text-center mb-4">
          <h3 className="text-xl font-bold text-gray-900 dark:text-white mb-1">{nombre}</h3>
          {grado && (
            <p className="text-sm font-semibold text-indigo-600 dark:text-indigo-400 mb-1">{grado}</p>
          )}
          {area && (
            <div className="inline-flex items-center gap-2 px-3 py-1 rounded-full bg-gray-100 dark:bg-gray-700 text-xs font-medium text-gray-700 dark:text-gray-300">
              <span className="w-2 h-2 rounded-full bg-indigo-500" />
              {area}
            </div>
          )}
        </div>

        {/* Email */}
        {email && (
          <div className="flex items-center justify-center gap-2 mb-4 text-sm text-gray-600 dark:text-gray-400">
            <FiMail className="text-base" />
            <a href={`mailto:${email}`} className="hover:text-indigo-600 dark:hover:text-indigo-400 transition-colors">
              {email}
            </a>
          </div>
        )}

        {/* Asignaturas */}
        {asignaturas.length > 0 && (
          <div className="mb-4">
            <div className="inline-flex items-center gap-2 text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
              <FiBookOpen className="text-indigo-500" />
              Asignaturas
            </div>
            <SubjectChips items={asignaturas} onClick={onSubjectClick} />
          </div>
        )}

        {/* Enlaces */}
        <LinksRow links={enlaces} />
      </div>

      {/* Borde animado */}
      <motion.div
        className="absolute inset-0 rounded-2xl pointer-events-none"
        initial={{ opacity: 0 }}
        whileHover={{ opacity: 1 }}
        transition={{ duration: 0.3 }}
      >
        <div className="absolute inset-0 rounded-2xl ring-2 ring-indigo-500/50 ring-offset-2 ring-offset-white dark:ring-offset-gray-800" />
      </motion.div>
    </motion.article>
  );
}

/* =========================
   PAGE
   ========================= */
export default function PersonalDocente({ docentes = [] }) {
  // si no hay docentes en absoluto (backend vacío)
  const noDocentes = !Array.isArray(docentes) || docentes.length === 0;

  const [q, setQ] = useState("");
  const [area, setArea] = useState("Todas");
  const [materia, setMateria] = useState("Todas");

  // Construir opciones desde lo que viene del backend
  const AREAS = useMemo(() => {
    if (noDocentes) return ["Todas"];
    return ["Todas", ...unique(docentes.map((d) => d?.area || "").filter(Boolean))];
  }, [docentes, noDocentes]);

  const MATERIAS = useMemo(() => {
    if (noDocentes) return ["Todas"];
    const all = docentes.flatMap((d) => (Array.isArray(d?.asignaturas) ? d.asignaturas : []));
    return ["Todas", ...unique(all)];
  }, [docentes, noDocentes]);

  // Filtrado
  const data = useMemo(() => {
    if (noDocentes) return [];
    const text = q.trim().toLowerCase();

    return docentes.filter((d) => {
      const nombre = (d?.nombre || "").toLowerCase();
      const grado = (d?.grado || "").toLowerCase();
      const areaD = (d?.area || "").toLowerCase();
      const asignaturas = Array.isArray(d?.asignaturas) ? d.asignaturas : [];

      const hitText =
        !text ||
        nombre.includes(text) ||
        grado.includes(text) ||
        areaD.includes(text) ||
        asignaturas.some((a) => (a || "").toLowerCase().includes(text));

      const hitArea = area === "Todas" || d?.area === area;
      const hitMat = materia === "Todas" || asignaturas.includes(materia);

      return hitText && hitArea && hitMat;
    });
  }, [docentes, q, area, materia, noDocentes]);

  const handleSubjectClick = (s) => setMateria(s);

  const clearFilters = () => {
    setQ("");
    setArea("Todas");
    setMateria("Todas");
  };

  return (
    <PublicLayout titulo={"Personal Docente"}>
      <div className="min-h-screen bg-gradient-to-b from-gray-50 to-white dark:from-gray-900 dark:to-gray-800">
        {/* Hero */}
        <section className="relative h-[50vh] flex items-center justify-center overflow-hidden">
          <div className="absolute inset-0 bg-gradient-to-br from-blue-600 via-indigo-600 to-purple-700" />
          <div
            className="absolute inset-0 opacity-20"
            style={{
              backgroundImage: `radial-gradient(circle at 2px 2px, white 1px, transparent 0)`,
              backgroundSize: "32px 32px",
            }}
          />

          <motion.div
            initial={{ opacity: 0, y: 30 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, ease: [0.22, 1, 0.36, 1] }}
            className="relative z-10 px-6 text-center text-white max-w-4xl mx-auto"
          >
            <motion.div
              initial={{ scale: 0.8, opacity: 0 }}
              animate={{ scale: 1, opacity: 1 }}
              transition={{ duration: 0.6, delay: 0.2 }}
              className="inline-block mb-4"
            >
              <div className="bg-white/10 backdrop-blur-sm rounded-full px-6 py-2 border border-white/20">
                <span className="text-sm font-semibold">Excelencia Académica</span>
              </div>
            </motion.div>

            <h1 className="text-5xl sm:text-6xl font-extrabold tracking-tight mb-4">Personal Docente</h1>
            <p className="text-xl text-blue-100 max-w-2xl mx-auto leading-relaxed">
              Conoce a nuestros docentes, sus especialidades y trayectoria profesional
            </p>
          </motion.div>
        </section>

        {/* Si no hay docentes en absoluto */}
        {noDocentes ? (
          <section className="py-20">
            <div className="mx-auto max-w-3xl px-6 text-center">
              <motion.div
                initial={{ opacity: 0, scale: 0.95 }}
                animate={{ opacity: 1, scale: 1 }}
                className="rounded-2xl border border-gray-200 dark:border-gray-700 bg-white/80 dark:bg-gray-800/80 backdrop-blur p-10"
              >
                <div className="inline-flex items-center justify-center w-20 h-20 rounded-full bg-gray-100 dark:bg-gray-700 mb-4">
                  <FiSearch className="text-3xl text-gray-400" />
                </div>
                <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-2">
                  Aún no hay profesores agregados
                </h3>
                <p className="text-gray-600 dark:text-gray-400">
                  Vuelve más tarde. Estamos preparando la lista del personal docente.
                </p>
              </motion.div>
            </div>
          </section>
        ) : (
          <>
            {/* Filtros */}
            <section className="sticky top-0 z-30 bg-white/80 dark:bg-gray-800/80 backdrop-blur-lg border-b border-gray-200 dark:border-gray-700 shadow-sm">
              <div className="mx-auto max-w-7xl px-6 lg:px-8 py-6">
                <div className="grid gap-4 sm:grid-cols-3 items-end">
                  {/* Búsqueda */}
                  <div>
                    <label className="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2" htmlFor="q">
                      Buscar
                    </label>
                    <div className="relative">
                      <FiSearch className="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400" />
                      <input
                        id="q"
                        type="text"
                        value={q}
                        onChange={(e) => setQ(e.target.value)}
                        placeholder="Nombre, grado o asignatura..."
                        className="w-full rounded-xl border-2 border-gray-200 dark:border-gray-600 bg-white dark:bg-gray-700
                                   pl-11 pr-4 py-3 text-sm text-gray-900 dark:text-white placeholder:text-gray-400
                                   focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-transparent transition-all duration-200"
                      />
                    </div>
                  </div>

                  {/* Área */}
                  <div>
                    <label className="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2" htmlFor="area">
                      Área
                    </label>
                    <div className="relative">
                      <FiFilter className="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400" />
                      <select
                        id="area"
                        value={area}
                        onChange={(e) => setArea(e.target.value)}
                        className="w-full rounded-xl border-2 border-gray-200 dark:border-gray-600 bg-white dark:bg-gray-700
                                   pl-11 pr-4 py-3 text-sm text-gray-900 dark:text-white focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-transparent transition-all duration-200 appearance-none cursor-pointer"
                      >
                        {AREAS.map((a) => (
                          <option key={a} value={a}>
                            {a}
                          </option>
                        ))}
                      </select>
                      <div className="absolute right-4 top-1/2 -translate-y-1/2 pointer-events-none text-gray-400">▼</div>
                    </div>
                  </div>

                  {/* Materia */}
                  <div>
                    <label className="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2" htmlFor="materia">
                      Asignatura
                    </label>
                    <div className="relative">
                      <FiBookOpen className="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400" />
                      <select
                        id="materia"
                        value={materia}
                        onChange={(e) => setMateria(e.target.value)}
                        className="w-full rounded-xl border-2 border-gray-200 dark:border-gray-600 bg-white dark:bg-gray-700
                                   pl-11 pr-4 py-3 text-sm text-gray-900 dark:text-white focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-transparent transition-all duration-200 appearance-none cursor-pointer"
                      >
                        {MATERIAS.map((m) => (
                          <option key={m} value={m}>
                            {m}
                          </option>
                        ))}
                      </select>
                      <div className="absolute right-4 top-1/2 -translate-y-1/2 pointer-events-none text-gray-400">▼</div>
                    </div>
                  </div>
                </div>

                {/* Tags de filtros activos */}
                {(area !== "Todas" || materia !== "Todas" || q) && (
                  <motion.div
                    initial={{ opacity: 0, y: -10 }}
                    animate={{ opacity: 1, y: 0 }}
                    className="mt-4 flex flex-wrap items-center gap-2"
                  >
                    <span className="text-sm font-medium text-gray-600 dark:text-gray-400">Filtros activos:</span>
                    {q && (
                      <span className="inline-flex items-center gap-2 rounded-full bg-indigo-100 dark:bg-indigo-900/30 px-3 py-1 text-sm font-medium text-indigo-700 dark:text-indigo-300">
                        <FiSearch className="text-xs" />
                        "{q}"
                      </span>
                    )}
                    {area !== "Todas" && (
                      <span className="inline-flex items-center gap-2 rounded-full bg-blue-100 dark:bg-blue-900/30 px-3 py-1 text-sm font-medium text-blue-700 dark:text-blue-300">
                        <FiFilter className="text-xs" />
                        {area}
                      </span>
                    )}
                    {materia !== "Todas" && (
                      <span className="inline-flex items-center gap-2 rounded-full bg-purple-100 dark:bg-purple-900/30 px-3 py-1 text-sm font-medium text-purple-700 dark:text-purple-300">
                        <FiBookOpen className="text-xs" />
                        {materia}
                      </span>
                    )}
                    <button
                      onClick={clearFilters}
                      className="text-sm text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 underline"
                    >
                      Limpiar filtros
                    </button>
                  </motion.div>
                )}
              </div>
            </section>

            {/* Grid de Docentes */}
            <section className="py-16">
              <div className="mx-auto max-w-7xl px-6 lg:px-8">
                {data.length === 0 ? (
                  <motion.div
                    initial={{ opacity: 0, scale: 0.95 }}
                    animate={{ opacity: 1, scale: 1 }}
                    className="text-center py-20"
                  >
                    <div className="inline-flex items-center justify-center w-20 h-20 rounded-full bg-gray-100 dark:bg-gray-700 mb-4">
                      <FiSearch className="text-3xl text-gray-400" />
                    </div>
                    <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">
                      No se encontraron resultados
                    </h3>
                    <p className="text-gray-600 dark:text-gray-400 mb-6">
                      Intenta ajustar los filtros de búsqueda
                    </p>
                    <button
                      onClick={clearFilters}
                      className="inline-flex items-center gap-2 px-6 py-3 rounded-xl bg-indigo-600 hover:bg-indigo-700 text-white font-medium transition-colors duration-200"
                    >
                      Limpiar filtros
                    </button>
                  </motion.div>
                ) : (
                  <>
                    <motion.div initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} className="text-center mb-12">
                      <p className="text-lg text-gray-600 dark:text-gray-400">
                        Mostrando{" "}
                        <span className="font-bold text-indigo-600 dark:text-indigo-400">{data.length}</span>{" "}
                        {data.length === 1 ? "docente" : "docentes"}
                      </p>
                    </motion.div>

                    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-x-8 gap-y-28">
                      {data.map((d, i) => (
                        <DocenteCard key={d.id ?? i} d={d} onSubjectClick={handleSubjectClick} index={i} />
                      ))}
                    </div>
                  </>
                )}
              </div>
            </section>
          </>
        )}
      </div>
    </PublicLayout>
  );
}
