import React, { useMemo, useState } from "react";
import { Link } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import { FiArrowRight } from "react-icons/fi";

/* =========================
   Animations
   ========================= */
const cardVariants = {
  hidden: { opacity: 0, y: 30 },
  show: { opacity: 1, y: 0, transition: { duration: 0.6, ease: "easeOut" } },
};

const listVariants = {
  hidden: { opacity: 0 },
  show: {
    opacity: 1,
    transition: { staggerChildren: 0.08, delayChildren: 0.05 },
  },
};

/* =========================================================
   FlipCard con CTA principal
   ========================================================= */
function FlipCard({ title, desc, href, image, extras = [], ctaLabel = "Ver más" }) {
  const disabled = !href || href === "#";
  const isExternal = (to = "") => /^https?:\/\//i.test(to);
  const isAbsolutePath = (to = "") => to.startsWith("/");

  const BaseCTA = ({ to, children }) =>
    disabled ? (
      <span
        className="inline-flex items-center gap-2 rounded-xl bg-gray-300 dark:bg-gray-600 text-white px-4 py-2 text-sm font-semibold cursor-not-allowed"
        title="Próximamente"
      >
        {children}
      </span>
    ) : isExternal(to) ? (
      <a
        href={to}
        target="_blank"
        rel="noreferrer"
        className="inline-flex items-center gap-2 rounded-xl bg-primary-700 text-white px-4 py-2 text-sm font-semibold hover:bg-primary-800 focus:outline-none focus-visible:ring-2 focus-visible:ring-offset-2 focus-visible:ring-primary-400"
      >
        {children}
      </a>
    ) : (
      <Link
        href={to}
        className="inline-flex items-center gap-2 rounded-xl bg-primary-700 text-white px-4 py-2 text-sm font-semibold hover:bg-primary-800 focus:outline-none focus-visible:ring-2 focus-visible:ring-offset-2 focus-visible:ring-primary-400"
      >
        {children}
      </Link>
    );

  return (
    <motion.div variants={cardVariants} className="group relative h-[min(92vw,460px)] sm:h-[400px] rounded-2xl [perspective:1000px]">
      <div className="relative h-full w-full rounded-2xl transition-transform duration-[900ms] ease-[cubic-bezier(.2,.8,.2,1)] [transform-style:preserve-3d] group-hover:[transform:rotateY(180deg)]">
        {/* Cara frontal */}
        <div className="absolute inset-0 rounded-2xl overflow-hidden [backface-visibility:hidden]">
          <img src={image} alt={title} className="h-full w-full object-cover" loading="lazy" />
          <div className="absolute inset-0 bg-gradient-to-b from-black/10 via-black/20 to-black/40" />
          <div className="absolute bottom-0 left-0 right-0 p-4">
            <h3 className="text-white text-lg font-bold drop-shadow">{title}</h3>
          </div>
        </div>

        {/* Cara trasera */}
        <div className="absolute inset-0 rounded-2xl bg-white/85 dark:bg-ink/85 backdrop-blur border border-white/30 dark:border-ink/60 p-4 flex flex-col justify-between [transform:rotateY(180deg)] [backface-visibility:hidden]">
          <div>
            <h3 className="text-lg font-bold text-ink dark:text-white">{title}</h3>
            <p className="mt-2 text-sm text-ink/80 dark:text-white/80 line-clamp-6">{desc}</p>
          </div>

          <div className="flex flex-wrap items-center gap-2">
            <BaseCTA to={href}>
              {ctaLabel} <FiArrowRight />
            </BaseCTA>

            {/* Botones secundarios */}
            {extras.map((e) =>
              e?.to ? (
                (isExternal(e.to) || isAbsolutePath(e.to)) ? (
                  <a
                    key={e.label}
                    href={e.to}
                    target={isExternal(e.to) ? "_blank" : undefined}
                    rel={isExternal(e.to) ? "noreferrer" : undefined}
                    className="inline-flex items-center gap-2 rounded-xl border border-primary-300/60 text-primary-800 dark:text-primary-200 px-3 py-2 text-xs font-semibold hover:bg-primary-50 dark:hover:bg-white/10"
                  >
                    {e.label}
                  </a>
                ) : (
                  <Link
                    key={e.label}
                    href={e.to}
                    className="inline-flex items-center gap-2 rounded-xl border border-primary-300/60 text-primary-800 dark:text-primary-200 px-3 py-2 text-xs font-semibold hover:bg-primary-50 dark:hover:bg-white/10"
                  >
                    {e.label}
                  </Link>
                )
              ) : null
            )}
          </div>
        </div>
      </div>
    </motion.div>
  );
}

/* =========================================================
   Normalizadores desde BACKEND
   ========================================================= */

// Pregrado (ya lo tenías)
const normalizePregradoFromDB = (p) => {
  const safeRoute =
    typeof route === "function" && route().current
      ? route("oferta.pregrado.detalle", p.slug)
      : `/oferta/pregrado/${p.slug}`;

  const descParts = [p.titulo, p.modalidad, p.duracion].filter(Boolean);
  return {
    title: p.nombre,
    desc: descParts.join(" · "),
    image: p.fotoTarjeta || p.imagenHero || "/assets/images/placeholder_card.jpg",
    href: safeRoute,
    ctaLabel: "Ver programa",
    extras: [],
  };
};

// 🔹 Posgrado (igual que pregrado, usando campos que envías en home())
const normalizePosgradoFromDB = (p) => {
  const safeRoute =
    typeof route === "function" && route().current
      ? route("oferta.posgrado.detalle", p.slug)
      : `/academia/posgrado/${p.slug}`;

  const descParts = [p.tipo || p.titulo, p.modalidad, p.duracion].filter(Boolean);
  return {
    title: p.nombre,
    desc: descParts.join(" · "),
    image: p.fotoTarjeta || p.imagenHero || "/assets/images/placeholder_card.jpg",
    href: safeRoute,
    ctaLabel: "Ver programa",
    extras: [],
  };
};

/* =========================================================
   Componente principal
   ========================================================= */
export default function SectionProgramsDetriFlip({ pregrados = [], postgrados = [] }) {
    console.log("🎉", pregrados, postgrados);
  const [tab, setTab] = useState("pregrado");

  const programs = useMemo(() => {
    if (tab === "pregrado") {
      const hasFromDB = Array.isArray(pregrados) && pregrados.length > 0;
      if (hasFromDB) return pregrados.map(normalizePregradoFromDB);
      // fallback si no hay data
      return [
        {
          title: "Tecnologías de la Información",
          desc: "Ingeniería en TI · Presencial · 10 semestres",
          image: "/assets/images/carreras/tecnologias_2.png",
          href: "/oferta/pregrado/ti",
          ctaLabel: "Ver programa",
          extras: [],
        },
      ];
    }

    // 🔹 POSGRADO desde backend
    const hasPos = Array.isArray(postgrados) && postgrados.length > 0;
    if (hasPos) return postgrados.map(normalizePosgradoFromDB);

    // fallback si no hay data
    return [
      {
        title: "Maestría en Tecnologías",
        desc: "Profesional · Híbrida · 3 semestres",
        image: "/assets/images/maestria/ti.png",
        href: "/academia/posgrado/tecnologias",
        ctaLabel: "Ver programa",
        extras: [],
      },
    ];
  }, [tab, pregrados, postgrados]);

  return (
    <section className="relative overflow-hidden bg-gradient-to-b from-gray-100 via-gray-200/90 to-gray-300/80 dark:from-ink dark:to-ink/95 py-16 sm:py-20">
      <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8">
        {/* Encabezado */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true, amount: 0.3 }}
          className="text-center max-w-3xl mx-auto"
        >
          <span className="inline-flex items-center rounded-full px-3 py-1 text-xs font-semibold bg-primary-600/10 text-primary-800 dark:text-primary-300 border border-primary-600/20">
            Programas Académicos
          </span>
          <h2 className="mt-2 text-3xl sm:text-4xl font-extrabold tracking-tight text-ink dark:text-white drop-shadow-[0_2px_6px_rgba(0,0,0,0.6)]">
            Pregrado y Posgrado
          </h2>
        </motion.div>

        {/* Pestañas grandes */}
        <div
          role="tablist"
          aria-label="Programas académicos"
          className="relative mt-8 mx-auto max-w-xl rounded-2xl border border-primary-700/20 dark:border-primary-300/15 bg-white/70 dark:bg-white/5 backdrop-blur"
        >
          <div className="grid grid-cols-2 relative">
            <motion.div
              layout
              className="absolute bottom-0 h-1 bg-primary-700/90 rounded-b-2xl"
              animate={{ left: tab === "pregrado" ? "0%" : "50%", width: "50%" }}
              transition={{ type: "spring", stiffness: 300, damping: 28 }}
            />
            <button
              role="tab"
              aria-selected={tab === "pregrado"}
              onClick={() => setTab("pregrado")}
              className={`relative w-full px-6 py-4 text-center text-base sm:text-lg font-semibold rounded-t-2xl transition-colors select-none ${
                tab === "pregrado"
                  ? "text-primary-800 dark:text-primary-200"
                  : "text-ink/70 dark:text-white/70 hover:text-ink dark:hover:text-white"
              }`}
            >
              Pregrado
            </button>
            <button
              role="tab"
              aria-selected={tab === "posgrado"}
              onClick={() => setTab("posgrado")}
              className={`relative w-full px-6 py-4 text-center text-base sm:text-lg font-semibold rounded-t-2xl transition-colors select-none ${
                tab === "posgrado"
                  ? "text-primary-800 dark:text-primary-200"
                  : "text-ink/70 dark:text-white/70 hover:text-ink dark:hover:text-white"
              }`}
            >
              Posgrado
            </button>
          </div>
        </div>

        {/* Grid de tarjetas */}
        <AnimatePresence mode="wait">
          <motion.div
            key={tab}
            variants={listVariants}
            initial="hidden"
            animate="show"
            exit={{ opacity: 0, y: 10, transition: { duration: 0.18 } }}
            className="mt-10 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6"
          >
            {programs.map((p) => (
              <FlipCard key={`${tab}-${p.title}`} {...p} />
            ))}
          </motion.div>
        </AnimatePresence>
      </div>
    </section>
  );
}
