// resources/js/Components/HomeSectionProyectos.jsx
import React from "react";
import { Link } from "@inertiajs/react";
import { motion, useReducedMotion } from "framer-motion";
import { FiChevronRight } from "react-icons/fi";

/**
 * Sección Home — Proyectos de Titulación (estilo FIEC ESPOL con cinta animada)
 *
 * Props:
 * - href: string               -> URL a la página de proyectos (default: "/proyectos")
 * - titleLeft: string          -> Texto izquierda (default: "Proyectos")
 * - titleRight: string         -> Texto derecha (default: "de Graduación")
 * - subtitle?: string          -> Línea secundaria opcional
 * - bgImage: string            -> Imagen de fondo 1920×800 recomendado
 * - ctaLabel?: string          -> Texto del botón ("Ver más")
 * - className?: string         -> Clases extra
 */
export default function HomeSectionProyectos({
  href = "/vida-estudiantil/tic",
  titleLeft = "Proyectos",
  titleRight = "de Graduación",
  subtitle = "Investigación aplicada, innovación y soluciones reales desde DETRI",
  bgImage = "/assets/images/epn.png",
  ctaLabel = "Ver más",
  className = "",
}) {
  const prefersReduced = useReducedMotion();

  // Variantes de animación (respetan reduced motion)
  const ribbonVariants = prefersReduced
    ? { hidden: { opacity: 0 }, show: { opacity: 1, transition: { duration: 0.3 } } }
    : {
        hidden: { width: 0, opacity: 0 },
        show: {
          width: "100%",
          opacity: 1,
          transition: { duration: 1.1, ease: [0.17, 0.84, 0.44, 1], delay: 0.15 },
        },
      };

  const leftTitle = prefersReduced
    ? { hidden: { opacity: 0, x: 0 }, show: { opacity: 1, x: 0, transition: { duration: 0.35 } } }
    : {
        hidden: { opacity: 0, x: -80, skewX: -8 },
        show: {
          opacity: 1,
          x: 0,
          skewX: 0,
          transition: { type: "spring", stiffness: 140, damping: 16, delay: 0.25 },
        },
      };

  const rightTitle = prefersReduced
    ? { hidden: { opacity: 0, x: 0 }, show: { opacity: 1, x: 0, transition: { duration: 0.35 } } }
    : {
        hidden: { opacity: 0, x: 80, skewX: 8 },
        show: {
          opacity: 1,
          x: 0,
          skewX: 0,
          transition: { type: "spring", stiffness: 140, damping: 16, delay: 0.32 },
        },
      };

  return (
    <section
      className={`relative overflow-hidden rounded-3xl shadow-2xl ring-1 ring-white/10 ${className}`}
      aria-label="Proyectos de Titulación"
    >
      {/* Fondo */}
      <div className="absolute inset-0">
        <img
          src={bgImage}
          alt="Fondo Proyectos de Titulación"
          className="h-full w-full object-cover object-center"
          loading="lazy"
        />
        {/* Velo oscuro para contraste */}
        <div className="absolute inset-0 bg-gradient-to-b from-black/60 via-black/40 to-black/60" />
        {/* Textura sutil */}
        <div className="absolute inset-0 mix-blend-overlay pointer-events-none bg-[radial-gradient(480px_200px_at_10%_8%,rgba(255,255,255,0.08),transparent),radial-gradient(560px_260px_at_90%_92%,rgba(255,255,255,0.06),transparent)]" />
      </div>

      {/* Cinta animada (ribbon) */}
      <div className="relative mx-auto max-w-7xl px-6 py-16 md:py-24 lg:py-28">
        <motion.div
          initial="hidden"
          whileInView="show"
          viewport={{ once: true, margin: "-100px" }}
          variants={ribbonVariants}
          className="relative h-16 md:h-20 rounded-xl bg-secondary/50 backdrop-blur-md ring-1 ring-white/20 overflow-hidden"
        >
          {/* Brillos laterales de la cinta */}
          <div className="pointer-events-none absolute inset-y-0 left-0 w-1/3 bg-gradient-to-r from-white/15 to-transparent" />
          <div className="pointer-events-none absolute inset-y-0 right-0 w-1/3 bg-gradient-to-l from-white/15 to-transparent" />

          {/* Título dividido en dos mitades que entran desde lados opuestos */}
          <div className="absolute inset-0 flex items-center justify-center gap-3 md:gap-5 px-4">
            <motion.span
              initial="hidden"
              whileInView="show"
              viewport={{ once: true, margin: "-120px" }}
              variants={leftTitle}
              className="text-white font-extrabold text-2xl md:text-4xl drop-shadow-[0_2px_8px_rgba(0,0,0,0.45)]"
            >
              {titleLeft}
            </motion.span>
            <motion.span
              initial="hidden"
              whileInView="show"
              viewport={{ once: true, margin: "-120px" }}
              variants={rightTitle}
              className="text-white/95 italic font-semibold text-2xl md:text-4xl drop-shadow-[0_2px_8px_rgba(0,0,0,0.45)]"
            >
              {titleRight}
            </motion.span>
          </div>
        </motion.div>

        {/* Subtítulo */}
        {subtitle && (
          <motion.p
            initial={{ opacity: 0, y: 8 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true, margin: "-120px" }}
            transition={{ duration: 0.5, delay: 0.28 }}
            className="mt-5 md:mt-7 max-w-3xl text-base md:text-lg text-white/90"
          >
            {subtitle}
          </motion.p>
        )}

        {/* Botón CTA */}
        <motion.div
          initial={{ opacity: 0, y: 8 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true, margin: "-120px" }}
          transition={{ duration: 0.5, delay: 0.36 }}
          className="mt-8"
        >
          <Link
            href={href}
            className="inline-flex items-center gap-2 rounded-full bg-primary px-5 py-3 text-white font-semibold shadow-lg ring-1 ring-white/10 transition hover:-translate-y-0.5 hover:shadow-xl focus:outline-none focus-visible:ring-2 focus-visible:ring-white/60"
          >
            {ctaLabel}
            <FiChevronRight className="text-xl" />
          </Link>
        </motion.div>
      </div>

      {/* Halo/borde animado sutil en hover */}
      <motion.div
        initial={{ opacity: 0 }}
        whileHover={{ opacity: 1 }}
        className="pointer-events-none absolute inset-0 rounded-3xl ring-2 ring-secondary/40"
      />
    </section>
  );
}

/* =============================
 * Uso típico (Home.jsx)
 * =============================
 * <HomeSectionProyectos
 *   href="/proyectos"
 *   titleLeft="Proyectos"
 *   titleRight="de Graduación"
 *   subtitle="Investigación aplicada, innovación y soluciones reales desde DETRI"
 *   bgImage="/assets/images/hero/proyectos_detri_bg.jpg"
 *   ctaLabel="Ver más"
 *   className="mt-12"
 * />
 */
