// resources/js/Components/SectionIntroDetri.jsx
import React, { useEffect, useMemo, useRef, useState } from "react";
import { Link } from "@inertiajs/react";
import { motion, useScroll, useTransform, AnimatePresence } from "framer-motion";
import { FiArrowRight, FiChevronLeft, FiChevronRight } from "react-icons/fi";

export default function SectionIntroDetri() {
  const secRef = useRef(null);

  // Parallax sutil de decoraciones
  const { scrollYProgress } = useScroll({ target: secRef, offset: ["start end", "end start"] });
  const y1 = useTransform(scrollYProgress, [0, 1], [0, -30]);
  const y2 = useTransform(scrollYProgress, [0, 1], [0,  30]);
  const opacityRibbon = useTransform(scrollYProgress, [0, 0.6], [0.2, 0.6]);

  return (
    <section
      ref={secRef}
      className="relative overflow-hidden
                 bg-gradient-to-b from-gray-100 via-gray-200/90 to-gray-300/80
                 dark:from-ink dark:to-ink/95"
    >
      {/* Cinta/halo superior */}
      <motion.div
        style={{ opacity: opacityRibbon, y: y1 }}
        className="pointer-events-none absolute inset-x-0 -top-10 h-24
                   bg-gradient-to-b from-primary-600/15 via-secondary-600/10 to-transparent blur-2xl"
      />

      {/* Shapes decorativos */}
      <motion.div
        style={{ y: y2 }}
        className="pointer-events-none absolute -right-24 top-20 h-64 w-64 rounded-full
                   bg-gradient-to-br from-primary-500/10 to-secondary-600/10 blur-2xl"
      />
      <motion.div
        style={{ y: y1 }}
        className="pointer-events-none absolute -left-24 bottom-10 h-72 w-72 rounded-full
                   bg-gradient-to-tr from-secondary-600/10 to-primary-500/10 blur-2xl"
      />

      <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8 py-16 sm:py-20">
        {/* Encabezado */}
        <motion.div
          initial={{ opacity: 0, y: 40 }}
          whileInView={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8, ease: "easeOut" }}
          viewport={{ once: true, amount: 0.3 }}
          className="text-center max-w-3xl mx-auto"
        >
          <span
            className="inline-flex items-center rounded-full px-3 py-1 text-xs font-semibold
                       bg-primary-600/10 text-primary-800 dark:text-primary-300 border border-primary-600/20
                       drop-shadow-[0_1px_2px_rgba(0,0,0,0.6)]"
          >
            DETRI — EPN
          </span>
          <h2
            className="mt-3 text-3xl sm:text-4xl font-extrabold tracking-tight
                       text-ink dark:text-white
                       drop-shadow-[0_2px_6px_rgba(0,0,0,0.6)]"
          >
            Ingeniería que transforma el futuro del Ecuador
          </h2>
          <p
            className="mt-3 text-base sm:text-lg text-ink/80 dark:text-white/80
                       drop-shadow-[0_1px_3px_rgba(0,0,0,0.5)]"
          >
            Docencia de excelencia, investigación aplicada y vinculación con la sociedad,
            en electrónica, telecomunicaciones y tecnologías de la información.
          </p>
        </motion.div>

        {/* Layout 2 columnas */}
        <div className="mt-10 sm:mt-14 grid grid-cols-1 lg:grid-cols-2 gap-8 lg:gap-12 items-center">
          {/* Columna texto */}
          <motion.div
            initial={{ opacity: 0, x: -50 }}
            whileInView={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.7, ease: "easeOut" }}
            viewport={{ once: true, amount: 0.3 }}
            className="order-2 lg:order-1"
          >
            <ul className="space-y-4">
              {[
                {
                  t: "Programas académicos actualizados",
                  d: "Mallas con enfoque en competencias, proyectos y laboratorios."
                },
                {
                  t: "Investigación con impacto",
                  d: "Grupos y líneas estratégicas orientadas a resolver problemas reales."
                },
                {
                  t: "Infraestructura de laboratorios",
                  d: "Equipamiento para prácticas, prototipado y transferencia tecnológica."
                },
              ].map((it, idx) => (
                <motion.li
                  key={it.t}
                  initial={{ opacity: 0, x: -20 }}
                  whileInView={{ opacity: 1, x: 0 }}
                  transition={{ duration: 0.5, delay: 0.1 * idx }}
                  viewport={{ once: true, amount: 0.3 }}
                  className="flex gap-3"
                >
                  <div
                    className="mt-1 h-6 w-6 shrink-0 rounded-md
                               bg-gradient-to-br from-primary-500 to-secondary-600
                               text-white flex items-center justify-center text-sm font-bold"
                  >
                    {idx + 1}
                  </div>
                  <div>
                    <h3 className="text-base sm:text-lg font-semibold text-ink dark:text-white drop-shadow-[0_1px_3px_rgba(0,0,0,0.6)]">
                      {it.t}
                    </h3>
                    <p className="text-sm sm:text-base text-ink/80 dark:text-white/80 drop-shadow-[0_1px_2px_rgba(0,0,0,0.5)]">
                      {it.d}
                    </p>
                  </div>
                </motion.li>
              ))}
            </ul>

            {/* Botones */}
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.6, delay: 0.2 }}
              viewport={{ once: true, amount: 0.3 }}
              className="mt-6 flex flex-wrap items-center gap-3"
            >
              <Link
                href="/oferta/pregrado"
                className="inline-flex items-center gap-2 rounded-xl bg-primary-700 text-white px-5 py-2.5
                           text-sm font-semibold hover:bg-primary-800 focus:outline-none
                           focus-visible:ring-2 focus-visible:ring-offset-2 focus-visible:ring-primary-400"
              >
                Ver programas <FiArrowRight />
              </Link>
              <Link
                href="/investigacion/centros-laboratorios"
                className="inline-flex items-center gap-2 rounded-xl px-5 py-2.5 text-sm font-semibold
                           text-primary-800 dark:text-primary-300 bg-primary-600/10 border border-primary-600/20
                           hover:bg-primary-600/15"
              >
                Nuestros laboratorios
              </Link>
            </motion.div>

            {/* Métricas */}
            <motion.div
              initial={{ opacity: 0, y: 40 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, delay: 0.3 }}
              viewport={{ once: true, amount: 0.3 }}
              className="mt-8 grid grid-cols-3 gap-4"
            >
              {[
                { n: "+1.2k", l: "Estudiantes" },
                { n: "18", l: "Laboratorios" },
                { n: "50+", l: "Pub. / año" },
              ].map((m) => (
                <div
                  key={m.l}
                  className="rounded-2xl px-4 py-3 text-center
                             bg-white/50 dark:bg-ink/70
                             supports-[backdrop-filter]:bg-white/30 dark:supports-[backdrop-filter]:bg-ink/50
                             backdrop-blur border border-primary-100/40 dark:border-ink/60"
                >
                  <div className="text-xl sm:text-2xl font-extrabold text-ink dark:text-white drop-shadow-[0_1px_3px_rgba(0,0,0,0.6)]">
                    {m.n}
                  </div>
                  <div className="text-xs sm:text-sm text-ink/80 dark:text-white/80 drop-shadow-[0_1px_2px_rgba(0,0,0,0.5)]">
                    {m.l}
                  </div>
                </div>
              ))}
            </motion.div>
          </motion.div>

          {/* Columna imagen → SLIDER */}
          <motion.div
            initial={{ opacity: 0, x: 50 }}
            whileInView={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.7, ease: "easeOut" }}
            viewport={{ once: true, amount: 0.3 }}
            className="order-1 lg:order-2 relative"
          >
            <motion.div
              style={{ y: y1 }}
              className="absolute -inset-3 rounded-3xl bg-gradient-to-tr
                         from-primary-500/10 via-secondary-600/10 to-transparent blur-xl"
            />
            <div className="relative rounded-3xl overflow-hidden border
                            border-primary-100/60 dark:border-ink/60
                            bg-white/50 dark:bg-ink/60
                            supports-[backdrop-filter]:bg-white/30 dark:supports-[backdrop-filter]:bg-ink/40
                            backdrop-blur shadow-[0_10px_35px_rgba(0,0,0,0.08)]">
              <LabsSlider />
              {/* Cinta inferior */}
              <div className="absolute bottom-3 left-3 right-3">
                <div className="rounded-2xl px-4 py-2.5 text-sm sm:text-base
                                bg-gradient-to-r from-primary-700/90 to-secondary-700/90 text-white
                                shadow-lg flex items-center justify-between">
                  <span>Excelencia académica · Innovación · Impacto</span>
                  <span className="hidden sm:inline font-semibold">DETRI — EPN</span>
                </div>
              </div>
            </div>
          </motion.div>
        </div>
      </div>
    </section>
  );
}

/* =========================
   Slider de Laboratorios
   ========================= */
function LabsSlider() {
  const IMAGES_COUNT = 5; // <-- cambia este número a la cantidad que tengas
  const slides = useMemo(
    () => Array.from({ length: IMAGES_COUNT }, (_, i) => `/assets/images/laboratorios_${i + 1}.png`),
    [IMAGES_COUNT]
  );

  const [index, setIndex] = useState(0);
  const [direction, setDirection] = useState(0); // -1 izquierda, 1 derecha
  const [paused, setPaused] = useState(false);
  const containerRef = useRef(null);

  // Autoplay
  useEffect(() => {
    if (paused) return;
    const id = setInterval(() => {
      setDirection(1);
      setIndex((i) => (i + 1) % slides.length);
    }, 5000);
    return () => clearInterval(id);
  }, [paused, slides.length]);

  const prev = () => {
    setDirection(-1);
    setIndex((i) => (i - 1 + slides.length) % slides.length);
  };
  const next = () => {
    setDirection(1);
    setIndex((i) => (i + 1) % slides.length);
  };

  // Swipe táctil
  const touch = useRef({ x: 0, y: 0, t: 0 });
  const onTouchStart = (e) => {
    const t = e.touches[0];
    touch.current = { x: t.clientX, y: t.clientY, t: Date.now() };
  };
  const onTouchEnd = (e) => {
    const dx = (e.changedTouches?.[0]?.clientX ?? 0) - touch.current.x;
    const dt = Date.now() - touch.current.t;
    if (Math.abs(dx) > 40 && dt < 600) {
      dx > 0 ? prev() : next();
    }
  };

  // Variants para transición entre slides
  const variants = {
    enter: (dir) => ({ x: dir > 0 ? 60 : -60, opacity: 0, scale: 0.98 }),
    center: { x: 0, opacity: 1, scale: 1, transition: { duration: 0.45, ease: "easeOut" } },
    exit: (dir) => ({ x: dir > 0 ? -60 : 60, opacity: 0, scale: 0.98, transition: { duration: 0.35, ease: "easeIn" } }),
  };

  return (
    <div
      ref={containerRef}
      className="relative w-full h-[280px] sm:h-[360px] lg:h-[420px] select-none"
      onMouseEnter={() => setPaused(true)}
      onMouseLeave={() => setPaused(false)}
      onTouchStart={onTouchStart}
      onTouchEnd={onTouchEnd}
      role="region"
      aria-label="Galería de laboratorios DETRI"
    >
      {/* Slide actual */}
      <AnimatePresence custom={direction} mode="popLayout">
        <motion.img
          key={slides[index]}
          src={slides[index]}
          alt={`Laboratorio ${index + 1}`}
          className="absolute inset-0 w-full h-full object-cover"
          custom={direction}
          initial="enter"
          animate="center"
          exit="exit"
          variants={variants}
          draggable={false}
        />
      </AnimatePresence>

      {/* Controles */}
      <button
        aria-label="Anterior"
        onClick={prev}
        className="absolute left-2 top-1/2 -translate-y-1/2 p-2 rounded-full
                   bg-white/70 hover:bg-white shadow-md border border-white/60
                   text-ink/80 backdrop-blur"
      >
        <FiChevronLeft />
      </button>
      <button
        aria-label="Siguiente"
        onClick={next}
        className="absolute right-2 top-1/2 -translate-y-1/2 p-2 rounded-full
                   bg-white/70 hover:bg-white shadow-md border border-white/60
                   text-ink/80 backdrop-blur"
      >
        <FiChevronRight />
      </button>

      {/* Bullets */}
      <div className="absolute bottom-3 left-0 right-0 flex items-center justify-center gap-2">
        {slides.map((_, i) => (
          <button
            key={i}
            aria-label={`Ir a la imagen ${i + 1}`}
            onClick={() => {
              setDirection(i > index ? 1 : -1);
              setIndex(i);
            }}
            className={`h-2.5 w-2.5 rounded-full transition
                        ${i === index ? "bg-primary-600" : "bg-white/60 hover:bg-white"}`}
          />
        ))}
      </div>
    </div>
  );
}
