import React, { useState } from "react";
import { useForm, router, Head } from "@inertiajs/react";
import DashboardLayout from "@/Layouts/AuthenticatedLayout";
import {
  FiSave, FiCheckCircle, FiUpload, FiBookOpen, FiLink, FiImage, FiTag, FiPlus, FiTrash2,
} from "react-icons/fi";

const LINK_OPTIONS = [
  { value: "scholar",  label: "Google Scholar" },
  { value: "orcid",    label: "ORCID" },
  { value: "scopus",   label: "Scopus" },
  { value: "research", label: "ResearchGate" },
  { value: "linkedin", label: "LinkedIn" },
  { value: "otro",     label: "Otro…" },
];

// Puedes reemplazar por catálogo que envíes desde el backend si lo deseas
const DEFAULT_AREA_OPTIONS = [
  "Telecomunicaciones",
  "Redes de Información",
  "Electrónica",
  "Ciencias de la Computación",
  "Sistemas",
];

export default function DocentePerfilIndex({ auth, perfil = {}, flash }) {
  const [fotoPreview, setFotoPreview] = useState(perfil?.foto || null);

  // inputs para agregar items
  const [asigInput, setAsigInput] = useState("");
  const [linkType, setLinkType] = useState(LINK_OPTIONS[0].value);
  const [linkKey, setLinkKey] = useState("");
  const [linkUrl, setLinkUrl] = useState("");

  const areaOptions = DEFAULT_AREA_OPTIONS;

  const { data, setData, processing, errors, post, transform } = useForm({
    _method: "patch",
    nombre: perfil?.nombre ?? "",
    email:  perfil?.email ?? "",
    grado:  perfil?.grado ?? "",
    area:   perfil?.area ?? "",
    foto:   perfil?.foto ?? "",
    foto_file: null,
    // arrays
    asignaturas: Array.isArray(perfil?.asignaturas) ? perfil.asignaturas : [],
    // enlaces como objeto asociativo { key: url }
    enlaces: (perfil?.enlaces && typeof perfil.enlaces === "object") ? perfil.enlaces : {},
  });

  // Si necesitas transformar algo antes de enviar, lo dejas listo aquí
  transform((payload) => ({
    ...payload,
    // nada extra; ya mandamos arrays/objeto reales
  }));

  const onFotoChange = (e) => {
    const file = e.target.files?.[0];
    if (!file) return;
    setData("foto_file", file);
    const reader = new FileReader();
    reader.onload = (ev) => setFotoPreview(ev.target.result);
    reader.readAsDataURL(file);
  };

  const addAsignatura = () => {
    const v = (asigInput || "").trim();
    if (!v) return;
    if ((data.asignaturas || []).includes(v)) {
      setAsigInput("");
      return;
    }
    setData("asignaturas", [...(data.asignaturas || []), v]);
    setAsigInput("");
  };

  const removeAsignatura = (idx) => {
    const next = [...(data.asignaturas || [])];
    next.splice(idx, 1);
    setData("asignaturas", next);
  };

  const addEnlace = () => {
    const key = (linkType === "otro" ? linkKey : linkType).trim();
    const url = (linkUrl || "").trim();
    if (!key || !url) return;
    setData("enlaces", {
      ...(data.enlaces || {}),
      [key]: url,
    });
    setLinkUrl("");
    setLinkKey("");
    setLinkType(LINK_OPTIONS[0].value);
  };

  const removeEnlace = (k) => {
    const next = { ...(data.enlaces || {}) };
    delete next[k];
    setData("enlaces", next);
  };

  const submit = (e) => {
    e.preventDefault();
    router.post(route("docente.perfil.update"), {
      ...data,
      preserveScroll: true,
      forceFormData: true, // necesario por la imagen
    });
  };

  return (
    <DashboardLayout auth={auth} titulo="Mi Perfil Docente">
      <Head title="Mi Perfil Docente" />

      <div className="max-w-4xl mx-auto">
        <form onSubmit={submit} className="p-6 space-y-4 rounded-2xl border border-primary-200/60 bg-white/90 shadow-sm dark:border-primary-800/60 dark:bg-primary-950/80">
          <h2 className="mb-2 text-lg font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-200 dark:to-secondary-300">
            Editar mi perfil
          </h2>

          {/* Básicos */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-2">
            <div>
              <label className="block text-sm font-semibold">Nombre</label>
              <input
                type="text"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.nombre}
                onChange={(e) => setData("nombre", e.target.value)}
              />
              {errors.nombre && <p className="text-sm text-red-500">{errors.nombre}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Correo</label>
              <input
                type="email"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.email}
                onChange={(e) => setData("email", e.target.value)}
              />
              {errors.email && <p className="text-sm text-red-500">{errors.email}</p>}
            </div>
          </div>

          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div>
              <label className="block text-sm font-semibold">Grado</label>
              <input
                type="text"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.grado}
                onChange={(e) => setData("grado", e.target.value)}
              />
              {errors.grado && <p className="text-sm text-red-500">{errors.grado}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Área</label>
              <input
                list="areaOptions"
                type="text"
                placeholder="Ej: Telecomunicaciones, Electrónica…"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.area}
                onChange={(e) => setData("area", e.target.value)}
              />
              <datalist id="areaOptions">
                {areaOptions.map((a) => (
                  <option key={a} value={a} />
                ))}
              </datalist>
              {errors.area && <p className="text-sm text-red-500">{errors.area}</p>}
            </div>

            {/* Foto: upload + preview */}
            <div>
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiImage /> Foto del docente
              </label>
              <input
                type="file"
                accept="image/*"
                onChange={onFotoChange}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm file:mr-3 file:rounded-lg file:border-0 file:bg-primary-100 file:px-3 file:py-1.5 dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              {(fotoPreview || data.foto) && (
                <div className="mt-2 flex items-center gap-3">
                  <img
                    src={fotoPreview || data.foto}
                    alt="preview"
                    className="h-14 w-14 rounded-xl object-cover border border-primary-200/60 dark:border-primary-800/60"
                  />
                  {fotoPreview && (
                    <span className="text-xs text-primary-700/80 dark:text-primary-200/80">
                      Vista previa (sin guardar)
                    </span>
                  )}
                </div>
              )}
              {errors.foto_file && <p className="text-sm text-red-500">{errors.foto_file}</p>}
            </div>
          </div>

          {/* Asignaturas: input + botón Agregar + chips */}
          <div>
            <label className="block text-sm font-semibold flex items-center gap-2">
              <FiBookOpen /> Asignaturas
            </label>
            <div className="mt-1 flex gap-2">
              <input
                type="text"
                value={asigInput}
                onChange={(e) => setAsigInput(e.target.value)}
                onKeyDown={(e) => e.key === "Enter" && (e.preventDefault(), addAsignatura())}
                placeholder="Ej: Redes I"
                className="flex-1 rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              <button
                type="button"
                onClick={addAsignatura}
                className="inline-flex items-center gap-2 rounded-xl bg-primary-600 px-3 py-2 text-sm font-semibold text-white hover:bg-primary-700"
              >
                <FiPlus /> Agregar
              </button>
            </div>

            <div className="mt-2 flex flex-wrap gap-2">
              {data.asignaturas?.map((asig, idx) => (
                <span
                  key={`${asig}-${idx}`}
                  className="group inline-flex items-center gap-2 rounded-xl border border-primary-200/60 bg-primary-50/70 px-3 py-1 text-xs font-medium dark:border-primary-800/60 dark:bg-primary-900/40"
                >
                  <FiTag />
                  {asig}
                  <button
                    type="button"
                    onClick={() => removeAsignatura(idx)}
                    className="rounded-md px-1 text-xs opacity-70 hover:bg-primary-100 dark:hover:bg-primary-900/60"
                    title="Quitar"
                  >
                    ×
                  </button>
                </span>
              ))}
              {(!data.asignaturas || data.asignaturas.length === 0) && (
                <span className="text-xs text-gray-400 italic">Aún sin asignaturas</span>
              )}
            </div>
            {errors.asignaturas && <p className="text-sm text-red-500">{errors.asignaturas}</p>}
          </div>

          {/* Enlaces: select + (otro) + url + botón Agregar + lista */}
          <div>
            <label className="block text-sm font-semibold flex items-center gap-2">
              <FiLink /> Enlaces
            </label>

            <div className="mt-2 grid grid-cols-1 gap-2 sm:grid-cols-3">
              <select
                value={linkType}
                onChange={(e) => setLinkType(e.target.value)}
                className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              >
                {LINK_OPTIONS.map((o) => (
                  <option key={o.value} value={o.value}>
                    {o.label}
                  </option>
                ))}
              </select>

              {linkType === "otro" && (
                <input
                  type="text"
                  placeholder="Nombre del enlace (ej: Portfolio)"
                  value={linkKey}
                  onChange={(e) => setLinkKey(e.target.value)}
                  className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                />
              )}

              <input
                type="text"
                placeholder="https://..."
                value={linkUrl}
                onChange={(e) => setLinkUrl(e.target.value)}
                onKeyDown={(e) => e.key === "Enter" && (e.preventDefault(), addEnlace())}
                className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
            </div>

            <div className="mt-2">
              <button
                type="button"
                onClick={addEnlace}
                className="inline-flex items-center gap-2 rounded-xl bg-primary-600 px-3 py-2 text-sm font-semibold text-white hover:bg-primary-700"
              >
                <FiPlus /> Agregar enlace
              </button>
            </div>

            {/* Lista de enlaces actuales */}
            <div className="mt-3 space-y-2">
              {Object.entries(data.enlaces || {}).length === 0 && (
                <div className="text-xs text-gray-400 italic">Sin enlaces</div>
              )}
              {Object.entries(data.enlaces || {}).map(([k, url]) => (
                <div
                  key={k}
                  className="flex items-center justify-between rounded-xl border border-primary-200/60 bg-primary-50/50 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-900/40"
                >
                  <div className="flex items-center gap-2">
                    <span className="min-w-[100px] font-semibold capitalize">{k}</span>
                    <a
                      href={url}
                      target="_blank"
                      rel="noreferrer"
                      className="truncate text-primary-700 underline dark:text-primary-300"
                      title={url}
                    >
                      {url}
                    </a>
                  </div>
                  <button
                    type="button"
                    onClick={() => removeEnlace(k)}
                    className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                    title="Eliminar enlace"
                  >
                    <FiTrash2 />
                  </button>
                </div>
              ))}
            </div>

            {errors.enlaces && <p className="text-sm text-red-500">{errors.enlaces}</p>}
          </div>

          {/* Footer */}
          <div className="mt-4 flex justify-end gap-2">
            <button
              type="submit"
              disabled={processing}
              className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700 disabled:opacity-60"
            >
              <FiSave className="mr-1 inline-block" />
              {processing ? "Guardando..." : "Guardar cambios"}
            </button>
          </div>

          {flash?.success && (
            <div className="mt-1 flex items-center gap-2 text-sm text-green-700">
              <FiCheckCircle /> {flash.success}
            </div>
          )}
        </form>
      </div>
    </DashboardLayout>
  );
}
