import React, { useMemo } from "react";
import { motion, AnimatePresence } from "framer-motion";
import {
  FiBookOpen, FiTrendingUp, FiUser, FiMail, FiBriefcase, FiAlertCircle
} from "react-icons/fi";
import AuthenticatedLayout from "@/Layouts/AuthenticatedLayout";
import {
  LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer,
  BarChart, Bar, PieChart, Pie, Cell, Legend
} from "recharts";

const cls = (...xs) => xs.filter(Boolean).join(" ");
const fadeIn = { hidden: { opacity: 0, y: 12 }, show: { opacity: 1, y: 0, transition: { duration: .5 } } };
const spring = { type: "spring", stiffness: 120, damping: 18 };

const Card = ({ title, subtitle, right, children, className }) => (
  <motion.div layout transition={spring} className={cls("rounded-2xl bg-white shadow-sm border border-gray-100", className)}>
    <div className="px-4 pt-4 pb-2 flex flex-wrap items-center justify-between gap-3">
      <div>
        <h3 className="text-sm font-semibold text-gray-800">{title}</h3>
        {subtitle && <p className="text-xs text-gray-500">{subtitle}</p>}
      </div>
      {right}
    </div>
    <div className="p-4 pt-0">{children}</div>
  </motion.div>
);

const KPI = ({ icon: Icon, label, value }) => (
  <motion.div variants={fadeIn} className="rounded-2xl p-4 bg-white shadow-sm border border-gray-100">
    <div className="flex items-center gap-3">
      <span className="inline-flex p-2 rounded-xl bg-primary-50 text-primary-700">
        <Icon size={20} />
      </span>
      <div>
        <div className="text-xs uppercase tracking-wide text-gray-500">{label}</div>
        <div className="text-2xl font-bold text-gray-900">{value ?? 0}</div>
      </div>
    </div>
  </motion.div>
);

/* Palette para charts */
const COLORS = ["#1e3a8a","#0ea5e9","#10b981","#f59e0b","#ef4444","#8b5cf6","#14b8a6","#3b82f6"];

function toLineData(series) {
  if (!series || !series.labels || !series.data) return [];
  return series.labels.map((m, i) => ({ name: m, value: series.data[i] ?? 0 }));
}
function normalizeCountRows(rows, key = "name") {
  if (!rows || !rows.length) return [];
  const entryKey = Object.keys(rows[0] || {}).find(k => k !== "total") || key;
  return rows.map(r => ({ name: r[entryKey] ?? "N/D", value: Number(r.total) || 0 }));
}

function EmptyState({ text = "Sin información", icon }) {
  return (
    <div className="h-full w-full flex items-center justify-center">
      <div className="text-center text-gray-400">
        <div className="mx-auto mb-2 inline-flex p-2 rounded-xl bg-gray-50">{icon || <FiAlertCircle />}</div>
        <p className="text-sm">{text}</p>
      </div>
    </div>
  );
}

export default function Index({ auth, range, profile, totals = {}, series = {}, tops = {}, latest = {} }) {
  const serieProyectos     = toLineData(series.proyectos);
  const seriePublicaciones = toLineData(series.publicaciones);
  const serieTic           = toLineData(series.tic_projects);

  const barProyectosLinea  = normalizeCountRows(tops.proyectos_por_linea, "linea_investigacion");
  const ultimasPubs        = latest?.publicaciones || [];

  const kpis = useMemo(() => ([
    { icon: FiTrendingUp, label: "Proyectos",     value: totals.proyectos },
    { icon: FiBookOpen,   label: "Publicaciones", value: totals.publicaciones },
    { icon: FiBriefcase,  label: "TIC Projects",  value: totals.tic_projects },
  ]), [totals]);

  return (
    <AuthenticatedLayout auth={auth} titulo="Dashboard Docente">
      <div className="px-4 sm:px-6 lg:px-8 py-6 space-y-6">
        {/* Perfil docente */}
        <Card
          title="Resumen del docente"
          subtitle={range?.from && range?.to ? `Periodo: ${range.from} — ${range.to}` : null}
        >
          {profile ? (
            <div className="flex flex-col sm:flex-row sm:items-center gap-4">
              <div className="flex items-center gap-3 min-w-0">
                <div className="h-12 w-12 rounded-xl bg-primary-50 text-primary-700 flex items-center justify-center">
                  <FiUser />
                </div>
                <div className="min-w-0">
                  <div className="font-semibold text-gray-900 truncate">
                    {profile.nombre || "Docente"}
                  </div>
                  <div className="text-xs text-gray-500 truncate">
                    {profile.area || "Área no registrada"}
                  </div>
                  <div className="text-xs text-gray-500 flex items-center gap-1">
                    <FiMail /> {profile.correo || "—"}
                  </div>
                </div>
              </div>
            </div>
          ) : (
            <EmptyState text="No se pudo identificar al docente." />
          )}
        </Card>

        {/* KPIs */}
        <AnimatePresence>
          <motion.div
            initial="hidden"
            animate="show"
            variants={{ show: { transition: { staggerChildren: 0.05 } } }}
            className="grid grid-cols-1 xs:grid-cols-2 lg:grid-cols-3 gap-4"
          >
            {kpis.map((k, i) => <KPI key={i} {...k} />)}
          </motion.div>
        </AnimatePresence>

        {/* Series */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
          <Card title="Proyectos por mes (últimos 12 meses)">
            <div className="h-64">
              {serieProyectos.length ? (
                <ResponsiveContainer width="100%" height="100%">
                  <LineChart data={serieProyectos} margin={{ left: 8, right: 8, top: 10, bottom: 0 }}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" fontSize={12} />
                    <YAxis allowDecimals={false} fontSize={12} />
                    <Tooltip />
                    <Line type="monotone" dataKey="value" stroke="#1e3a8a" strokeWidth={2} dot={false} />
                  </LineChart>
                </ResponsiveContainer>
              ) : <EmptyState text="Sin datos de proyectos en el período." />}
            </div>
          </Card>

          <Card title="Publicaciones por mes (últimos 12 meses)">
            <div className="h-64">
              {seriePublicaciones.length ? (
                <ResponsiveContainer width="100%" height="100%">
                  <LineChart data={seriePublicaciones} margin={{ left: 8, right: 8, top: 10, bottom: 0 }}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" fontSize={12} />
                    <YAxis allowDecimals={false} fontSize={12} />
                    <Tooltip />
                    <Line type="monotone" dataKey="value" stroke="#10b981" strokeWidth={2} dot={false} />
                  </LineChart>
                </ResponsiveContainer>
              ) : <EmptyState text="Sin datos de publicaciones en el período." />}
            </div>
          </Card>
        </div>

        {/* Barras + Lista últimas publicaciones */}
        <div className="grid grid-cols-1 xl:grid-cols-3 gap-4">
          <Card title="Proyectos por línea de investigación" subtitle="Top 8" className="xl:col-span-2">
            <div className="h-64">
              {barProyectosLinea.length ? (
                <ResponsiveContainer width="100%" height="100%">
                  <BarChart data={barProyectosLinea} margin={{ left: 8, right: 8, top: 10, bottom: 0 }}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" fontSize={12} />
                    <YAxis allowDecimals={false} fontSize={12} />
                    <Tooltip />
                    <Bar dataKey="value" fill="#0ea5e9" radius={[6,6,0,0]} />
                  </BarChart>
                </ResponsiveContainer>
              ) : <EmptyState text="Sin líneas de investigación asociadas." />}
            </div>
          </Card>

          <Card title="Últimas publicaciones">
            <div className="space-y-3">
              {ultimasPubs.length ? (
                <ul className="divide-y divide-gray-100">
                  {ultimasPubs.map((p) => (
                    <li key={p.id} className="py-3">
                      <div className="font-medium text-gray-800">{p.titulo || "Publicación"}</div>
                      <div className="text-xs text-gray-500">
                        {(p.revista || p.editorial || "—")}
                        {p.anio ? <> · {p.anio}</> : null}
                      </div>
                      {p.autores ? (
                        <div className="text-xs text-gray-500 truncate">Autores: {p.autores}</div>
                      ) : null}
                    </li>
                  ))}
                </ul>
              ) : (
                <EmptyState text="Aún no hay publicaciones recientes." />
              )}
            </div>
          </Card>
        </div>

        {/* Serie TIC Projects opcional */}
        <Card title="TIC Projects por mes (últimos 12 meses)">
          <div className="h-64">
            {serieTic.length ? (
              <ResponsiveContainer width="100%" height="100%">
                <LineChart data={serieTic} margin={{ left: 8, right: 8, top: 10, bottom: 0 }}>
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis dataKey="name" fontSize={12} />
                  <YAxis allowDecimals={false} fontSize={12} />
                  <Tooltip />
                  <Line type="monotone" dataKey="value" stroke="#f59e0b" strokeWidth={2} dot={false} />
                </LineChart>
              </ResponsiveContainer>
            ) : <EmptyState text="Sin datos de TIC Projects en el período." />}
          </div>
        </Card>
      </div>
    </AuthenticatedLayout>
  );
}
