import React, { useEffect, useState } from "react";
import { useForm, router } from "@inertiajs/react";
import DashboardLayout from "@/Layouts/AuthenticatedLayout";
import Modal from "@/Components/Modal";
import { FiEdit, FiPlus, FiTrash2, FiRefreshCw, FiPower, FiSearch, FiEye, FiEyeOff } from "react-icons/fi";

export default function UsuariosIndex({ auth, users, roles, filters, flash }) {
  // UI state
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState(null);
  const [showPwd, setShowPwd] = useState(false);

  // Filtros locales (búsqueda/rol/estado/papelera)
  const [q, setQ] = useState(filters?.q ?? "");
  const [role, setRole] = useState(filters?.role ?? "");
  const [status, setStatus] = useState(filters?.status ?? "");
  const [trash, setTrash] = useState(filters?.trash ?? "none");

  // Formulario (crear/editar)
  const { data, setData, reset, post, patch, processing, errors } = useForm({
    name: "",
    email: "",
    role: "Docente",
    is_active: true,
    password: "", // opcional. En crear: si vacío, se genera. En editar: si vacío, no se cambia.
  });

  useEffect(() => {
    if (flash?.success) console.log("✅", flash.success);
    if (flash?.error) console.error("❌", flash.error);
  }, [flash]);

  // Abrir modal Crear
  const openCreate = () => {
    reset();
    setEditing(null);
    setData("role", "Docente");
    setData("is_active", true);
    setShowPwd(false);
    setShowModal(true);
  };

  // Abrir modal Editar
  const openEdit = (user) => {
    setEditing(user);
    setData({
      name: user.name,
      email: user.email,
      role: user.role,
      is_active: user.is_active,
      password: "", // vacío => no cambia
    });
    setShowPwd(false);
    setShowModal(true);
  };

  // Submit modal
  const submit = (e) => {
    e.preventDefault();
    if (editing) {
      patch(route("admin.usuarios.update", editing.id), {
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    } else {
      post(route("admin.usuarios.store"), {
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    }
  };

  // Acciones rápidas
  const toggleStatus = (id) => router.patch(route("admin.usuarios.toggle", id), {}, { preserveScroll: true });
  const destroy = (id) => {
    if (confirm("¿Enviar este usuario a la papelera?")) {
      router.delete(route("admin.usuarios.destroy", id), { preserveScroll: true });
    }
  };
  const restore = (id) => router.patch(route("admin.usuarios.restore", id), {}, { preserveScroll: true });
  const forceDelete = (id) => {
    if (confirm("⚠️ Esto elimina definitivamente. ¿Continuar?")) {
      router.delete(route("admin.usuarios.force", id), { preserveScroll: true });
    }
  };

  // Buscar / filtrar
  const applyFilters = () => {
    router.get(
      route("admin.usuarios.index"),
      { q, role, status, trash, page: 1 },
      { preserveScroll: true, replace: true }
    );
  };

  // Paginación
  const goto = (url) => {
    if (!url) return;
    router.visit(url, { preserveScroll: true, replace: true });
  };

  return (
    <DashboardLayout auth={auth} titulo="Gestión de Usuarios">
      {/* Barra superior: filtros + botón Nuevo */}
      <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
        {/* Filtros */}
        <div className="flex flex-col gap-2 sm:flex-row sm:items-center">
          <div className="flex items-center rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 dark:border-primary-800/60 dark:bg-primary-950/80">
            <FiSearch className="opacity-70" />
            <input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              onKeyDown={(e) => e.key === "Enter" && applyFilters()}
              placeholder="Buscar por nombre o correo…"
              className="ml-2 w-64 bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            />
          </div>

          <select
            value={role}
            onChange={(e) => setRole(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Rol: Todos</option>
            {roles.map((r) => (
              <option key={r} value={r}>
                {r}
              </option>
            ))}
          </select>

          <select
            value={status}
            onChange={(e) => setStatus(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Estado: Todos</option>
            <option value="active">Activos</option>
            <option value="inactive">Inactivos</option>
          </select>

          <select
            value={trash}
            onChange={(e) => setTrash(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="none">Sin papelera</option>
            <option value="with">Incluir papelera</option>
            <option value="only">Solo papelera</option>
          </select>

          <button
            onClick={applyFilters}
            className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700"
          >
            Aplicar
          </button>
        </div>

        {/* Botón NUEVO */}
        {trash !== "only" && (
          <button
            onClick={openCreate}
            className="flex items-center gap-2 rounded-xl bg-gradient-to-r from-primary-600 to-secondary-600 px-4 py-2 text-sm font-semibold text-white shadow hover:opacity-90"
          >
            <FiPlus className="text-lg" /> Nuevo
          </button>
        )}
      </div>

      {/* Tabla */}
      <div className="overflow-x-auto rounded-2xl border border-primary-200/60 bg-white/90 shadow-sm backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/80">
        <table className="min-w-full divide-y divide-primary-200 dark:divide-primary-800">
          <thead className="bg-primary-50 dark:bg-primary-900/40">
            <tr className="text-sm text-primary-700 dark:text-primary-200">
              <th className="px-4 py-2 text-left">Nombre</th>
              <th className="px-4 py-2 text-left">Correo</th>
              <th className="px-4 py-2 text-left">Rol</th>
              <th className="px-4 py-2 text-center">Estado</th>
              <th className="px-4 py-2 text-center">Acciones</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-primary-100 dark:divide-primary-800">
            {users.data.map((u) => (
              <tr key={u.id} className="text-sm hover:bg-primary-50/60 dark:hover:bg-primary-900/30">
                <td className="px-4 py-2 font-medium">{u.name}</td>
                <td className="px-4 py-2">{u.email}</td>
                <td className="px-4 py-2">{u.role}</td>
                <td className="px-4 py-2 text-center">
                  {u.deleted_at ? (
                    <span className="text-xs italic text-gray-400">Eliminado</span>
                  ) : (
                    <button
                      onClick={() => toggleStatus(u.id)}
                      className={`inline-flex items-center gap-1.5 rounded-lg px-3 py-1 text-xs font-semibold ${
                        u.is_active
                          ? "bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-200"
                          : "bg-red-100 text-red-700 dark:bg-red-900/30 dark:text-red-200"
                      }`}
                    >
                      <FiPower />
                      {u.is_active ? "Activo" : "Inactivo"}
                    </button>
                  )}
                </td>
                <td className="px-4 py-2">
                  {!u.deleted_at ? (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => openEdit(u)}
                        className="rounded-md p-1.5 text-primary-600 hover:bg-primary-100 dark:text-primary-300 dark:hover:bg-primary-900/40"
                        title="Editar"
                      >
                        <FiEdit />
                      </button>
                      <button
                        onClick={() => destroy(u.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar (papelera)"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  ) : (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => restore(u.id)}
                        className="rounded-md p-1.5 text-green-600 hover:bg-green-100 dark:text-green-300 dark:hover:bg-green-900/40"
                        title="Restaurar"
                      >
                        <FiRefreshCw />
                      </button>
                      <button
                        onClick={() => forceDelete(u.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar definitivo"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  )}
                </td>
              </tr>
            ))}
            {users.data.length === 0 && (
              <tr>
                <td colSpan="5" className="px-4 py-6 text-center text-gray-400">
                  No hay usuarios para mostrar.
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Paginación */}
      <div className="mt-4 flex items-center justify-between text-sm">
        <div className="text-primary-700/80 dark:text-primary-200/80">
          Mostrando {users.from ?? 0}–{users.to ?? 0} de {users.total ?? 0}
        </div>
        <div className="flex gap-2">
          <button
            disabled={!users.prev_page_url}
            onClick={() => goto(users.prev_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Anterior
          </button>
          <button
            disabled={!users.next_page_url}
            onClick={() => goto(users.next_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Siguiente
          </button>
        </div>
      </div>

      {/* Modal (crear/editar) */}
      <Modal show={showModal} onClose={() => setShowModal(false)} maxWidth="lg">
        <form onSubmit={submit} className="p-6 space-y-4">
          <h2 className="mb-2 text-lg font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-200 dark:to-secondary-300">
            {editing ? "Editar usuario" : "Nuevo usuario"}
          </h2>

          <div>
            <label className="block text-sm font-semibold text-primary-800 dark:text-primary-100">Nombre</label>
            <input
              type="text"
              className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
              value={data.name}
              onChange={(e) => setData("name", e.target.value)}
            />
            {errors.name && <p className="text-sm text-red-500">{errors.name}</p>}
          </div>

          <div>
            <label className="block text-sm font-semibold text-primary-800 dark:text-primary-100">Correo</label>
            <input
              type="email"
              className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
              value={data.email}
              onChange={(e) => setData("email", e.target.value)}
            />
            {errors.email && <p className="text-sm text-red-500">{errors.email}</p>}
          </div>

          <div className="grid grid-cols-1 gap-3 sm:grid-cols-2">
            <div>
              <label className="block text-sm font-semibold text-primary-800 dark:text-primary-100">Rol</label>
              <select
                value={data.role}
                onChange={(e) => setData("role", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
              >
                {roles.map((r) => (
                  <option key={r} value={r}>
                    {r}
                  </option>
                ))}
              </select>
              {errors.role && <p className="text-sm text-red-500">{errors.role}</p>}
            </div>

            <div className="flex items-center gap-2 pt-6">
              <input
                id="is_active"
                type="checkbox"
                checked={data.is_active}
                onChange={(e) => setData("is_active", e.target.checked)}
              />
              <label htmlFor="is_active" className="text-sm">Usuario activo</label>
            </div>
          </div>

          {/* Contraseña (crear y editar) */}
          <div>
            <label className="block text-sm font-semibold text-primary-800 dark:text-primary-100">
              Contraseña {editing ? <span className="text-xs text-primary-500">(opcional — deja en blanco para no cambiar)</span> : <span className="text-xs text-primary-500">(opcional — se generará si la dejas vacía)</span>}
            </label>
            <div className="relative">
              <input
                type={showPwd ? "text" : "password"}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 pr-10 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
                value={data.password}
                onChange={(e) => setData("password", e.target.value)}
                placeholder={editing ? "Deja vacío para mantener la actual" : "Deja vacío para generar una aleatoria"}
              />
              <button
                type="button"
                onClick={() => setShowPwd((v) => !v)}
                className="absolute right-3 top-1/2 -translate-y-1/2 p-1 rounded hover:bg-primary-100 dark:hover:bg-primary-900/40"
                title={showPwd ? "Ocultar" : "Mostrar"}
              >
                {showPwd ? <FiEyeOff /> : <FiEye />}
              </button>
            </div>
            {errors.password && <p className="text-sm text-red-500">{errors.password}</p>}
          </div>

          <div className="mt-4 flex justify-end gap-2">
            <button
              type="button"
              onClick={() => setShowModal(false)}
              className="rounded-xl border border-primary-200/60 px-4 py-2 text-sm dark:border-primary-800/60"
            >
              Cancelar
            </button>
            <button
              type="submit"
              disabled={processing}
              className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700 disabled:opacity-60"
            >
              {processing ? "Guardando..." : editing ? "Actualizar" : "Crear"}
            </button>
          </div>
        </form>
      </Modal>
    </DashboardLayout>
  );
}
