import React, { useEffect, useMemo, useState } from "react";
import { useForm, router } from "@inertiajs/react";
import DashboardLayout from "@/Layouts/AuthenticatedLayout";
import Modal from "@/Components/Modal";
import {
  FiEdit, FiPlus, FiTrash2, FiRefreshCw, FiSearch, FiX,
  FiCheckCircle, FiMinusCircle, FiHash, FiBookOpen, FiLayers,
} from "react-icons/fi";

/**
 * props esperadas desde AdminTicProjectController@index:
 * {
 *   auth,
 *   projects, // paginator: data: [{ id, nombreProyecto, carrera, profesorCargo, resumen, semestre, lineaInvestigacion, areasTrabajo[], asignaturasRelacionadas[], estado, deleted_at, created_at }]
 *   filters,  // { q, carrera, profesor, semestre, linea, asignatura, estado, trash, perPage, sort, dir }
 *   catalogs: { carreras, profesores, semestres, lineas, asignaturas, estados },
 *   flash
 * }
 */

// Helpers CSV <-> array (únicos, trim, sin vacíos)
const toUniqueArray = (text) =>
  (text || "")
    .split(",")
    .map((s) => s.trim())
    .filter(Boolean)
    .filter((v, i, a) => a.indexOf(v) === i);

const toCSV = (arr) => (Array.isArray(arr) ? arr.join(", ") : "");


const csvJoin = (arr) => (Array.isArray(arr) ? arr.join(", ") : "");
const csvSplit = (text) =>
  (text || "")
    .split(",")
    .map((s) => s.trim())
    .filter(Boolean);

export default function AdminTicProjectsIndex({ auth, projects, filters, catalogs, flash }) {
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState(null);

  // Filtros locales (mismo look & feel que publicaciones)
  const [q, setQ] = useState(filters?.q ?? "");
  const [carrera, setCarrera] = useState(filters?.carrera ?? "");
  const [profesor, setProfesor] = useState(filters?.profesor ?? "");
  const [semestre, setSemestre] = useState(filters?.semestre ?? "");
  const [linea, setLinea] = useState(filters?.linea ?? "");
  const [asignatura, setAsignatura] = useState(filters?.asignatura ?? "");
  const [estado, setEstado] = useState(filters?.estado ?? "");
  const [trash, setTrash] = useState(filters?.trash ?? "none");
  const [sort, setSort] = useState(filters?.sort ?? "created_at");
  const [dir, setDir] = useState(filters?.dir ?? "desc");

  const perPage = filters?.perPage ?? 12;

  const {
    data, setData, reset, post, patch, processing, errors, clearErrors,
  } = useForm({
    nombre_proyecto: "",
    carrera: "",
    profesor_cargo: "",
    resumen: "",
    semestre: "",
    linea_investigacion: "",
    areas_trabajo: [],
    asignaturas_relacionadas: [],
    estado: catalogs?.estados?.[0] ?? "borrador",
  });

  useEffect(() => {
    if (flash?.success) console.log("✅", flash.success);
    if (flash?.error) console.error("❌", flash.error);
  }, [flash]);

  const openCreate = () => {
    reset();
    clearErrors();
    setEditing(null);
    setData({
      nombre_proyecto: "",
      carrera: "",
      profesor_cargo: "",
      resumen: "",
      semestre: "",
      linea_investigacion: "",
      areas_trabajo: [],
      asignaturas_relacionadas: [],
      estado: catalogs?.estados?.[0] ?? "borrador",
    });
    setShowModal(true);
  };

  const openEdit = (p) => {
    clearErrors();
    setEditing(p);
    setData({
      nombre_proyecto: p.nombreProyecto ?? "",
      carrera: p.carrera ?? "",
      profesor_cargo: p.profesorCargo ?? "",
      resumen: p.resumen ?? "",
      semestre: p.semestre ?? "",
      linea_investigacion: p.lineaInvestigacion ?? "",
      areas_trabajo: Array.isArray(p.areasTrabajo) ? p.areasTrabajo : [],
      asignaturas_relacionadas: Array.isArray(p.asignaturasRelacionadas) ? p.asignaturasRelacionadas : [],
      estado: p.estado ?? (catalogs?.estados?.[0] ?? "borrador"),
    });
    setShowModal(true);
  };

  const submit = (e) => {
    e.preventDefault();
    const payload = { ...data };

    // asegura arrays desde inputs CSV locales
    if (typeof payload.areas_trabajo === "string") {
      payload.areas_trabajo = csvSplit(payload.areas_trabajo);
    }
    if (typeof payload.asignaturas_relacionadas === "string") {
      payload.asignaturas_relacionadas = csvSplit(payload.asignaturas_relacionadas);
    }

    if (editing) {
      patch(route("admin.tic.update", editing.id), {
        data: payload,
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    } else {
      post(route("admin.tic.store"), {
        data: payload,
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    }
  };

  // Acciones
  const destroy = (id) => {
    if (confirm("¿Enviar este proyecto a la papelera?")) {
      router.delete(route("admin.tic-projects.destroy", id), { preserveScroll: true });
    }
  };
  const restore = (id) =>
    router.patch(route("admin.tic-projects.restore", id), {}, { preserveScroll: true });
  const forceDelete = (id) => {
    if (confirm("⚠️ Esto elimina definitivamente. ¿Continuar?")) {
      router.delete(route("admin.tic-projects.force-destroy", id), { preserveScroll: true });
    }
  };

  const applyFilters = () => {
    router.get(
      route("admin.tic-projects.index"),
      {
        q, carrera, profesor, semestre, linea, asignatura, estado, trash, sort, dir,
        per_page: perPage,
        page: 1,
      },
      { preserveScroll: true, replace: true }
    );
  };

  const goto = (url) => {
    if (!url) return;
    router.visit(url, { preserveScroll: true, replace: true });
  };

  const ActiveBadge = ({ v }) =>
    v ? (
      <span className="inline-flex items-center gap-1 rounded-lg bg-green-100 px-2 py-0.5 text-xs font-semibold text-green-700 dark:bg-green-900/30 dark:text-green-300">
        <FiCheckCircle /> Publicado
      </span>
    ) : (
      <span className="inline-flex items-center gap-1 rounded-lg bg-yellow-100 px-2 py-0.5 text-xs font-semibold text-yellow-700 dark:bg-yellow-900/30 dark:text-yellow-300">
        <FiMinusCircle /> Borrador
      </span>
    );

  // Decide estado booleano publicado/borrador para badge
  const isPublished = (estadoStr) => (estadoStr || "").toLowerCase() === "publicado";

  return (
    <DashboardLayout auth={auth} titulo="Gestión · Proyectos TIC">
      {/* Filtros */}
      <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
        <div className="flex flex-col gap-2 sm:flex-row sm:items-center sm:flex-wrap">
          <div className="flex items-center rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 dark:border-primary-800/60 dark:bg-primary-950/80">
            <FiSearch className="opacity-70" />
            <input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              onKeyDown={(e) => e.key === "Enter" && applyFilters()}
              placeholder="Buscar por nombre, profesor, carrera, línea…"
              className="ml-2 w-80 bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            />
          </div>

          <select value={carrera} onChange={(e)=>setCarrera(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="">Carrera: Todas</option>
            {catalogs?.carreras?.map((v)=> <option key={v} value={v}>{v}</option>)}
          </select>

          <select value={profesor} onChange={(e)=>setProfesor(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="">Profesor: Todos</option>
            {catalogs?.profesores?.map((v)=> <option key={v} value={v}>{v}</option>)}
          </select>

          <select value={semestre} onChange={(e)=>setSemestre(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="">Semestre: Todos</option>
            {catalogs?.semestres?.map((v)=> <option key={v} value={v}>{v}</option>)}
          </select>

          <select value={linea} onChange={(e)=>setLinea(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="">Línea: Todas</option>
            {catalogs?.lineas?.map((v)=> <option key={v} value={v}>{v}</option>)}
          </select>

          <select value={asignatura} onChange={(e)=>setAsignatura(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="">Asignatura: Todas</option>
            {catalogs?.asignaturas?.map((v)=> <option key={v} value={v}>{v}</option>)}
          </select>

          <select value={estado} onChange={(e)=>setEstado(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="">Estado: Todos</option>
            {catalogs?.estados?.map((es)=> <option key={es} value={es}>{es}</option>)}
          </select>

          <select value={trash} onChange={(e)=>setTrash(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="none">Sin papelera</option>
            <option value="with">Incluir papelera</option>
            <option value="only">Solo papelera</option>
          </select>

          <select value={sort} onChange={(e)=>setSort(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="created_at">Orden: Creación</option>
            <option value="nombre">Orden: Nombre</option>
            <option value="profesor">Orden: Profesor</option>
            <option value="carrera">Orden: Carrera</option>
            <option value="semestre">Orden: Semestre</option>
            <option value="linea">Orden: Línea</option>
            <option value="estado">Orden: Estado</option>
          </select>

          <select value={dir} onChange={(e)=>setDir(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="desc">Descendente</option>
            <option value="asc">Ascendente</option>
          </select>

          <button
            onClick={applyFilters}
            className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700"
          >
            Aplicar
          </button>
        </div>

        {/* NUEVO */}
        {trash !== "only" && (
          <button
            onClick={openCreate}
            className="flex items-center justify-center gap-2 rounded-xl bg-gradient-to-r from-primary-600 to-secondary-600 px-4 py-2 text-sm font-semibold text-white shadow hover:opacity-90"
          >
            <FiPlus className="text-lg" /> Nuevo
          </button>
        )}
      </div>

      {/* Tabla */}
      <div className="overflow-x-auto rounded-2xl border border-primary-200/60 bg-white/90 shadow-sm backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/80">
        <table className="min-w-full divide-y divide-primary-200 dark:divide-primary-800">
          <thead className="bg-primary-50 dark:bg-primary-900/40">
            <tr className="text-sm text-primary-700 dark:text-primary-200">
              <th className="px-4 py-2 text-left">Nombre</th>
              <th className="px-4 py-2 text-left">Carrera</th>
              <th className="px-4 py-2 text-left">Profesor</th>
              <th className="px-4 py-2 text-left">Sem.</th>
              <th className="px-4 py-2 text-left">Línea</th>
              <th className="px-4 py-2 text-left">Áreas</th>
              <th className="px-4 py-2 text-left">Asignaturas</th>
              <th className="px-4 py-2 text-center">Estado</th>
              <th className="px-4 py-2 text-center">Acciones</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-primary-100 dark:divide-primary-800">
            {projects.data.map((p) => (
              <tr key={p.id} className="text-sm hover:bg-primary-50/60 dark:hover:bg-primary-900/30">
                <td className="px-4 py-2">
                  <div className="font-medium">{p.nombreProyecto}</div>
                  {p.resumen && (
                    <div className="text-xs text-primary-600/80 line-clamp-1">{p.resumen}</div>
                  )}
                </td>
                <td className="px-4 py-2">{p.carrera || <span className="text-xs text-gray-400 italic">—</span>}</td>
                <td className="px-4 py-2">{p.profesorCargo || <span className="text-xs text-gray-400 italic">—</span>}</td>
                <td className="px-4 py-2">{p.semestre || <span className="text-xs text-gray-400 italic">—</span>}</td>
                <td className="px-4 py-2">{p.lineaInvestigacion || <span className="text-xs text-gray-400 italic">—</span>}</td>
                <td className="px-4 py-2">
                  <div className="flex flex-wrap gap-1">
                    {(p.areasTrabajo || []).slice(0, 3).map((a, i) => (
                      <span key={i} className="inline-flex items-center gap-1 rounded-md bg-primary-50 px-2 py-0.5 text-xs font-semibold text-primary-700 border border-primary-100">
                        <FiLayers /> {a}
                      </span>
                    ))}
                    {(p.areasTrabajo?.length || 0) > 3 && (
                      <span className="text-xs text-primary-600">+{p.areasTrabajo.length - 3}</span>
                    )}
                  </div>
                </td>
                <td className="px-4 py-2">
                  <div className="flex flex-wrap gap-1">
                    {(p.asignaturasRelacionadas || []).slice(0, 3).map((a, i) => (
                      <span key={i} className="inline-flex items-center gap-1 rounded-md bg-secondary-50 px-2 py-0.5 text-xs font-semibold text-secondary-700 border border-secondary-100">
                        <FiBookOpen /> {a}
                      </span>
                    ))}
                    {(p.asignaturasRelacionadas?.length || 0) > 3 && (
                      <span className="text-xs text-secondary-700">+{p.asignaturasRelacionadas.length - 3}</span>
                    )}
                  </div>
                </td>
                <td className="px-4 py-2 text-center">
                  <ActiveBadge v={isPublished(p.estado)} />
                </td>
                <td className="px-4 py-2">
                  {!p.deleted_at ? (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => openEdit(p)}
                        className="rounded-md p-1.5 text-primary-600 hover:bg-primary-100 dark:text-primary-300 dark:hover:bg-primary-900/40"
                        title="Editar"
                      >
                        <FiEdit />
                      </button>
                      <button
                        onClick={() => destroy(p.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar (papelera)"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  ) : (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => restore(p.id)}
                        className="rounded-md p-1.5 text-green-600 hover:bg-green-100 dark:text-green-300 dark:hover:bg-green-900/40"
                        title="Restaurar"
                      >
                        <FiRefreshCw />
                      </button>
                      <button
                        onClick={() => forceDelete(p.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar definitivo"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  )}
                </td>
              </tr>
            ))}
            {projects.data.length === 0 && (
              <tr>
                <td colSpan="9" className="px-4 py-6 text-center text-gray-400">
                  No hay proyectos para mostrar.
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Paginación */}
      <div className="mt-4 flex items-center justify-between text-sm">
        <div className="text-primary-700/80 dark:text-primary-200/80">
          Mostrando {projects.from ?? 0}–{projects.to ?? 0} de {projects.total ?? 0}
        </div>
        <div className="flex gap-2">
          <button
            disabled={!projects.prev_page_url}
            onClick={() => goto(projects.prev_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Anterior
          </button>
          <button
            disabled={!projects.next_page_url}
            onClick={() => goto(projects.next_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Siguiente
          </button>
        </div>
      </div>

      {/* Modal Crear/Editar */}
      <Modal show={showModal} onClose={() => setShowModal(false)} maxWidth="3xl">
        <form onSubmit={submit} className="p-6 space-y-5">
          <h2 className="mb-2 text-lg font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-200 dark:to-secondary-300">
            {editing ? "Editar proyecto TIC" : "Nuevo proyecto TIC"}
          </h2>

          {/* Básicos */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-2">
            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold">Nombre del proyecto</label>
              <input
                type="text"
                value={data.nombre_proyecto}
                onChange={(e)=>setData("nombre_proyecto", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              {errors.nombre_proyecto && <p className="text-sm text-red-500">{errors.nombre_proyecto}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Carrera</label>
              <input
                type="text"
                value={data.carrera}
                onChange={(e)=>setData("carrera", e.target.value)}
                list="carreras"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              <datalist id="carreras">
                {catalogs?.carreras?.map((v)=> <option key={v} value={v} />)}
              </datalist>
              {errors.carrera && <p className="text-sm text-red-500">{errors.carrera}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Profesor a cargo</label>
              <input
                type="text"
                value={data.profesor_cargo}
                onChange={(e)=>setData("profesor_cargo", e.target.value)}
                list="profesores"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              <datalist id="profesores">
                {catalogs?.profesores?.map((v)=> <option key={v} value={v} />)}
              </datalist>
              {errors.profesor_cargo && <p className="text-sm text-red-500">{errors.profesor_cargo}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Semestre</label>
              <input
                type="text"
                value={data.semestre}
                onChange={(e)=>setData("semestre", e.target.value)}
                list="semestres"
                placeholder="2025A, 2025B…"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              <datalist id="semestres">
                {catalogs?.semestres?.map((v)=> <option key={v} value={v} />)}
              </datalist>
              {errors.semestre && <p className="text-sm text-red-500">{errors.semestre}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Línea de investigación</label>
              <input
                type="text"
                value={data.linea_investigacion}
                onChange={(e)=>setData("linea_investigacion", e.target.value)}
                list="lineas"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              <datalist id="lineas">
                {catalogs?.lineas?.map((v)=> <option key={v} value={v} />)}
              </datalist>
              {errors.linea_investigacion && <p className="text-sm text-red-500">{errors.linea_investigacion}</p>}
            </div>
          </div>

          {/* Resumen */}
          <div>
            <label className="block text-sm font-semibold">Resumen</label>
            <textarea
              rows={3}
              value={data.resumen}
              onChange={(e)=>setData("resumen", e.target.value)}
              className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
            />
            {errors.resumen && <p className="text-sm text-red-500">{errors.resumen}</p>}
          </div>

          {/* Arrays (CSV) */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-2">
            <div>
              <label className="block text-sm font-semibold">Áreas de trabajo (CSV)</label>
              <input
  type="text"
  value={toCSV(data.areas_trabajo)}
  onChange={(e) => setData("areas_trabajo", toUniqueArray(e.target.value))}
  placeholder="Redes de Datos, Ciberseguridad, …"
  className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
/>

              {errors.areas_trabajo && <p className="text-sm text-red-500">{errors.areas_trabajo}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Asignaturas relacionadas (CSV)</label>
              <input
  type="text"
  value={toCSV(data.asignaturas_relacionadas)}
  onChange={(e) => setData("asignaturas_relacionadas", toUniqueArray(e.target.value))}
  placeholder="Aplicaciones Distribuidas, IoT, …"
  className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
/>

              {errors.asignaturas_relacionadas && <p className="text-sm text-red-500">{errors.asignaturas_relacionadas}</p>}
            </div>
          </div>

          {/* Estado */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-2">
            <div>
              <label className="block text-sm font-semibold">Estado</label>
              <select
                value={data.estado}
                onChange={(e)=>setData("estado", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              >
                {catalogs?.estados?.map((es)=> <option key={es} value={es}>{es}</option>)}
              </select>
              {errors.estado && <p className="text-sm text-red-500">{errors.estado}</p>}
            </div>
          </div>

          {/* Footer */}
          <div className="mt-2 flex justify-end gap-2">
            <button
              type="button"
              onClick={() => setShowModal(false)}
              className="rounded-xl border border-primary-200/60 px-4 py-2 text-sm dark:border-primary-800/60"
            >
              Cancelar
            </button>
            <button
              type="submit"
              disabled={processing}
              className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700 disabled:opacity-60"
            >
              {processing ? "Guardando..." : editing ? "Actualizar" : "Crear"}
            </button>
          </div>
        </form>
      </Modal>
    </DashboardLayout>
  );
}
