import React, { useEffect, useState } from "react";
import { useForm, router } from "@inertiajs/react";
import DashboardLayout from "@/Layouts/AuthenticatedLayout";
import Modal from "@/Components/Modal";
import {
  FiEdit, FiPlus, FiTrash2, FiRefreshCw, FiSearch, FiCalendar,
  FiX, FiCheckCircle, FiMinusCircle, FiExternalLink, FiHash, FiBookOpen,
} from "react-icons/fi";

/**
 * props esperadas:
 * { auth, publicaciones, filters, estados, years, flash }
 * - publicaciones: paginator con data: { id, anio, estado, titulo, autores, revista, editorial, extracto, citado_por, versiones, url, activo, deleted_at }
 */

export default function PublicacionesIndex({ auth, publicaciones, filters, estados, years, flash }) {
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState(null);

  // Filtros locales
  const [q, setQ] = useState(filters?.q ?? "");
  const [estado, setEstado] = useState(filters?.estado ?? "");
  const [revista, setRevista] = useState(filters?.revista ?? "");
  const [editorial, setEditorial] = useState(filters?.editorial ?? "");
  const [anio, setAnio] = useState(filters?.anio ?? "");
  const [anioFrom, setAnioFrom] = useState(filters?.anio_from ?? "");
  const [anioTo, setAnioTo] = useState(filters?.anio_to ?? "");
  const [citadoMin, setCitadoMin] = useState(filters?.citado_min ?? "");
  const [citadoMax, setCitadoMax] = useState(filters?.citado_max ?? "");
  const [activo, setActivo] = useState(filters?.activo ?? ""); // '', '1', '0'
  const [trash, setTrash] = useState(filters?.trash ?? "none");
  const [sort, setSort] = useState(filters?.sort ?? "anio");
  const [dir, setDir] = useState(filters?.dir ?? "desc");

  const {
    data, setData, reset, post, patch, processing, errors, clearErrors,
  } = useForm({
    anio: "",
    estado: estados?.[0] ?? "publicado",
    titulo: "",
    autores: "",
    revista: "",
    editorial: "",
    extracto: "",
    citado_por: 0,
    versiones: 1,
    url: "",
    activo: true,
  });

  useEffect(() => {
    if (flash?.success) console.log("✅", flash.success);
    if (flash?.error) console.error("❌", flash.error);
  }, [flash]);

  const openCreate = () => {
    reset();
    clearErrors();
    setEditing(null);
    setData({
      anio: "",
      estado: estados?.[0] ?? "publicado",
      titulo: "",
      autores: "",
      revista: "",
      editorial: "",
      extracto: "",
      citado_por: 0,
      versiones: 1,
      url: "",
      activo: true,
    });
    setShowModal(true);
  };

  const openEdit = (p) => {
    clearErrors();
    setEditing(p);
    setData({
      anio: p.anio ?? "",
      estado: p.estado ?? (estados?.[0] ?? "publicado"),
      titulo: p.titulo ?? "",
      autores: p.autores ?? "",
      revista: p.revista ?? "",
      editorial: p.editorial ?? "",
      extracto: p.extracto ?? "",
      citado_por: p.citado_por ?? 0,
      versiones: p.versiones ?? 1,
      url: p.url ?? "",
      activo: !!p.activo,
    });
    setShowModal(true);
  };

  const submit = (e) => {
    e.preventDefault();
    if (editing) {
      patch(route("admin.publicaciones.update", editing.id), {
        data,
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    } else {
      post(route("admin.publicaciones.store"), {
        data,
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    }
  };

  // Acciones
  const destroy = (id) => {
    if (confirm("¿Enviar esta publicación a la papelera?")) {
      router.delete(route("admin.publicaciones.destroy", id), { preserveScroll: true });
    }
  };
  const restore = (id) =>
    router.patch(route("admin.publicaciones.restore", id), {}, { preserveScroll: true });
  const forceDelete = (id) => {
    if (confirm("⚠️ Esto elimina definitivamente. ¿Continuar?")) {
      router.delete(route("admin.publicaciones.force", id), { preserveScroll: true });
    }
  };

  const applyFilters = () => {
    router.get(
      route("admin.publicaciones.index"),
      {
        q, estado, revista, editorial,
        anio, anio_from: anioFrom, anio_to: anioTo,
        citado_min: citadoMin, citado_max: citadoMax,
        activo, trash, sort, dir, page: 1
      },
      { preserveScroll: true, replace: true }
    );
  };

  const goto = (url) => {
    if (!url) return;
    router.visit(url, { preserveScroll: true, replace: true });
  };

  const ActiveBadge = ({ v }) =>
    v ? (
      <span className="inline-flex items-center gap-1 rounded-lg bg-green-100 px-2 py-0.5 text-xs font-semibold text-green-700 dark:bg-green-900/30 dark:text-green-300">
        <FiCheckCircle /> Activa
      </span>
    ) : (
      <span className="inline-flex items-center gap-1 rounded-lg bg-red-100 px-2 py-0.5 text-xs font-semibold text-red-700 dark:bg-red-900/30 dark:text-red-300">
        <FiMinusCircle /> Inactiva
      </span>
    );

  return (
    <DashboardLayout auth={auth} titulo="Gestión de Publicaciones">
      {/* Filtros */}
      <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
        <div className="flex flex-col gap-2 sm:flex-row sm:items-center sm:flex-wrap">
          <div className="flex items-center rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 dark:border-primary-800/60 dark:bg-primary-950/80">
            <FiSearch className="opacity-70" />
            <input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              onKeyDown={(e) => e.key === "Enter" && applyFilters()}
              placeholder="Buscar por título, autores, revista, editorial…"
              className="ml-2 w-80 bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            />
          </div>

          <select value={estado} onChange={(e)=>setEstado(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="">Estado: Todos</option>
            {estados?.map((es)=> <option key={es} value={es}>{es}</option>)}
          </select>

          <input
            value={revista}
            onChange={(e)=>setRevista(e.target.value)}
            placeholder="Revista"
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          />
          <input
            value={editorial}
            onChange={(e)=>setEditorial(e.target.value)}
            placeholder="Editorial"
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          />

          <select value={anio} onChange={(e)=>setAnio(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="">Año exacto</option>
            {years?.map((y)=> <option key={y} value={y}>{y}</option>)}
          </select>

          <div className="flex items-center gap-2">
            <select value={anioFrom} onChange={(e)=>setAnioFrom(e.target.value)}
              className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
              <option value="">Desde</option>
              {years?.map((y)=> <option key={y} value={y}>{y}</option>)}
            </select>
            <select value={anioTo} onChange={(e)=>setAnioTo(e.target.value)}
              className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
              <option value="">Hasta</option>
              {years?.map((y)=> <option key={y} value={y}>{y}</option>)}
            </select>
          </div>

          <input
            type="number" min="0" placeholder="Citado desde"
            value={citadoMin} onChange={(e)=>setCitadoMin(e.target.value)}
            className="w-36 rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          />
          <input
            type="number" min="0" placeholder="Citado hasta"
            value={citadoMax} onChange={(e)=>setCitadoMax(e.target.value)}
            className="w-36 rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          />

          <select value={activo} onChange={(e)=>setActivo(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="">Estado: Todos</option>
            <option value="1">Activas</option>
            <option value="0">Inactivas</option>
          </select>

          <select value={trash} onChange={(e)=>setTrash(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="none">Sin papelera</option>
            <option value="with">Incluir papelera</option>
            <option value="only">Solo papelera</option>
          </select>

          <select value={sort} onChange={(e)=>setSort(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="anio">Orden: Año</option>
            <option value="titulo">Orden: Título</option>
            <option value="revista">Orden: Revista</option>
            <option value="editorial">Orden: Editorial</option>
            <option value="citado_por">Orden: Citado por</option>
            <option value="created_at">Orden: Creación</option>
          </select>

          <select value={dir} onChange={(e)=>setDir(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80">
            <option value="desc">Descendente</option>
            <option value="asc">Ascendente</option>
          </select>

          <button
            onClick={applyFilters}
            className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700"
          >
            Aplicar
          </button>
        </div>

        {/* NUEVO */}
        {trash !== "only" && (
          <button
            onClick={openCreate}
            className="flex items-center justify-center gap-2 rounded-xl bg-gradient-to-r from-primary-600 to-secondary-600 px-4 py-2 text-sm font-semibold text-white shadow hover:opacity-90"
          >
            <FiPlus className="text-lg" /> Nuevo
          </button>
        )}
      </div>

      {/* Tabla */}
      <div className="overflow-x-auto rounded-2xl border border-primary-200/60 bg-white/90 shadow-sm backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/80">
        <table className="min-w-full divide-y divide-primary-200 dark:divide-primary-800">
          <thead className="bg-primary-50 dark:bg-primary-900/40">
            <tr className="text-sm text-primary-700 dark:text-primary-200">
              <th className="px-4 py-2 text-left">Año</th>
              <th className="px-4 py-2 text-left">Estado</th>
              <th className="px-4 py-2 text-left">Título</th>
              <th className="px-4 py-2 text-left">Autores</th>
              <th className="px-4 py-2 text-left">Revista</th>
              <th className="px-4 py-2 text-left">Editorial</th>
              <th className="px-4 py-2 text-left">Citado por</th>
              <th className="px-4 py-2 text-left">Ver.</th>
              <th className="px-4 py-2 text-center">Activa</th>
              <th className="px-4 py-2 text-center">Acciones</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-primary-100 dark:divide-primary-800">
            {publicaciones.data.map((p) => (
              <tr key={p.id} className="text-sm hover:bg-primary-50/60 dark:hover:bg-primary-900/30">
                <td className="px-4 py-2 font-mono text-xs">{p.anio}</td>
                <td className="px-4 py-2 capitalize">{p.estado}</td>
                <td className="px-4 py-2">
                  <div className="font-medium">{p.titulo}</div>
                  {p.url && (
                    <a
                      href={p.url}
                      target="_blank"
                      rel="noreferrer"
                      className="inline-flex items-center gap-1 text-xs text-primary-600 hover:underline"
                    >
                      <FiExternalLink /> enlace
                    </a>
                  )}
                </td>
                <td className="px-4 py-2">{p.autores}</td>
                <td className="px-4 py-2">{p.revista || <span className="text-xs text-gray-400 italic">—</span>}</td>
                <td className="px-4 py-2">{p.editorial || <span className="text-xs text-gray-400 italic">—</span>}</td>
                <td className="px-4 py-2">{p.citado_por}</td>
                <td className="px-4 py-2">{p.versiones}</td>
                <td className="px-4 py-2 text-center">
                  <ActiveBadge v={!!p.activo} />
                </td>
                <td className="px-4 py-2">
                  {!p.deleted_at ? (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => openEdit(p)}
                        className="rounded-md p-1.5 text-primary-600 hover:bg-primary-100 dark:text-primary-300 dark:hover:bg-primary-900/40"
                        title="Editar"
                      >
                        <FiEdit />
                      </button>
                      <button
                        onClick={() => destroy(p.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar (papelera)"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  ) : (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => restore(p.id)}
                        className="rounded-md p-1.5 text-green-600 hover:bg-green-100 dark:text-green-300 dark:hover:bg-green-900/40"
                        title="Restaurar"
                      >
                        <FiRefreshCw />
                      </button>
                      <button
                        onClick={() => forceDelete(p.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar definitivo"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  )}
                </td>
              </tr>
            ))}
            {publicaciones.data.length === 0 && (
              <tr>
                <td colSpan="10" className="px-4 py-6 text-center text-gray-400">
                  No hay publicaciones para mostrar.
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Paginación */}
      <div className="mt-4 flex items-center justify-between text-sm">
        <div className="text-primary-700/80 dark:text-primary-200/80">
          Mostrando {publicaciones.from ?? 0}–{publicaciones.to ?? 0} de {publicaciones.total ?? 0}
        </div>
        <div className="flex gap-2">
          <button
            disabled={!publicaciones.prev_page_url}
            onClick={() => goto(publicaciones.prev_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Anterior
          </button>
          <button
            disabled={!publicaciones.next_page_url}
            onClick={() => goto(publicaciones.next_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Siguiente
          </button>
        </div>
      </div>

      {/* Modal Crear/Editar */}
      <Modal show={showModal} onClose={() => setShowModal(false)} maxWidth="3xl">
        <form onSubmit={submit} className="p-6 space-y-5">
          <h2 className="mb-2 text-lg font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-200 dark:to-secondary-300">
            {editing ? "Editar publicación" : "Nueva publicación"}
          </h2>

          {/* Básicos */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-4">
            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Año</label>
              <select
                value={data.anio}
                onChange={(e)=>setData("anio", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              >
                <option value="">Seleccionar…</option>
                {years?.map((y)=> <option key={y} value={y}>{y}</option>)}
              </select>
              {errors.anio && <p className="text-sm text-red-500">{errors.anio}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Estado</label>
              <select
                value={data.estado}
                onChange={(e)=>setData("estado", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              >
                {estados?.map((es)=> <option key={es} value={es}>{es}</option>)}
              </select>
              {errors.estado && <p className="text-sm text-red-500">{errors.estado}</p>}
            </div>

            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold">Título</label>
              <input
                type="text"
                value={data.titulo}
                onChange={(e)=>setData("titulo", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              {errors.titulo && <p className="text-sm text-red-500">{errors.titulo}</p>}
            </div>
          </div>

          {/* Autores, revista, editorial */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Autores</label>
              <input
                type="text"
                value={data.autores}
                onChange={(e)=>setData("autores", e.target.value)}
                placeholder="A Apellido, B Apellido, C Apellido"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              {errors.autores && <p className="text-sm text-red-500">{errors.autores}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Revista</label>
              <input
                type="text"
                value={data.revista}
                onChange={(e)=>setData("revista", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              {errors.revista && <p className="text-sm text-red-500">{errors.revista}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Editorial</label>
              <input
                type="text"
                value={data.editorial}
                onChange={(e)=>setData("editorial", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              {errors.editorial && <p className="text-sm text-red-500">{errors.editorial}</p>}
            </div>
          </div>

          {/* Extracto */}
          <div>
            <label className="block text-sm font-semibold">Extracto</label>
            <textarea
              rows={3}
              value={data.extracto}
              onChange={(e)=>setData("extracto", e.target.value)}
              className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
            />
            {errors.extracto && <p className="text-sm text-red-500">{errors.extracto}</p>}
          </div>

          {/* Métricas y URL */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Citado por</label>
              <input
                type="number" min="0"
                value={data.citado_por}
                onChange={(e)=>setData("citado_por", Number(e.target.value))}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              {errors.citado_por && <p className="text-sm text-red-500">{errors.citado_por}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Versiones</label>
              <input
                type="number" min="1"
                value={data.versiones}
                onChange={(e)=>setData("versiones", Number(e.target.value))}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              {errors.versiones && <p className="text-sm text-red-500">{errors.versiones}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">URL</label>
              <input
                type="url"
                value={data.url}
                onChange={(e)=>setData("url", e.target.value)}
                placeholder="https://ejemplo.com/articulo"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              {errors.url && <p className="text-sm text-red-500">{errors.url}</p>}
            </div>
          </div>

          {/* Activo */}
          <div className="sm:col-span-1">
            <label className="block text-sm font-semibold">Estado</label>
            <div className="mt-1 flex items-center gap-2">
              <input
                id="activo"
                type="checkbox"
                className="h-4 w-4"
                checked={!!data.activo}
                onChange={(e) => setData("activo", e.target.checked)}
              />
              <label htmlFor="activo" className="text-sm">
                Activa
              </label>
            </div>
          </div>

          {/* Footer */}
          <div className="mt-2 flex justify-end gap-2">
            <button
              type="button"
              onClick={() => setShowModal(false)}
              className="rounded-xl border border-primary-200/60 px-4 py-2 text-sm dark:border-primary-800/60"
            >
              Cancelar
            </button>
            <button
              type="submit"
              disabled={processing}
              className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700 disabled:opacity-60"
            >
              {processing ? "Guardando..." : editing ? "Actualizar" : "Crear"}
            </button>
          </div>
        </form>
      </Modal>
    </DashboardLayout>
  );
}
