import React, { useEffect, useMemo, useState } from "react";
import { useForm, router } from "@inertiajs/react";
import DashboardLayout from "@/Layouts/AuthenticatedLayout";
import Modal from "@/Components/Modal";
import {
  FiEdit,
  FiPlus,
  FiTrash2,
  FiRefreshCw,
  FiSearch,
  FiTag,
  FiType,
  FiUser,
  FiCalendar,
  FiLayers,
} from "react-icons/fi";

export default function ProyectosIndex({ auth, proyectos, lineas, estados, anos, filters, flash }) {
  // UI
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState(null);

  // Filtros locales
  const [q, setQ] = useState(filters?.q ?? "");
  const [linea, setLinea] = useState(filters?.linea ?? "");
  const [estado, setEstado] = useState(filters?.estado ?? "");
  const [profesor, setProfesor] = useState(filters?.profesor ?? "");
  const [ano, setAno] = useState(filters?.ano ?? "");
  const [trash, setTrash] = useState(filters?.trash ?? "none");

  // Form crear/editar
  const { data, setData, reset, post, patch, processing, errors, clearErrors } = useForm({
    titulo: "",
    linea_investigacion: "",
    profesor_a_cargo: "",
    ano: "",
    estado: "En Desarrollo",
    descripcion: "",
    keywords: [], // array de strings
  });

  useEffect(() => {
    if (flash?.success) console.log("✅", flash.success);
    if (flash?.error) console.error("❌", flash.error);
  }, [flash]);

  // Modal: Crear
  const openCreate = () => {
    reset();
    setEditing(null);
    clearErrors();
    setShowModal(true);
  };

  // Modal: Editar
  const openEdit = (p) => {
    setEditing(p);
    setData({
      titulo: p.titulo ?? "",
      linea_investigacion: p.linea_investigacion ?? "",
      profesor_a_cargo: p.profesor_a_cargo ?? "",
      ano: p.ano ?? "",
      estado: p.estado ?? "En Desarrollo",
      descripcion: p.descripcion ?? "",
      keywords: Array.isArray(p.keywords) ? p.keywords : [],
    });
    clearErrors();
    setShowModal(true);
  };

  // Submit modal
  const submit = (e) => {
    e.preventDefault();
    if (editing) {
      patch(route("admin.proyectos.update", editing.id), {
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    } else {
      post(route("admin.proyectos.store"), {
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    }
  };

  // Acciones
  const destroy = (id) => {
    if (confirm("¿Enviar este proyecto a la papelera?")) {
      router.delete(route("admin.proyectos.destroy", id), { preserveScroll: true });
    }
  };
  const restore = (id) =>
    router.patch(route("admin.proyectos.restore", id), {}, { preserveScroll: true });
  const forceDelete = (id) => {
    if (confirm("⚠️ Esto elimina definitivamente. ¿Continuar?")) {
      router.delete(route("admin.proyectos.force", id), { preserveScroll: true });
    }
  };

  // Filtro/búsqueda
  const applyFilters = () => {
    router.get(
      route("admin.proyectos.index"),
      { q, linea, estado, profesor, ano, trash, page: 1 },
      { preserveScroll: true, replace: true }
    );
  };

  // Paginación
  const goto = (url) => url && router.visit(url, { preserveScroll: true, replace: true });

  // Helpers UI
  const keywordsPreview = (arr) =>
    Array.isArray(arr) && arr.length > 0 ? (
      <div className="flex flex-wrap gap-1">
        {arr.map((k, i) => (
          <span
            key={`${k}-${i}`}
            className="inline-flex items-center gap-1 rounded-lg border border-primary-200/60 bg-primary-50/70 px-2 py-0.5 text-[11px] font-medium dark:border-primary-800/60 dark:bg-primary-900/40"
          >
            <FiTag className="opacity-70" />
            {k}
          </span>
        ))}
      </div>
    ) : (
      <span className="text-xs text-gray-400 italic">Sin keywords</span>
    );

  // ------ Keywords dinámicos ------
  const [kwInput, setKwInput] = useState("");
  const addKeyword = () => {
    const v = kwInput.trim();
    if (!v) return;
    if ((data.keywords || []).includes(v)) return;
    setData("keywords", [...(data.keywords || []), v]);
    setKwInput("");
  };
  const removeKeyword = (idx) => {
    const arr = [...(data.keywords || [])];
    arr.splice(idx, 1);
    setData("keywords", arr);
  };

  // Catálogos
  const lineaOptions = useMemo(() => (Array.isArray(lineas) ? lineas : []), [lineas]);
  const estadoOptions = useMemo(() => (Array.isArray(estados) ? estados : []), [estados]);
  const anoOptions = useMemo(() => (Array.isArray(anos) ? anos : []), [anos]);

  return (
    <DashboardLayout auth={auth} titulo="Gestión de Proyectos de Investigación">
      {/* Barra superior filtros + NUEVO */}
      <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
        {/* Filtros */}
        <div className="flex flex-col gap-2 sm:flex-row sm:items-center">
          <div className="flex items-center rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 dark:border-primary-800/60 dark:bg-primary-950/80">
            <FiSearch className="opacity-70" />
            <input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              onKeyDown={(e) => e.key === "Enter" && applyFilters()}
              placeholder="Buscar por título, profesor o descripción…"
              className="ml-2 w-72 bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            />
          </div>

          <select
            value={linea}
            onChange={(e) => setLinea(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Línea: Todas</option>
            {lineaOptions.map((l) => (
              <option key={l} value={l}>
                {l}
              </option>
            ))}
          </select>

          <select
            value={estado}
            onChange={(e) => setEstado(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Estado: Todos</option>
            {estadoOptions.map((e2) => (
              <option key={e2} value={e2}>
                {e2}
              </option>
            ))}
          </select>

          <input
            value={profesor}
            onChange={(e) => setProfesor(e.target.value)}
            placeholder="Profesor a cargo…"
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          />

          <select
            value={ano}
            onChange={(e) => setAno(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Año: Todos</option>
            {anoOptions.map((y) => (
              <option key={y} value={y}>
                {y}
              </option>
            ))}
          </select>

          <select
            value={trash}
            onChange={(e) => setTrash(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="none">Sin papelera</option>
            <option value="with">Incluir papelera</option>
            <option value="only">Solo papelera</option>
          </select>

          <button
            onClick={applyFilters}
            className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700"
          >
            Aplicar
          </button>
        </div>

        {/* Nuevo Proyecto */}
        {trash !== "only" && (
          <button
            onClick={openCreate}
            className="flex items-center justify-center gap-2 rounded-xl bg-gradient-to-r from-primary-600 to-secondary-600 px-4 py-2 text-sm font-semibold text-white shadow hover:opacity-90"
          >
            <FiPlus className="text-lg" /> Nuevo
          </button>
        )}
      </div>

      {/* Tabla */}
      <div className="overflow-x-auto rounded-2xl border border-primary-200/60 bg-white/90 shadow-sm backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/80">
        <table className="min-w-full divide-y divide-primary-200 dark:divide-primary-800">
          <thead className="bg-primary-50 dark:bg-primary-900/40">
            <tr className="text-sm text-primary-700 dark:text-primary-200">
              <th className="px-4 py-2 text-left">Título</th>
              <th className="px-4 py-2 text-left">Línea</th>
              <th className="px-4 py-2 text-left">Profesor/a</th>
              <th className="px-4 py-2 text-left">Año</th>
              <th className="px-4 py-2 text-left">Estado</th>
              <th className="px-4 py-2 text-left">Keywords</th>
              <th className="px-4 py-2 text-center">Acciones</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-primary-100 dark:divide-primary-800">
            {proyectos.data.map((p) => (
              <tr key={p.id} className="text-sm hover:bg-primary-50/60 dark:hover:bg-primary-900/30">
                <td className="px-4 py-2 font-medium">{p.titulo}</td>
                <td className="px-4 py-2">{p.linea_investigacion}</td>
                <td className="px-4 py-2">{p.profesor_a_cargo}</td>
                <td className="px-4 py-2">{p.ano}</td>
                <td className="px-4 py-2">
                  <span
                    className={`rounded-md px-2 py-0.5 text-xs font-semibold ${
                      p.estado === "Completado"
                        ? "bg-green-100 text-green-700 dark:bg-green-900/40 dark:text-green-300"
                        : "bg-yellow-100 text-yellow-700 dark:bg-yellow-900/40 dark:text-yellow-300"
                    }`}
                  >
                    {p.estado}
                  </span>
                </td>
                <td className="px-4 py-2">{keywordsPreview(p.keywords)}</td>
                <td className="px-4 py-2">
                  {!p.deleted_at ? (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => openEdit(p)}
                        className="rounded-md p-1.5 text-primary-600 hover:bg-primary-100 dark:text-primary-300 dark:hover:bg-primary-900/40"
                        title="Editar"
                      >
                        <FiEdit />
                      </button>
                      <button
                        onClick={() => destroy(p.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar (papelera)"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  ) : (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => restore(p.id)}
                        className="rounded-md p-1.5 text-green-600 hover:bg-green-100 dark:text-green-300 dark:hover:bg-green-900/40"
                        title="Restaurar"
                      >
                        <FiRefreshCw />
                      </button>
                      <button
                        onClick={() => forceDelete(p.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar definitivo"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  )}
                </td>
              </tr>
            ))}
            {proyectos.data.length === 0 && (
              <tr>
                <td colSpan="7" className="px-4 py-6 text-center text-gray-400">
                  No hay proyectos para mostrar.
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Paginación */}
      <div className="mt-4 flex items-center justify-between text-sm">
        <div className="text-primary-700/80 dark:text-primary-200/80">
          Mostrando {proyectos.from ?? 0}–{proyectos.to ?? 0} de {proyectos.total ?? 0}
        </div>
        <div className="flex gap-2">
          <button
            disabled={!proyectos.prev_page_url}
            onClick={() => goto(proyectos.prev_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Anterior
          </button>
          <button
            disabled={!proyectos.next_page_url}
            onClick={() => goto(proyectos.next_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Siguiente
          </button>
        </div>
      </div>

      {/* Modal Crear/Editar */}
      <Modal show={showModal} onClose={() => setShowModal(false)} maxWidth="2xl">
        <form onSubmit={submit} className="p-6 space-y-4">
          <h2 className="mb-2 text-lg font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-200 dark:to-secondary-300">
            {editing ? "Editar proyecto" : "Nuevo proyecto"}
          </h2>

          {/* Básicos */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-2">
            <div>
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiType /> Título
              </label>
              <input
                type="text"
                value={data.titulo}
                onChange={(e) => setData("titulo", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              {errors.titulo && <p className="text-sm text-red-500">{errors.titulo}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiLayers /> Línea de investigación
              </label>
              <select
                value={data.linea_investigacion}
                onChange={(e) => setData("linea_investigacion", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              >
                <option value="">Selecciona…</option>
                {lineaOptions.map((l) => (
                  <option key={l} value={l}>{l}</option>
                ))}
              </select>
              {errors.linea_investigacion && <p className="text-sm text-red-500">{errors.linea_investigacion}</p>}
            </div>
          </div>

          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiUser /> Profesor/a a cargo
              </label>
              <input
                type="text"
                value={data.profesor_a_cargo}
                onChange={(e) => setData("profesor_a_cargo", e.target.value)}
                placeholder="Nombre completo"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              {errors.profesor_a_cargo && <p className="text-sm text-red-500">{errors.profesor_a_cargo}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiCalendar /> Año
              </label>
              <select
                value={data.ano}
                onChange={(e) => setData("ano", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              >
                <option value="">Selecciona…</option>
                {anoOptions.map((y) => (
                  <option key={y} value={y}>{y}</option>
                ))}
              </select>
              {errors.ano && <p className="text-sm text-red-500">{errors.ano}</p>}
            </div>
          </div>

          <div className="grid grid-cols-1 gap-3 sm:grid-cols-2">
            <div>
              <label className="block text-sm font-semibold">Estado</label>
              <select
                value={data.estado}
                onChange={(e) => setData("estado", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              >
                {estadoOptions.map((e2) => (
                  <option key={e2} value={e2}>{e2}</option>
                ))}
              </select>
              {errors.estado && <p className="text-sm text-red-500">{errors.estado}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Descripción</label>
              <textarea
                rows={3}
                value={data.descripcion}
                onChange={(e) => setData("descripcion", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                placeholder="Resumen breve del proyecto…"
              />
              {errors.descripcion && <p className="text-sm text-red-500">{errors.descripcion}</p>}
            </div>
          </div>

          {/* Keywords */}
          <div>
            <label className="block text-sm font-semibold flex items-center gap-2">
              <FiTag /> Keywords
            </label>
            <div className="mt-1 flex gap-2">
              <input
                type="text"
                value={kwInput}
                onChange={(e) => setKwInput(e.target.value)}
                onKeyDown={(e) => e.key === "Enter" && (e.preventDefault(), addKeyword())}
                placeholder="Ej: IoT"
                className="flex-1 rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              <button
                type="button"
                onClick={addKeyword}
                className="rounded-xl bg-primary-600 px-3 py-2 text-sm font-semibold text-white hover:bg-primary-700"
              >
                Agregar
              </button>
            </div>

            <div className="mt-2 flex flex-wrap gap-2">
              {data.keywords?.map((kw, idx) => (
                <span
                  key={`${kw}-${idx}`}
                  className="group inline-flex items-center gap-2 rounded-xl border border-primary-200/60 bg-primary-50/70 px-3 py-1 text-xs font-medium dark:border-primary-800/60 dark:bg-primary-900/40"
                >
                  {kw}
                  <button
                    type="button"
                    onClick={() => removeKeyword(idx)}
                    className="rounded-md px-1 text-xs opacity-70 hover:bg-primary-100 dark:hover:bg-primary-900/60"
                    title="Quitar"
                  >
                    ×
                  </button>
                </span>
              ))}
              {(!data.keywords || data.keywords.length === 0) && (
                <span className="text-xs text-gray-400 italic">Aún sin keywords</span>
              )}
            </div>
            {errors.keywords && <p className="text-sm text-red-500">{errors.keywords}</p>}
          </div>

          {/* Footer modal */}
          <div className="mt-4 flex justify-end gap-2">
            <button
              type="button"
              onClick={() => setShowModal(false)}
              className="rounded-xl border border-primary-200/60 px-4 py-2 text-sm dark:border-primary-800/60"
            >
              Cancelar
            </button>
            <button
              type="submit"
              disabled={processing}
              className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700 disabled:opacity-60"
            >
              {processing ? "Guardando..." : editing ? "Actualizar" : "Crear"}
            </button>
          </div>
        </form>
      </Modal>
    </DashboardLayout>
  );
}
