import React, { useEffect, useState } from "react";
import { useForm, usePage, router } from "@inertiajs/react";
import DashboardLayout from "@/Layouts/AuthenticatedLayout";
import Modal from "@/Components/Modal";
import {
  FiPlus,
  FiEdit,
  FiTrash2,
  FiRefreshCw,
  FiEye,
  FiEyeOff,
  FiSearch,
  FiArrowUp,
  FiArrowDown,
  FiImage,
} from "react-icons/fi";

export default function PregradosIndex() {
  const { auth, pregrados, filtros, filters, modalidades, flash } = usePage().props;
  const f = filters ?? filtros ?? {};

  // Filtros/orden
  const [q, setQ] = useState(f.q ?? "");
  const [modalidad, setModalidad] = useState(f.modalidad ?? "");
  const [visible, setVisible] = useState(f.visible ?? "all");
  const [trash, setTrash] = useState(f.trash ?? "none");
  const [sort, setSort] = useState(f.sort ?? "orden");
  const [dir, setDir] = useState(f.dir ?? "asc");

  // UI
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState(null);

  // Previews locales de archivos
  const [prevCoord, setPrevCoord] = useState(null);
  const [prevHero, setPrevHero] = useState(null);
  const [prevCard, setPrevCard] = useState(null);

  // Form
  const { data, setData, reset, post, patch, processing, errors } = useForm({
    nombre: "",
    duracion: "",
    modalidad: "",
    titulo: "",
    coordinador_nombre: "",
    coordinador_email: "",
    // archivos
    coordinador_foto: null,
    imagen_hero: null,
    foto_tarjeta: null,
    malla: null, // PDF
    // textos
    informacion: "",
    perfil: "",
    requisitos: [], // <— ahora array real
    is_visible: true,
    orden: 0,
  });

  // input para agregar requisitos
  const [reqInput, setReqInput] = useState("");

  useEffect(() => {
    if (flash?.success) console.log("✅", flash.success);
    if (flash?.error) console.error("❌", flash.error);
  }, [flash]);

  // Helpers
  const storageUrl = (path) => {
    if (!path) return null;
    return path.startsWith("http") ? path : `/storage/${path}`;
  };

  // Filtros
  const applyFilters = () => {
    router.get(
      route("admin.pregrados.index"),
      { q, modalidad, visible, trash, sort, dir, page: 1 },
      { preserveScroll: true, replace: true }
    );
  };

  const toggleSort = (key) => {
    if (sort === key) {
      setDir(dir === "asc" ? "desc" : "asc");
    } else {
      setSort(key);
      setDir("asc");
    }
    setTimeout(applyFilters, 0);
  };

  // Crear
  const openCreate = () => {
    reset();
    setEditing(null);
    setData("is_visible", true);
    setData("orden", 0);
    setData("requisitos", []);
    setPrevCoord(null);
    setPrevHero(null);
    setPrevCard(null);
    setReqInput("");
    setShowModal(true);
  };

  // Editar
  const openEdit = (item) => {
    setEditing(item);
    setData({
      nombre: item.nombre || "",
      duracion: item.duracion || "",
      modalidad: item.modalidad || "",
      titulo: item.titulo || "",
      coordinador_nombre: item.coordinador?.nombre || "",
      coordinador_email: item.coordinador?.email || "",
      coordinador_foto: null,
      imagen_hero: null,
      foto_tarjeta: null,
      malla: null,
      informacion: item.informacion || "",
      perfil: item.perfil || "",
      requisitos: Array.isArray(item.requisitos) ? item.requisitos : [],
      is_visible: item.is_visible,
      orden: item.orden ?? 0,
    });
    // Previews: si ya tienen guardadas
    setPrevCoord(item?.coordinador?.foto ? storageUrl(item.coordinador.foto) : null);
    setPrevHero(item?.imagen_hero ? storageUrl(item.imagen_hero) : null);
    setPrevCard(item?.foto_tarjeta ? storageUrl(item.foto_tarjeta) : null);
    setReqInput("");
    setShowModal(true);
  };

  // Submit (FormData)
  const submit = (e) => {
    e.preventDefault();
    const payload = { ...data }; // requisitos ya está como array

    if (editing) {
      patch(route("admin.pregrados.update", editing.id), {
        data: payload,
        forceFormData: true,
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    } else {
      post(route("admin.pregrados.store"), {
        data: payload,
        forceFormData: true,
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    }
  };

  // Acciones rápidas
  const toggleVisible = (id) =>
    router.patch(route("admin.pregrados.toggle", id), {}, { preserveScroll: true });
  const destroy = (id) => {
    if (confirm("¿Enviar a papelera?")) {
      router.delete(route("admin.pregrados.destroy", id), { preserveScroll: true });
    }
  };
  const restore = (id) =>
    router.patch(route("admin.pregrados.restore", id), {}, { preserveScroll: true });
  const forceDelete = (id) => {
    if (confirm("⚠️ Eliminación definitiva. ¿Continuar?")) {
      router.delete(route("admin.pregrados.force", id), { preserveScroll: true });
    }
  };
  const goto = (url) => url && router.visit(url, { preserveScroll: true, replace: true });

  // ===== Requisitos (chips) =====
  const addRequisito = () => {
    const v = (reqInput || "").trim();
    if (!v) return;
    if ((data.requisitos || []).includes(v)) {
      setReqInput("");
      return;
    }
    setData("requisitos", [...(data.requisitos || []), v]);
    setReqInput("");
  };
  const removeRequisito = (idx) => {
    const next = [...(data.requisitos || [])];
    next.splice(idx, 1);
    setData("requisitos", next);
  };

  // ===== Files con preview =====
  const onFilePreview = (file, setterPreview, setFormKey) => {
    if (!file) {
      setterPreview(null);
      setData(setFormKey, null);
      return;
    }
    setData(setFormKey, file);
    const reader = new FileReader();
    reader.onload = (ev) => setterPreview(ev.target.result);
    reader.readAsDataURL(file);
  };

  const Chip = ({ children, onRemove }) => (
    <span className="group inline-flex items-center gap-2 rounded-xl border border-primary-200/60 bg-primary-50/70 px-3 py-1 text-xs font-medium dark:border-primary-800/60 dark:bg-primary-900/40">
      {children}
      <button
        type="button"
        onClick={onRemove}
        className="rounded-md px-1 text-xs opacity-70 hover:bg-primary-100 dark:hover:bg-primary-900/60"
        title="Quitar"
      >
        ×
      </button>
    </span>
  );

  return (
    <DashboardLayout auth={auth} titulo="Carreras de Pregrado">
      {/* Filtros + Nuevo */}
      <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
        <div className="flex flex-col gap-2 sm:flex-row sm:items-center">
          <div className="flex items-center rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 dark:border-primary-800/60 dark:bg-primary-950/80">
            <FiSearch className="opacity-70" />
            <input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              onKeyDown={(e) => e.key === "Enter" && applyFilters()}
              placeholder="Buscar por nombre, título o coordinador…"
              className="ml-2 w-72 bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            />
          </div>

          <select
            value={modalidad}
            onChange={(e) => setModalidad(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Modalidad: Todas</option>
            {modalidades.map((m) => (
              <option key={m} value={m}>
                {m}
              </option>
            ))}
          </select>

          <select
            value={visible}
            onChange={(e) => setVisible(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="all">Visibilidad: Todas</option>
            <option value="1">Solo visibles</option>
            <option value="0">Solo ocultas</option>
          </select>

          <select
            value={trash}
            onChange={(e) => setTrash(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="none">Sin papelera</option>
            <option value="with">Incluir papelera</option>
            <option value="only">Solo papelera</option>
          </select>

          <button
            onClick={applyFilters}
            className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700"
          >
            Aplicar
          </button>
        </div>

        {trash !== "only" && (
          <button
            onClick={openCreate}
            className="flex justify-center items-center gap-2 rounded-xl bg-gradient-to-r from-primary-600 to-secondary-600 px-4 py-2 text-sm font-semibold text-white shadow hover:opacity-90"
          >
            <FiPlus className="inline-block text-lg" /> Nueva
          </button>
        )}
      </div>

      {/* Tabla */}
      <div className="overflow-x-auto rounded-2xl border border-primary-200/60 bg-white/90 shadow-sm backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/80">
        <table className="min-w-full divide-y divide-primary-200 dark:divide-primary-800">
          <thead className="bg-primary-50 dark:bg-primary-900/40">
            <tr className="text-sm text-primary-700 dark:text-primary-200">
              <th className="px-4 py-2 text-left cursor-pointer" onClick={() => toggleSort("orden")}>
                Orden {sort === "orden" ? (dir === "asc" ? <FiArrowUp className="inline" /> : <FiArrowDown className="inline" />) : null}
              </th>
              <th className="px-4 py-2 text-left cursor-pointer" onClick={() => toggleSort("nombre")}>
                Nombre {sort === "nombre" ? (dir === "asc" ? <FiArrowUp className="inline" /> : <FiArrowDown className="inline" />) : null}
              </th>
              <th className="px-4 py-2 text-left">Slug</th>
              <th className="px-4 py-2 text-left">Modalidad</th>
              <th className="px-4 py-2 text-center">Visible</th>
              <th className="px-4 py-2 text-center">Acciones</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-primary-100 dark:divide-primary-800">
            {pregrados.data.map((p) => (
              <tr key={p.id} className="text-sm hover:bg-primary-50/60 dark:hover:bg-primary-900/30">
                <td className="px-4 py-2">{p.orden}</td>
                <td className="px-4 py-2 font-medium">{p.nombre}</td>
                <td className="px-4 py-2">{p.slug}</td>
                <td className="px-4 py-2">{p.modalidad || <span className="text-gray-400">—</span>}</td>
                <td className="px-4 py-2 text-center">
                  {p.deleted_at ? (
                    <span className="text-xs italic text-gray-400">Eliminada</span>
                  ) : (
                    <button
                      onClick={() => toggleVisible(p.id)}
                      className={`inline-flex items-center gap-1.5 rounded-lg px-3 py-1 text-xs font-semibold ${
                        p.is_visible
                          ? "bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-200"
                          : "bg-amber-100 text-amber-700 dark:bg-amber-900/30 dark:text-amber-200"
                      }`}
                    >
                      {p.is_visible ? <FiEye /> : <FiEyeOff />}
                      {p.is_visible ? "Visible" : "Oculta"}
                    </button>
                  )}
                </td>
                <td className="px-4 py-2">
                  {!p.deleted_at ? (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => openEdit(p)}
                        className="rounded-md p-1.5 text-primary-600 hover:bg-primary-100 dark:text-primary-300 dark:hover:bg-primary-900/40"
                        title="Editar"
                      >
                        <FiEdit />
                      </button>
                      <button
                        onClick={() => {
                          if (confirm("¿Enviar a papelera?")) destroy(p.id);
                        }}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar (papelera)"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  ) : (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => restore(p.id)}
                        className="rounded-md p-1.5 text-green-600 hover:bg-green-100 dark:text-green-300 dark:hover:bg-green-900/40"
                        title="Restaurar"
                      >
                        <FiRefreshCw />
                      </button>
                      <button
                        onClick={() => {
                          if (confirm("⚠️ Eliminación definitiva. ¿Continuar?")) forceDelete(p.id);
                        }}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar definitivo"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  )}
                </td>
              </tr>
            ))}
            {pregrados.data.length === 0 && (
              <tr>
                <td colSpan="6" className="px-4 py-6 text-center text-gray-400">
                  No hay carreras para mostrar.
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Paginación */}
      <div className="mt-4 flex items-center justify-between text-sm">
        <div className="text-primary-700/80 dark:text-primary-200/80">
          Mostrando {pregrados.from ?? 0}–{pregrados.to ?? 0} de {pregrados.total ?? 0}
        </div>
        <div className="flex gap-2">
          <button
            disabled={!pregrados.prev_page_url}
            onClick={() => goto(pregrados.prev_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Anterior
          </button>
          <button
            disabled={!pregrados.next_page_url}
            onClick={() => goto(pregrados.next_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Siguiente
          </button>
        </div>
      </div>

      {/* Modal crear/editar */}
      <Modal show={showModal} onClose={() => setShowModal(false)} maxWidth="2xl">
        <form onSubmit={submit} className="p-6 space-y-4 overflow-y-auto max-h-[70vh]">
          <h2 className="mb-2 text-lg font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-200 dark:to-secondary-300">
            {editing ? "Editar carrera" : "Nueva carrera"}
          </h2>

          <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
            <div>
              <label className="block text-sm font-semibold">Nombre</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
                value={data.nombre}
                onChange={(e) => setData("nombre", e.target.value)}
              />
              {errors.nombre && <p className="text-sm text-red-500">{errors.nombre}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Duración</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
                value={data.duracion}
                onChange={(e) => setData("duracion", e.target.value)}
                placeholder="10 semestres"
              />
            </div>

            <div>
              <label className="block text-sm font-semibold">Modalidad</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
                value={data.modalidad}
                onChange={(e) => setData("modalidad", e.target.value)}
                placeholder="Presencial"
              />
            </div>

            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold">Título</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
                value={data.titulo}
                onChange={(e) => setData("titulo", e.target.value)}
              />
            </div>

            {/* Coordinador */}
            <div>
              <label className="block text-sm font-semibold">Coord. Nombre</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
                value={data.coordinador_nombre}
                onChange={(e) => setData("coordinador_nombre", e.target.value)}
              />
            </div>

            <div>
              <label className="block text-sm font-semibold">Coord. Email</label>
              <input
                type="email"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
                value={data.coordinador_email}
                onChange={(e) => setData("coordinador_email", e.target.value)}
              />
              {errors.coordinador_email && (
                <p className="text-sm text-red-500">{errors.coordinador_email}</p>
              )}
            </div>

            {/* Archivos con preview live */}
            <div className="sm:col-span-2 grid grid-cols-1 gap-4 md:grid-cols-2">
              {/* Foto coordinador */}
              <div>
                <label className="block text-sm font-semibold flex items-center gap-2">
                  <FiImage /> Foto del coordinador (JPG/PNG)
                </label>
                {(prevCoord || editing?.coordinador?.foto) && (
                  <img
                    src={prevCoord || storageUrl(editing?.coordinador?.foto)}
                    alt="Coordinador"
                    className="mt-2 h-20 w-20 rounded-xl object-cover border border-primary-200/60 dark:border-primary-800/60"
                  />
                )}
                <label className="mt-2 inline-flex cursor-pointer items-center gap-2 rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm font-semibold hover:bg-primary-50 dark:border-primary-800/60 dark:bg-primary-900/60">
                  Subir imagen
                  <input
                    type="file"
                    accept="image/*"
                    className="hidden"
                    onChange={(e) =>
                      onFilePreview(e.target.files?.[0] ?? null, setPrevCoord, "coordinador_foto")
                    }
                  />
                </label>
                {errors.coordinador_foto && <p className="text-sm text-red-500">{errors.coordinador_foto}</p>}
              </div>

              {/* Imagen hero */}
              <div>
                <label className="block text-sm font-semibold flex items-center gap-2">
                  <FiImage /> Imagen hero (JPG/PNG)
                </label>
                {(prevHero || editing?.imagen_hero) && (
                  <img
                    src={prevHero || storageUrl(editing?.imagen_hero)}
                    alt="Hero"
                    className="mt-2 h-20 w-32 rounded-xl object-cover border border-primary-200/60 dark:border-primary-800/60"
                  />
                )}
                <label className="mt-2 inline-flex cursor-pointer items-center gap-2 rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm font-semibold hover:bg-primary-50 dark:border-primary-800/60 dark:bg-primary-900/60">
                  Subir imagen
                  <input
                    type="file"
                    accept="image/*"
                    className="hidden"
                    onChange={(e) =>
                      onFilePreview(e.target.files?.[0] ?? null, setPrevHero, "imagen_hero")
                    }
                  />
                </label>
                {errors.imagen_hero && <p className="text-sm text-red-500">{errors.imagen_hero}</p>}
              </div>

              {/* Foto tarjeta */}
              <div>
                <label className="block text-sm font-semibold flex items-center gap-2">
                  <FiImage /> Foto tarjeta (JPG/PNG)
                </label>
                {(prevCard || editing?.foto_tarjeta) && (
                  <img
                    src={prevCard || storageUrl(editing?.foto_tarjeta)}
                    alt="Tarjeta"
                    className="mt-2 h-20 w-32 rounded-xl object-cover border border-primary-200/60 dark:border-primary-800/60"
                  />
                )}
                <label className="mt-2 inline-flex cursor-pointer items-center gap-2 rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm font-semibold hover:bg-primary-50 dark:border-primary-800/60 dark:bg-primary-900/60">
                  Subir imagen
                  <input
                    type="file"
                    accept="image/*"
                    className="hidden"
                    onChange={(e) =>
                      onFilePreview(e.target.files?.[0] ?? null, setPrevCard, "foto_tarjeta")
                    }
                  />
                </label>
                {errors.foto_tarjeta && <p className="text-sm text-red-500">{errors.foto_tarjeta}</p>}
              </div>

              {/* Malla PDF */}
              <div>
                <label className="block text-sm font-semibold">Malla (PDF)</label>
                {editing?.malla_url && (
                  <a
                    href={storageUrl(editing.malla_url)}
                    target="_blank"
                    rel="noreferrer"
                    className="mt-2 inline-block text-sm underline"
                  >
                    Ver PDF actual
                  </a>
                )}
                <input
                  type="file"
                  accept="application/pdf"
                  className="mt-2 block w-full text-sm"
                  onChange={(e) => setData("malla", e.target.files?.[0] ?? null)}
                />
                {errors.malla && <p className="text-sm text-red-500">{errors.malla}</p>}
              </div>
            </div>

            {/* Textos largos */}
            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold">Información</label>
              <textarea
                rows={3}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
                value={data.informacion}
                onChange={(e) => setData("informacion", e.target.value)}
              />
            </div>

            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold">Perfil</label>
              <textarea
                rows={3}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
                value={data.perfil}
                onChange={(e) => setData("perfil", e.target.value)}
              />
            </div>

            {/* Requisitos con chips */}
            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold">Requisitos</label>
              <div className="mt-1 flex gap-2">
                <input
                  type="text"
                  value={reqInput}
                  onChange={(e) => setReqInput(e.target.value)}
                  onKeyDown={(e) => e.key === "Enter" && (e.preventDefault(), addRequisito())}
                  placeholder="Ej: Título de bachiller"
                  className="flex-1 rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
                />
                <button
                  type="button"
                  onClick={addRequisito}
                  className="rounded-xl bg-primary-600 px-3 py-2 text-sm font-semibold text-white hover:bg-primary-700"
                >
                  <FiPlus className="inline-block" /> Agregar
                </button>
              </div>

              <div className="mt-2 flex flex-wrap gap-2">
                {data.requisitos?.length ? (
                  data.requisitos.map((r, idx) => (
                    <Chip key={`${r}-${idx}`} onRemove={() => removeRequisito(idx)}>
                      {r}
                    </Chip>
                  ))
                ) : (
                  <span className="text-xs text-gray-400 italic">Aún sin requisitos</span>
                )}
              </div>
              {errors.requisitos && <p className="text-sm text-red-500">{errors.requisitos}</p>}
            </div>

            {/* Orden + visible */}
            <div>
              <label className="block text-sm font-semibold">Orden</label>
              <input
                type="number"
                min="0"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 dark:text-white"
                value={data.orden}
                onChange={(e) => setData("orden", Number(e.target.value))}
              />
            </div>

            <div className="flex items-center gap-2 pt-6">
              <input
                id="is_visible"
                type="checkbox"
                checked={data.is_visible}
                onChange={(e) => setData("is_visible", e.target.checked)}
              />
              <label htmlFor="is_visible" className="text-sm">
                Visible en público
              </label>
            </div>
          </div>

          <div className="mt-4 flex justify-end gap-2">
            <button
              type="button"
              onClick={() => setShowModal(false)}
              className="rounded-xl border border-primary-200/60 px-4 py-2 text-sm dark:border-primary-800/60"
            >
              Cancelar
            </button>
            <button
              type="submit"
              disabled={processing}
              className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700 disabled:opacity-60"
            >
              {processing ? "Guardando..." : editing ? "Actualizar" : "Crear"}
            </button>
          </div>
        </form>
      </Modal>
    </DashboardLayout>
  );
}
