// resources/js/Pages/Admin/Postgrados/Index.jsx
import React, { useEffect, useState } from "react";
import { useForm, usePage, router } from "@inertiajs/react";
import DashboardLayout from "@/Layouts/AuthenticatedLayout";
import Modal from "@/Components/Modal";
import {
  FiPlus,
  FiEdit,
  FiTrash2,
  FiRefreshCw,
  FiEye,
  FiEyeOff,
  FiSearch,
  FiArrowUp,
  FiArrowDown,
  FiFileText,
  FiExternalLink,
  FiImage,
  FiTag,
} from "react-icons/fi";

export default function PostgradosIndex() {
  const { auth, postgrados, filtros, filters, modalidades, flash } = usePage().props;
  const f = filters ?? filtros ?? {};

  // Filtros/orden
  const [q, setQ] = useState(f.q ?? "");
  const [modalidad, setModalidad] = useState(f.modalidad ?? "");
  const [visible, setVisible] = useState(f.visible ?? "all");
  const [trash, setTrash] = useState(f.trash ?? "none");
  const [sort, setSort] = useState(f.sort ?? "orden");
  const [dir, setDir] = useState(f.dir ?? "asc");

  // UI
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState(null);

  // Form
  const { data, setData, reset, post, patch, processing, errors, clearErrors } = useForm({
    // base
    nombre: "",
    duracion: "",
    modalidad: "",
    titulo: "",
    // extras postgrado
    tipo: "",
    area: "",
    resolucion_senescyt: "",
    resumen: "",
    inscripciones_url: "",
    // coordinador
    coordinador_nombre: "",
    coordinador_email: "",
    // Archivos (File | null)
    coordinador_foto: null,
    imagen_hero: null,
    foto_tarjeta: null,
    malla: null,      // PDF
    brochure: null,   // PDF
    // Textos
    informacion: "",
    perfil: "",
    is_visible: true,
    orden: 0,
  });

  // Requisitos como chips
  const [requisitos, setRequisitos] = useState([]);
  const [reqInput, setReqInput] = useState("");

  // Previews de archivos
  const [coordPreview, setCoordPreview] = useState(null);
  const [heroPreview, setHeroPreview] = useState(null);
  const [cardPreview, setCardPreview] = useState(null);

  useEffect(() => {
    if (flash?.success) console.log("✅", flash.success);
    if (flash?.error) console.error("❌", flash.error);
  }, [flash]);

  const storageUrl = (path) => {
    if (!path) return null;
    return path.startsWith("http") ? path : `/storage/${path}`;
    // Ajusta si tu backend devuelve otro formato de ruta
  };

  // Filtros
  const applyFilters = () => {
    router.get(
      route("admin.postgrados.index"),
      { q, modalidad, visible, trash, sort, dir, page: 1 },
      { preserveScroll: true, replace: true }
    );
  };

  const toggleSort = (key) => {
    if (sort === key) {
      setDir(dir === "asc" ? "desc" : "asc");
    } else {
      setSort(key);
      setDir("asc");
    }
    setTimeout(applyFilters, 0);
  };

  // Helpers requisitos
  const addReq = () => {
    const val = (reqInput || "").trim();
    if (!val) return;
    if (!requisitos.includes(val)) setRequisitos((prev) => [...prev, val]);
    setReqInput("");
  };
  const removeReq = (idx) => {
    setRequisitos((prev) => prev.filter((_, i) => i !== idx));
  };
  const pasteReqs = (text) => {
    const parts = (text || "")
      .split(/\r?\n|,/)
      .map((s) => s.trim())
      .filter(Boolean);
    if (parts.length) {
      const next = [...requisitos];
      parts.forEach((p) => {
        if (!next.includes(p)) next.push(p);
      });
      setRequisitos(next);
    }
  };

  // Previews archivos
  const toPreview = (file, setter) => {
    if (!file) return setter(null);
    const reader = new FileReader();
    reader.onload = (ev) => setter(ev.target.result);
    reader.readAsDataURL(file);
  };

  // Abrir modal Crear
  const openCreate = () => {
    reset();
    clearErrors();
    setEditing(null);
    setData("is_visible", true);
    setData("orden", 0);
    setRequisitos([]);
    setReqInput("");
    setCoordPreview(null);
    setHeroPreview(null);
    setCardPreview(null);
    setShowModal(true);
  };

  // Abrir modal Editar
  const openEdit = (item) => {
    clearErrors();
    setEditing(item);
    setData({
      nombre: item.nombre || "",
      duracion: item.duracion || "",
      modalidad: item.modalidad || "",
      titulo: item.titulo || "",
      // extras
      tipo: item.tipo || "",
      area: item.area || "",
      resolucion_senescyt: item.resolucion_senescyt || "",
      resumen: item.resumen || "",
      inscripciones_url: item.inscripciones_url || "",
      // coordinador
      coordinador_nombre: item.coordinador?.nombre || "",
      coordinador_email: item.coordinador?.email || "",
      // archivos (nuevos null)
      coordinador_foto: null,
      imagen_hero: null,
      foto_tarjeta: null,
      malla: null,
      brochure: null,
      // textos
      informacion: item.informacion || "",
      perfil: item.perfil || "",
      is_visible: item.is_visible,
      orden: item.orden ?? 0,
    });
    setRequisitos(Array.isArray(item.requisitos) ? item.requisitos : []);
    setReqInput("");
    // previews iniciales solo de existentes del backend (si quieres verlos)
    setCoordPreview(item.coordinador?.foto ? storageUrl(item.coordinador.foto) : null);
    setHeroPreview(item.imagen_hero ? storageUrl(item.imagen_hero) : null);
    setCardPreview(item.foto_tarjeta ? storageUrl(item.foto_tarjeta) : null);
    setShowModal(true);
  };

  // Submit
  const submit = (e) => {
    e.preventDefault();
    const payload = { ...data, requisitos };

    if (editing) {
      patch(route("admin.postgrados.update", editing.id), {
        data: payload,
        forceFormData: true,
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    } else {
      post(route("admin.postgrados.store"), {
        data: payload,
        forceFormData: true,
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    }
  };

  // Acciones rápidas
  const toggleVisible = (id) =>
    router.patch(route("admin.postgrados.toggleVisible", id), {}, { preserveScroll: true });

  const destroy = (id) => {
    if (confirm("¿Enviar a papelera?")) {
      router.delete(route("admin.postgrados.destroy", id), { preserveScroll: true });
    }
  };

  const restore = (id) =>
    router.patch(route("admin.postgrados.restore", id), {}, { preserveScroll: true });

  const forceDelete = (id) => {
    if (confirm("⚠️ Eliminación definitiva. ¿Continuar?")) {
      router.delete(route("admin.postgrados.forceDestroy", id), { preserveScroll: true });
    }
  };

  const goto = (url) => url && router.visit(url, { preserveScroll: true, replace: true });

  return (
    <DashboardLayout auth={auth} titulo="Programas de Postgrado">
      {/* Filtros + Nuevo */}
      <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
        <div className="flex flex-col gap-2 sm:flex-row sm:items-center">
          <div className="flex items-center rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 dark:border-primary-800/60 dark:bg-primary-950/80">
            <FiSearch className="opacity-70" />
            <input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              onKeyDown={(e) => e.key === "Enter" && applyFilters()}
              placeholder="Buscar por nombre, título o coordinador…"
              className="ml-2 w-72 bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            />
          </div>

          <select
            value={modalidad}
            onChange={(e) => setModalidad(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Modalidad: Todas</option>
            {modalidades.map((m) => (
              <option key={m} value={m}>
                {m}
              </option>
            ))}
          </select>

          <select
            value={visible}
            onChange={(e) => setVisible(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="all">Visibilidad: Todas</option>
            <option value="1">Solo visibles</option>
            <option value="0">Solo ocultas</option>
          </select>

          <select
            value={trash}
            onChange={(e) => setTrash(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="none">Sin papelera</option>
            <option value="with">Incluir papelera</option>
            <option value="only">Solo papelera</option>
          </select>

          <button
            onClick={applyFilters}
            className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700"
          >
            Aplicar
          </button>
        </div>

        {trash !== "only" && (
          <button
            onClick={openCreate}
            className="flex items-center gap-2 rounded-xl bg-gradient-to-r from-primary-600 to-secondary-600 px-4 py-2 text-sm font-semibold text-white shadow hover:opacity-90"
          >
            <FiPlus className="text-lg" /> Nuevo
          </button>
        )}
      </div>

      {/* Tabla */}
      <div className="overflow-x-auto rounded-2xl border border-primary-200/60 bg-white/90 shadow-sm backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/80">
        <table className="min-w-full divide-y divide-primary-200 dark:divide-primary-800">
          <thead className="bg-primary-50 dark:bg-primary-900/40">
            <tr className="text-sm text-primary-700 dark:text-primary-200">
              <th className="px-4 py-2 text-left cursor-pointer" onClick={() => toggleSort("orden")}>
                Orden {sort === "orden" ? (dir === "asc" ? <FiArrowUp className="inline" /> : <FiArrowDown className="inline" />) : null}
              </th>
              <th className="px-4 py-2 text-left cursor-pointer" onClick={() => toggleSort("nombre")}>
                Nombre {sort === "nombre" ? (dir === "asc" ? <FiArrowUp className="inline" /> : <FiArrowDown className="inline" />) : null}
              </th>
              <th className="px-4 py-2 text-left">Tipo</th>
              <th className="px-4 py-2 text-left">Área</th>
              <th className="px-4 py-2 text-left">Modalidad</th>
              <th className="px-4 py-2 text-left">SENESCYT</th>
              <th className="px-4 py-2 text-center">Brochure</th>
              <th className="px-4 py-2 text-center">Visible</th>
              <th className="px-4 py-2 text-center">Acciones</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-primary-100 dark:divide-primary-800">
            {postgrados.data.map((p) => (
              <tr key={p.id} className="text-sm hover:bg-primary-50/60 dark:hover:bg-primary-900/30">
                <td className="px-4 py-2">{p.orden}</td>
                <td className="px-4 py-2 font-medium">{p.nombre}</td>
                <td className="px-4 py-2">{p.tipo || <span className="text-gray-400">—</span>}</td>
                <td className="px-4 py-2">{p.area || <span className="text-gray-400">—</span>}</td>
                <td className="px-4 py-2">{p.modalidad || <span className="text-gray-400">—</span>}</td>
                <td className="px-4 py-2">{p.resolucion_senescyt || <span className="text-gray-400">—</span>}</td>
                <td className="px-4 py-2 text-center">
                  {p.brochure_url ? (
                    <a
                      href={storageUrl(p.brochure_url)}
                      target="_blank"
                      rel="noreferrer"
                      className="inline-flex items-center gap-1 text-primary-600 underline dark:text-primary-300"
                      title="Ver brochure"
                    >
                      <FiFileText /> Brochure
                    </a>
                  ) : (
                    <span className="text-gray-400">—</span>
                  )}
                </td>
                <td className="px-4 py-2 text-center">
                  {p.deleted_at ? (
                    <span className="text-xs italic text-gray-400">Eliminado</span>
                  ) : (
                    <button
                      onClick={() => toggleVisible(p.id)}
                      className={`inline-flex items-center gap-1.5 rounded-lg px-3 py-1 text-xs font-semibold ${
                        p.is_visible
                          ? "bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-200"
                          : "bg-amber-100 text-amber-700 dark:bg-amber-900/30 dark:text-amber-200"
                      }`}
                    >
                      {p.is_visible ? <FiEye /> : <FiEyeOff />}
                      {p.is_visible ? "Visible" : "Oculto"}
                    </button>
                  )}
                </td>
                <td className="px-4 py-2">
                  {!p.deleted_at ? (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => openEdit(p)}
                        className="rounded-md p-1.5 text-primary-600 hover:bg-primary-100 dark:text-primary-300 dark:hover:bg-primary-900/40"
                        title="Editar"
                      >
                        <FiEdit />
                      </button>
                      <button
                        onClick={() => destroy(p.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar (papelera)"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  ) : (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => restore(p.id)}
                        className="rounded-md p-1.5 text-green-600 hover:bg-green-100 dark:text-green-300 dark:hover:bg-green-900/40"
                        title="Restaurar"
                      >
                        <FiRefreshCw />
                      </button>
                      <button
                        onClick={() => forceDelete(p.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar definitivo"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  )}
                </td>
              </tr>
            ))}
            {postgrados.data.length === 0 && (
              <tr>
                <td colSpan="9" className="px-4 py-6 text-center text-gray-400">
                  No hay programas de postgrado para mostrar.
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Paginación */}
      <div className="mt-4 flex items-center justify-between text-sm">
        <div className="text-primary-700/80 dark:text-primary-200/80">
          Mostrando {postgrados.from ?? 0}–{postgrados.to ?? 0} de {postgrados.total ?? 0}
        </div>
        <div className="flex gap-2">
          <button
            disabled={!postgrados.prev_page_url}
            onClick={() => goto(postgrados.prev_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Anterior
          </button>
          <button
            disabled={!postgrados.next_page_url}
            onClick={() => goto(postgrados.next_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Siguiente
          </button>
        </div>
      </div>

      {/* Modal crear/editar */}
      <Modal show={showModal} onClose={() => setShowModal(false)} maxWidth="2xl">
        <form onSubmit={submit} className="p-6 space-y-4 overflow-y-auto max-h-[70vh]">
          <h2 className="mb-2 text-lg font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-200 dark:to-secondary-300">
            {editing ? "Editar postgrado" : "Nuevo postgrado"}
          </h2>

          <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
            {/* Base */}
            <div>
              <label className="block text-sm font-semibold">Nombre</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.nombre}
                onChange={(e) => setData("nombre", e.target.value)}
              />
              {errors.nombre && <p className="text-sm text-red-500">{errors.nombre}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Duración</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.duracion}
                onChange={(e) => setData("duracion", e.target.value)}
                placeholder="3 semestres"
              />
            </div>

            <div>
              <label className="block text-sm font-semibold">Modalidad</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.modalidad}
                onChange={(e) => setData("modalidad", e.target.value)}
                placeholder="Presencial / Híbrida / En línea"
              />
            </div>

            <div>
              <label className="block text-sm font-semibold">Título</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.titulo}
                onChange={(e) => setData("titulo", e.target.value)}
              />
            </div>

            {/* Extras Postgrado */}
            <div>
              <label className="block text-sm font-semibold">Tipo</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.tipo}
                onChange={(e) => setData("tipo", e.target.value)}
                placeholder="Maestría profesional / Especialización / Doctorado"
              />
            </div>

            <div>
              <label className="block text-sm font-semibold">Área</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.area}
                onChange={(e) => setData("area", e.target.value)}
                placeholder="IoT / Seguridad / Telecom"
              />
            </div>

            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold">Resolución SENESCYT</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.resolucion_senescyt}
                onChange={(e) => setData("resolucion_senescyt", e.target.value)}
                placeholder="RPC-SO-XX-No.XXXX-AAAA"
              />
            </div>

            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold">Resumen</label>
              <textarea
                rows={3}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.resumen}
                onChange={(e) => setData("resumen", e.target.value)}
              />
            </div>

            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold">URL de Inscripciones</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.inscripciones_url}
                onChange={(e) => setData("inscripciones_url", e.target.value)}
                placeholder="/postgrado/mi-programa/inscripciones"
              />
            </div>

            {/* Coordinador */}
            <div>
              <label className="block text-sm font-semibold">Coord. Nombre</label>
              <input
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.coordinador_nombre}
                onChange={(e) => setData("coordinador_nombre", e.target.value)}
              />
            </div>

            <div>
              <label className="block text-sm font-semibold">Coord. Email</label>
              <input
                type="email"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.coordinador_email}
                onChange={(e) => setData("coordinador_email", e.target.value)}
              />
              {errors.coordinador_email && <p className="text-sm text-red-500">{errors.coordinador_email}</p>}
            </div>

            {/* Archivos con preview */}
            <div className="sm:col-span-2 grid grid-cols-1 gap-4 md:grid-cols-2">
              <div>
                <label className="block text-sm font-semibold flex items-center gap-2">
                  <FiImage /> Foto coordinador (JPG/PNG)
                </label>
                {(coordPreview) && (
                  <img
                    src={coordPreview}
                    alt="Coordinador"
                    className="mt-1 h-20 w-20 rounded object-cover border border-primary-200/60 dark:border-primary-800/60"
                  />
                )}
                {!coordPreview && editing?.coordinador?.foto && (
                  <img
                    src={storageUrl(editing.coordinador.foto)}
                    alt="Coordinador actual"
                    className="mt-1 h-20 w-20 rounded object-cover border border-primary-200/60 dark:border-primary-800/60"
                  />
                )}
                <input
                  type="file"
                  accept="image/*"
                  className="mt-2 block w-full text-sm"
                  onChange={(e) => {
                    const file = e.target.files?.[0] ?? null;
                    setData("coordinador_foto", file);
                    toPreview(file, setCoordPreview);
                  }}
                />
                {errors.coordinador_foto && <p className="text-sm text-red-500">{errors.coordinador_foto}</p>}
              </div>

              <div>
                <label className="block text-sm font-semibold flex items-center gap-2">
                  <FiImage /> Imagen hero (JPG/PNG)
                </label>
                {heroPreview && (
                  <img
                    src={heroPreview}
                    alt="Hero"
                    className="mt-1 h-20 w-32 rounded object-cover border border-primary-200/60 dark:border-primary-800/60"
                  />
                )}
                {!heroPreview && editing?.imagen_hero && (
                  <img
                    src={storageUrl(editing.imagen_hero)}
                    alt="Hero actual"
                    className="mt-1 h-20 w-32 rounded object-cover border border-primary-200/60 dark:border-primary-800/60"
                  />
                )}
                <input
                  type="file"
                  accept="image/*"
                  className="mt-2 block w-full text-sm"
                  onChange={(e) => {
                    const file = e.target.files?.[0] ?? null;
                    setData("imagen_hero", file);
                    toPreview(file, setHeroPreview);
                  }}
                />
                {errors.imagen_hero && <p className="text-sm text-red-500">{errors.imagen_hero}</p>}
              </div>

              <div>
                <label className="block text-sm font-semibold flex items-center gap-2">
                  <FiImage /> Foto tarjeta (JPG/PNG)
                </label>
                {cardPreview && (
                  <img
                    src={cardPreview}
                    alt="Tarjeta"
                    className="mt-1 h-20 w-32 rounded object-cover border border-primary-200/60 dark:border-primary-800/60"
                  />
                )}
                {!cardPreview && editing?.foto_tarjeta && (
                  <img
                    src={storageUrl(editing.foto_tarjeta)}
                    alt="Tarjeta actual"
                    className="mt-1 h-20 w-32 rounded object-cover border border-primary-200/60 dark:border-primary-800/60"
                  />
                )}
                <input
                  type="file"
                  accept="image/*"
                  className="mt-2 block w-full text-sm"
                  onChange={(e) => {
                    const file = e.target.files?.[0] ?? null;
                    setData("foto_tarjeta", file);
                    toPreview(file, setCardPreview);
                  }}
                />
                {errors.foto_tarjeta && <p className="text-sm text-red-500">{errors.foto_tarjeta}</p>}
              </div>

              <div>
                <label className="block text-sm font-semibold">Malla (PDF)</label>
                {editing?.malla_url && (
                  <a
                    href={storageUrl(editing.malla_url)}
                    target="_blank"
                    rel="noreferrer"
                    className="mt-1 inline-flex items-center gap-1 text-sm underline"
                  >
                    <FiExternalLink /> Ver PDF actual
                  </a>
                )}
                <input
                  type="file"
                  accept="application/pdf"
                  className="mt-2 block w-full text-sm"
                  onChange={(e) => setData("malla", e.target.files?.[0] ?? null)}
                />
                {errors.malla && <p className="text-sm text-red-500">{errors.malla}</p>}
              </div>

              <div>
                <label className="block text-sm font-semibold">Brochure (PDF)</label>
                {editing?.brochure_url && (
                  <a
                    href={storageUrl(editing.brochure_url)}
                    target="_blank"
                    rel="noreferrer"
                    className="mt-1 inline-flex items-center gap-1 text-sm underline"
                  >
                    <FiExternalLink /> Ver brochure
                  </a>
                )}
                <input
                  type="file"
                  accept="application/pdf"
                  className="mt-2 block w-full text-sm"
                  onChange={(e) => setData("brochure", e.target.files?.[0] ?? null)}
                />
                {errors.brochure && <p className="text-sm text-red-500">{errors.brochure}</p>}
              </div>
            </div>

            {/* Textos largos */}
            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold">Información</label>
              <textarea
                rows={3}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.informacion}
                onChange={(e) => setData("informacion", e.target.value)}
              />
            </div>

            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold">Perfil</label>
              <textarea
                rows={3}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.perfil}
                onChange={(e) => setData("perfil", e.target.value)}
              />
            </div>

            {/* Requisitos estilo chips */}
            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiTag /> Requisitos
              </label>

              <div className="mt-1 flex gap-2">
                <input
                  type="text"
                  value={reqInput}
                  onChange={(e) => setReqInput(e.target.value)}
                  onKeyDown={(e) => e.key === "Enter" && (e.preventDefault(), addReq())}
                  placeholder="Escribe un requisito y presiona Enter"
                  className="flex-1 rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                />
                <button
                  type="button"
                  onClick={addReq}
                  className="rounded-xl bg-primary-600 px-3 py-2 text-sm font-semibold text-white hover:bg-primary-700"
                >
                  Agregar
                </button>
              </div>

              {/* Pegar varios (coma o líneas) */}
              <div className="mt-2">
                <textarea
                  rows={2}
                  onPaste={(e) => {
                    const text = e.clipboardData.getData("text");
                    if (text && /[\n,]/.test(text)) {
                      e.preventDefault();
                      pasteReqs(text);
                    }
                  }}
                  placeholder="(Opcional) Pega varios separados por coma o salto de línea"
                  className="w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-xs dark:border-primary-800/60 dark:bg-primary-950/80"
                />
              </div>

              <div className="mt-2 flex flex-wrap gap-2">
                {requisitos.map((r, idx) => (
                  <span
                    key={`${r}-${idx}`}
                    className="group inline-flex items-center gap-2 rounded-xl border border-primary-200/60 bg-primary-50/70 px-3 py-1 text-xs font-medium dark:border-primary-800/60 dark:bg-primary-900/40"
                  >
                    <FiTag />
                    {r}
                    <button
                      type="button"
                      onClick={() => removeReq(idx)}
                      className="rounded-md px-1 text-xs opacity-70 hover:bg-primary-100 dark:hover:bg-primary-900/60"
                      title="Quitar"
                    >
                      ×
                    </button>
                  </span>
                ))}
                {requisitos.length === 0 && (
                  <span className="text-xs text-gray-400 italic">Aún sin requisitos</span>
                )}
              </div>

              {errors.requisitos && <p className="text-sm text-red-500">{errors.requisitos}</p>}
            </div>

            {/* Orden + visible */}
            <div>
              <label className="block text-sm font-semibold">Orden</label>
              <input
                type="number"
                min="0"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.orden}
                onChange={(e) => setData("orden", Number(e.target.value))}
              />
            </div>

            <div className="flex items-center gap-2 pt-6">
              <input
                id="is_visible"
                type="checkbox"
                checked={data.is_visible}
                onChange={(e) => setData("is_visible", e.target.checked)}
              />
              <label htmlFor="is_visible" className="text-sm">Visible en público</label>
            </div>
          </div>

          <div className="mt-4 flex justify-end gap-2">
            <button
              type="button"
              onClick={() => setShowModal(false)}
              className="rounded-xl border border-primary-200/60 px-4 py-2 text-sm dark:border-primary-800/60"
            >
              Cancelar
            </button>
            <button
              type="submit"
              disabled={processing}
              className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700 disabled:opacity-60"
            >
              {processing ? "Guardando..." : editing ? "Actualizar" : "Crear"}
            </button>
          </div>
        </form>
      </Modal>
    </DashboardLayout>
  );
}
