// resources/js/Pages/Admin/Pasantias/Index.jsx
import React, { useMemo, useState } from "react";
import { Head, router, usePage } from "@inertiajs/react";
import {
  FiPlus, FiSearch, FiEdit2, FiTrash2, FiToggleLeft, FiToggleRight, FiTag,
} from "react-icons/fi";
import AuthenticatedLayout from "@/Layouts/AuthenticatedLayout";
import Modal from "@/Components/Modal";

const MODALIDADES = ["Todas", "Remoto", "Presencial", "Híbrido"];
const ESTADOS = ["Todas", "borrador", "publicado"];

function Field({ label, children }) {
  return (
    <label className="block">
      <div className="text-xs font-semibold text-primary-700 mb-1">{label}</div>
      {children}
    </label>
  );
}

function PasantiaForm({ initial = {}, onClose }) {
  // base form
  const [data, setData] = useState({
    titulo: initial.titulo ?? "",
    empresa: initial.empresa ?? "",
    modalidad: initial.modalidad ?? "Remoto",
    duracion: initial.duracion ?? "",
    carrera: initial.carrera ?? "",
    tipo: initial.tipo ?? "Pasantía",
    fecha_inicio: initial.fecha_inicio ?? "",
    fecha_maxima_postulacion: initial.fecha_maxima_postulacion ?? "",
    remunerado: initial.remunerado ?? false,
    monto_remuneracion: initial.monto_remuneracion ?? "",
    cupos: initial.cupos ?? 1,
    estado: initial.estado ?? "borrador",
  });

  // chips: requisitos y beneficios
  const [requisitos, setRequisitos] = useState(Array.isArray(initial.requisitos) ? initial.requisitos : []);
  const [beneficios, setBeneficios] = useState(Array.isArray(initial.beneficios) ? initial.beneficios : []);
  const [reqInput, setReqInput] = useState("");
  const [benInput, setBenInput] = useState("");

  const isEdit = Boolean(initial.id);

  // helpers chips
  const addChip = (value, list, setter) => {
    const v = (value || "").trim();
    if (!v) return;
    if (!list.includes(v)) setter([...list, v]);
  };
  const removeChip = (idx, list, setter) => {
    setter(list.filter((_, i) => i !== idx));
  };
  const pasteMany = (text, list, setter) => {
    const parts = (text || "")
      .split(/\r?\n|,/)
      .map(s => s.trim())
      .filter(Boolean);
    if (!parts.length) return;
    const next = [...list];
    parts.forEach(p => { if (!next.includes(p)) next.push(p); });
    setter(next);
  };

  const submit = (e) => {
    e.preventDefault();
    const payload = {
      ...data,
      requisitos,
      beneficios,
    };
    router[isEdit ? "put" : "post"](
      route(isEdit ? "admin.pasantias.update" : "admin.pasantias.store", isEdit ? initial.id : undefined),
      {
        data: payload,
        preserveScroll: true,
        onSuccess: onClose,
      }
    );
  };

  return (
    <form onSubmit={submit} className="space-y-4">
      <div className="grid md:grid-cols-2 gap-4">
        <Field label="Título">
          <input
          className="ml-2 w-full bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            value={data.titulo}
            onChange={e=>setData({ ...data, titulo: e.target.value })}
          />
        </Field>
        <Field label="Empresa">
          <input
                      className="ml-2 w-full bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            value={data.empresa}
            onChange={e=>setData({ ...data, empresa: e.target.value })}
          />
        </Field>
        <Field label="Modalidad">
          <select
                      className="ml-2 w-full bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            value={data.modalidad}
            onChange={e=>setData({ ...data, modalidad: e.target.value })}
          >
            {MODALIDADES.filter(m=>m!=="Todas").map(m => (<option key={m}>{m}</option>))}
          </select>
        </Field>
        <Field label="Duración (ej. 6 meses)">
          <input
                      className="ml-2 w-full bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            value={data.duracion}
            onChange={e=>setData({ ...data, duracion: e.target.value })}
          />
        </Field>
        <Field label="Carrera">
          <input
                      className="ml-2 w-full bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            value={data.carrera}
            onChange={e=>setData({ ...data, carrera: e.target.value })}
          />
        </Field>
        <Field label="Tipo">
          <input
                      className="ml-2 w-full bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            value={data.tipo}
            onChange={e=>setData({ ...data, tipo: e.target.value })}
          />
        </Field>
        <Field label="Fecha inicio">
          <input
            type="date"
                      className="ml-2 w-full bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            value={data.fecha_inicio ?? ""}
            onChange={e=>setData({ ...data, fecha_inicio: e.target.value })}
          />
        </Field>
        <Field label="Fecha máxima postulación">
          <input
            type="date"
                      className="ml-2 w-full bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            value={data.fecha_maxima_postulacion ?? ""}
            onChange={e=>setData({ ...data, fecha_maxima_postulacion: e.target.value })}
          />
        </Field>
        <Field label="Remunerado">
          <select
                      className="ml-2 w-full bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            value={String(data.remunerado)}
            onChange={e=>setData({ ...data, remunerado: e.target.value === "true" })}
          >
            <option value="false">No</option>
            <option value="true">Sí</option>
          </select>
        </Field>
        <Field label="Monto remuneración (USD)">
          <input
                      className="ml-2 w-full bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            type="number" min="0"
            value={data.monto_remuneracion ?? ""}
            onChange={e=>setData({ ...data, monto_remuneracion: e.target.value })}
          />
        </Field>
        <Field label="Cupos">
          <input
                      className="ml-2 w-full bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            type="number" min="1"
            value={data.cupos}
            onChange={e=>setData({ ...data, cupos: Number(e.target.value) })}
          />
        </Field>
        <Field label="Estado">
          <select
                      className="ml-2 w-full bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            value={data.estado}
            onChange={e=>setData({ ...data, estado: e.target.value })}
          >
            <option value="borrador">borrador</option>
            <option value="publicado">publicado</option>
          </select>
        </Field>
      </div>

      {/* Requisitos y Beneficios como chips */}
      <div className="grid md:grid-cols-2 gap-4">
        {/* Requisitos */}
        <div>
          <div className="text-xs font-semibold text-primary-700 mb-1">Requisitos</div>

          <div className="flex gap-2">
            <input
              className="flex-1 rounded-lg border-2 border-primary-200 p-2"
              placeholder="Escribe un requisito y Enter"
              value={reqInput}
              onChange={(e)=>setReqInput(e.target.value)}
              onKeyDown={(e)=> e.key === "Enter" && (e.preventDefault(), addChip(reqInput, requisitos, setRequisitos), setReqInput(""))}
            />
            <button
              type="button"
              className="px-3 py-2 rounded-lg text-white bg-gradient-to-r from-primary-600 to-primary-700"
              onClick={()=> { addChip(reqInput, requisitos, setRequisitos); setReqInput(""); }}
            >
              Agregar
            </button>
          </div>

          <textarea
            rows={2}
            className="mt-2 w-full rounded-lg border-2 border-primary-200 p-2 text-xs"
            placeholder="(Opcional) Pega varios separados por coma o salto de línea"
            onPaste={(e) => {
              const text = e.clipboardData.getData("text");
              if (text && /[\n,]/.test(text)) {
                e.preventDefault();
                pasteMany(text, requisitos, setRequisitos);
              }
            }}
          />

          <div className="mt-2 flex flex-wrap gap-2">
            {requisitos.map((r, idx) => (
              <span
                key={`${r}-${idx}`}
                className="group inline-flex items-center gap-2 rounded-xl border-2 border-primary-200 bg-primary-50/70 px-3 py-1 text-xs font-medium"
              >
                <FiTag /> {r}
                <button
                  type="button"
                  onClick={()=> removeChip(idx, requisitos, setRequisitos)}
                  className="rounded-md px-1 text-xs opacity-70 hover:bg-primary-100"
                  title="Quitar"
                >
                  ×
                </button>
              </span>
            ))}
            {requisitos.length === 0 && (
              <span className="text-xs text-gray-400 italic">Aún sin requisitos</span>
            )}
          </div>
        </div>

        {/* Beneficios */}
        <div>
          <div className="text-xs font-semibold text-primary-700 mb-1">Beneficios</div>

          <div className="flex gap-2">
            <input
              className="flex-1 rounded-lg border-2 border-primary-200 p-2"
              placeholder="Escribe un beneficio y Enter"
              value={benInput}
              onChange={(e)=>setBenInput(e.target.value)}
              onKeyDown={(e)=> e.key === "Enter" && (e.preventDefault(), addChip(benInput, beneficios, setBeneficios), setBenInput(""))}
            />
            <button
              type="button"
              className="px-3 py-2 rounded-lg text-white bg-gradient-to-r from-primary-600 to-primary-700"
              onClick={()=> { addChip(benInput, beneficios, setBeneficios); setBenInput(""); }}
            >
              Agregar
            </button>
          </div>

          <textarea
            rows={2}
            className="mt-2 w-full rounded-lg border-2 border-primary-200 p-2 text-xs"
            placeholder="(Opcional) Pega varios separados por coma o salto de línea"
            onPaste={(e) => {
              const text = e.clipboardData.getData("text");
              if (text && /[\n,]/.test(text)) {
                e.preventDefault();
                pasteMany(text, beneficios, setBeneficios);
              }
            }}
          />

          <div className="mt-2 flex flex-wrap gap-2">
            {beneficios.map((b, idx) => (
              <span
                key={`${b}-${idx}`}
                className="group inline-flex items-center gap-2 rounded-xl border-2 border-primary-200 bg-primary-50/70 px-3 py-1 text-xs font-medium"
              >
                <FiTag /> {b}
                <button
                  type="button"
                  onClick={()=> removeChip(idx, beneficios, setBeneficios)}
                  className="rounded-md px-1 text-xs opacity-70 hover:bg-primary-100"
                  title="Quitar"
                >
                  ×
                </button>
              </span>
            ))}
            {beneficios.length === 0 && (
              <span className="text-xs text-gray-400 italic">Aún sin beneficios</span>
            )}
          </div>
        </div>
      </div>

      <div className="flex gap-2 justify-end">
        <button type="button" onClick={onClose}
                className="px-4 py-2 rounded-lg border-2 border-primary-200">Cancelar</button>
        <button type="submit"
                className="px-4 py-2 rounded-lg text-white bg-gradient-to-r from-primary-600 to-primary-700">
          {isEdit ? "Guardar cambios" : "Crear"}
        </button>
      </div>
    </form>
  );
}

export default function Index({auth, pasantias, filters}) {
  const { flash } = usePage().props;
  const [showForm, setShowForm] = useState(false);
  const [editing, setEditing] = useState(null);
  const [f, setF] = useState({
    s: filters.s ?? "",
    modalidad: filters.modalidad ?? "Todas",
    carrera: filters.carrera ?? "Todas",
    estado: filters.estado ?? "Todas",
  });

  const submitFilters = () => {
    router.get(route("admin.pasantias.index"), f, { preserveState: true, preserveScroll: true });
  };

  const toggleEstado = (id) => {
    router.patch(route("admin.pasantias.toggle", id), {}, { preserveScroll: true });
  };

  const eliminar = (id) => {
    if (!confirm("¿Eliminar pasantía?")) return;
    router.delete(route("admin.pasantias.destroy", id), { preserveScroll: true });
  };

  return (
    <AuthenticatedLayout auth={auth} title="Pasantías">
      <Head title="Pasantías" />

      <div className="p-4 md:p-6">
        {/* Header acciones */}
        <div className="flex items-center justify-between mb-4">
          <h1 className="text-2xl font-bold">Pasantías</h1>
          <button
            onClick={() => { setEditing(null); setShowForm(true); }}
            className="inline-flex items-center gap-2 px-4 py-2 rounded-lg text-white bg-gradient-to-r from-primary-600 to-primary-700"
          >
            <FiPlus /> Nueva
          </button>
        </div>

        {/* Filtros */}
        <div className="grid md:grid-cols-4 gap-3 mb-4">
          <div className="col-span-2 flex">
            <input
              className="flex-1 rounded-l-lg border-2 border-primary-200 p-2"
              placeholder="Buscar por título, empresa o carrera"
              value={f.s}
              onChange={e=>setF({...f, s:e.target.value})}
            />
            <button
              onClick={submitFilters}
              className="rounded-r-lg px-3 border-2 border-l-0 border-primary-200"
            >
              <FiSearch />
            </button>
          </div>
          <select
            className="rounded-lg border-2 border-primary-200 p-2"
            value={f.modalidad}
            onChange={e=>setF({...f, modalidad:e.target.value})}
          >
            {MODALIDADES.map(m=>(<option key={m}>{m}</option>))}
          </select>
          <select
            className="rounded-lg border-2 border-primary-200 p-2"
            value={f.estado}
            onChange={e=>setF({...f, estado:e.target.value})}
          >
            {ESTADOS.map(m=>(<option key={m}>{m}</option>))}
          </select>
        </div>

        {/* Tabla */}
        <div className="overflow-x-auto bg-white rounded-xl border-2 border-primary-100">
          <table className="min-w-full text-sm">
            <thead>
              <tr className="text-left bg-primary-50">
                <th className="p-3">Título</th>
                <th className="p-3">Empresa</th>
                <th className="p-3">Modalidad</th>
                <th className="p-3">Carrera</th>
                <th className="p-3">Cupos</th>
                <th className="p-3">Estado</th>
                <th className="p-3 text-right">Acciones</th>
              </tr>
            </thead>
            <tbody>
              {pasantias.data.map(row => (
                <tr key={row.id} className="border-t border-primary-100">
                  <td className="p-3 font-semibold">{row.titulo}</td>
                  <td className="p-3">{row.empresa}</td>
                  <td className="p-3">{row.modalidad}</td>
                  <td className="p-3">{row.carrera}</td>
                  <td className="p-3">{row.cupos}</td>
                  <td className="p-3">
                    <span className={`px-2 py-1 rounded-lg text-xs font-bold ${row.estado === 'publicado'
                      ? 'bg-primary-100 text-primary-700'
                      : 'bg-primary-50 text-primary-700'}`}>
                      {row.estado}
                    </span>
                  </td>
                  <td className="p-3">
                    <div className="flex justify-end gap-2">
                      <button onClick={() => toggleEstado(row.id)} className="p-2 rounded-lg border">
                        {row.estado === 'publicado' ? <FiToggleRight/> : <FiToggleLeft/>}
                      </button>
                      <button onClick={() => { setEditing(row); setShowForm(true); }} className="p-2 rounded-lg border">
                        <FiEdit2/>
                      </button>
                      <button onClick={() => eliminar(row.id)} className="p-2 rounded-lg border text-red-600">
                        <FiTrash2/>
                      </button>
                    </div>
                  </td>
                </tr>
              ))}
              {pasantias.data.length === 0 && (
                <tr><td className="p-6 text-center text-primary-700" colSpan={7}>Sin resultados</td></tr>
              )}
            </tbody>
          </table>
        </div>

        {/* Paginación simple */}
        <div className="flex justify-end gap-2 mt-3">
          {pasantias.links.map((l, i) => (
            <button key={i} disabled={!l.url}
                    onClick={()=> l.url && router.get(l.url, {}, { preserveScroll:true })}
                    className={`px-3 py-1 rounded border ${l.active ? 'bg-primary-600 text-white' : ''}`}>
              <span dangerouslySetInnerHTML={{__html: l.label}} />
            </button>
          ))}
        </div>
      </div>

      {/* Modal unificado */}
      <Modal show={showForm} onClose={() => setShowForm(false)} maxWidth="2xl">
        <div className="p-6">
          <div className="mb-4 flex items-center justify-between">
            <h2 className="text-lg font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent">
              {editing ? "Editar pasantía" : "Nueva pasantía"}
            </h2>
            <button
              onClick={() => setShowForm(false)}
              className="rounded-lg border px-3 py-1 text-sm"
            >
              Cerrar
            </button>
          </div>

          <PasantiaForm initial={editing ?? {}} onClose={() => setShowForm(false)} />
        </div>
      </Modal>
    </AuthenticatedLayout>
  );
}
