import React, { useEffect, useState } from "react";
import { useForm, router } from "@inertiajs/react";
import DashboardLayout from "@/Layouts/AuthenticatedLayout";
import Modal from "@/Components/Modal";
import {
  FiEdit,
  FiPlus,
  FiTrash2,
  FiRefreshCw,
  FiSearch,
  FiCalendar,
  FiHash,
  FiX,
  FiTag,
  FiUser,
  FiCode,
  FiCheckCircle,
  FiMinusCircle,
} from "react-icons/fi";

/**
 * props esperadas desde el controlador:
 * { auth, lineas, filters, tipos, years, flash }
 * - lineas: paginator con data: [ { id, uid, slug, nombre, profesor, codigo, fecha, tipo, descripcion, activo, deleted_at } ]
 * - filters: { q, tipo, year, year_from, year_to, activo, trash, perPage, sort, dir }
 * - tipos: [ 'Prioritaria', 'Linea' ]
 * - years: [ '2025','2024', ... ]
 */

export default function LineasIndex({ auth, lineas, filters, tipos, years, flash }) {
  // UI state
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState(null);

  // Filtros locales
  const [q, setQ] = useState(filters?.q ?? "");
  const [tipo, setTipo] = useState(filters?.tipo ?? "");
  const [year, setYear] = useState(filters?.year ?? "");
  const [yearFrom, setYearFrom] = useState(filters?.year_from ?? "");
  const [yearTo, setYearTo] = useState(filters?.year_to ?? "");
  const [activo, setActivo] = useState(filters?.activo ?? ""); // '', '1', '0'
  const [trash, setTrash] = useState(filters?.trash ?? "none");
  const [sort, setSort] = useState(filters?.sort ?? "nombre");
  const [dir, setDir] = useState(filters?.dir ?? "asc");

  // Form (crear/editar)
  const {
    data,
    setData,
    reset,
    post,
    patch,
    processing,
    errors,
    clearErrors,
  } = useForm({
    uid: "",
    slug: "",
    nombre: "",
    profesor: "",
    codigo: "",
    fecha: "",
    tipo: tipos?.[0] ?? "Linea",
    descripcion: "",
    activo: true,
  });

  useEffect(() => {
    if (flash?.success) console.log("✅", flash.success);
    if (flash?.error) console.error("❌", flash.error);
  }, [flash]);

  // Modal: Crear
  const openCreate = () => {
    reset();
    setEditing(null);
    clearErrors();
    setData({
      uid: "",
      slug: "",
      nombre: "",
      profesor: "",
      codigo: "",
      fecha: "",
      tipo: tipos?.[0] ?? "Linea",
      descripcion: "",
      activo: true,
    });
    setShowModal(true);
  };

  // Modal: Editar
  const openEdit = (l) => {
    setEditing(l);
    clearErrors();
    setData({
      uid: l.uid ?? "",
      slug: l.slug ?? "",
      nombre: l.nombre ?? "",
      profesor: l.profesor ?? "",
      codigo: l.codigo ?? "",
      fecha: l.fecha ?? "",
      tipo: l.tipo ?? (tipos?.[0] ?? "Linea"),
      descripcion: l.descripcion ?? "",
      activo: !!l.activo,
    });
    setShowModal(true);
  };

  // Submit modal
  const submit = (e) => {
    e.preventDefault();
    if (editing) {
      patch(route("admin.lineas.update", editing.id), {
        data,
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    } else {
      post(route("admin.lineas_investigacion.store"), {
        data,
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    }
  };

  // Acciones
  const destroy = (id) => {
    if (confirm("¿Enviar esta línea a la papelera?")) {
      router.delete(route("admin.lineas.destroy", id), { preserveScroll: true });
    }
  };
  const restore = (id) =>
    router.patch(route("admin.lineas.restore", id), {}, { preserveScroll: true });
  const forceDelete = (id) => {
    if (confirm("⚠️ Esto elimina definitivamente. ¿Continuar?")) {
      router.delete(route("admin.lineas.force", id), { preserveScroll: true });
    }
  };

  // Filtro/búsqueda
  const applyFilters = () => {
    router.get(
      route("admin.lineas.index"),
      {
        q,
        tipo,
        year,
        year_from: yearFrom,
        year_to: yearTo,
        activo,
        trash,
        sort,
        dir,
        page: 1,
      },
      { preserveScroll: true, replace: true }
    );
  };

  // Paginación
  const goto = (url) => {
    if (!url) return;
    router.visit(url, { preserveScroll: true, replace: true });
  };

  // Helper UI
  const ActiveBadge = ({ v }) =>
    v ? (
      <span className="inline-flex items-center gap-1 rounded-lg bg-green-100 px-2 py-0.5 text-xs font-semibold text-green-700 dark:bg-green-900/30 dark:text-green-300">
        <FiCheckCircle /> Activa
      </span>
    ) : (
      <span className="inline-flex items-center gap-1 rounded-lg bg-red-100 px-2 py-0.5 text-xs font-semibold text-red-700 dark:bg-red-900/30 dark:text-red-300">
        <FiMinusCircle /> Inactiva
      </span>
    );

  return (
    <DashboardLayout auth={auth} titulo="Gestión de Líneas de Investigación">
      {/* Barra superior filtros + NUEVO */}
      <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
        {/* Filtros */}
        <div className="flex flex-col gap-2 sm:flex-row sm:items-center sm:flex-wrap">
          <div className="flex items-center rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 dark:border-primary-800/60 dark:bg-primary-950/80">
            <FiSearch className="opacity-70" />
            <input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              onKeyDown={(e) => e.key === "Enter" && applyFilters()}
              placeholder="Buscar por nombre, uid, profesor, código…"
              className="ml-2 w-72 bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            />
          </div>

          <select
            value={tipo}
            onChange={(e) => setTipo(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Tipo: Todos</option>
            {tipos?.map((t) => (
              <option key={t} value={t}>{t}</option>
            ))}
          </select>

          <select
            value={year}
            onChange={(e) => setYear(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Año exacto</option>
            {years?.map((y) => (
              <option key={y} value={y}>{y}</option>
            ))}
          </select>

          <div className="flex items-center gap-2">
            <select
              value={yearFrom}
              onChange={(e) => setYearFrom(e.target.value)}
              className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
            >
              <option value="">Desde</option>
              {years?.map((y) => (
                <option key={y} value={y}>{y}</option>
              ))}
            </select>
            <select
              value={yearTo}
              onChange={(e) => setYearTo(e.target.value)}
              className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
            >
              <option value="">Hasta</option>
              {years?.map((y) => (
                <option key={y} value={y}>{y}</option>
              ))}
            </select>
          </div>

          <select
            value={activo}
            onChange={(e) => setActivo(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Estado: Todos</option>
            <option value="1">Activas</option>
            <option value="0">Inactivas</option>
          </select>

          <select
            value={trash}
            onChange={(e) => setTrash(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="none">Sin papelera</option>
            <option value="with">Incluir papelera</option>
            <option value="only">Solo papelera</option>
          </select>

          <select
            value={sort}
            onChange={(e) => setSort(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="nombre">Orden: Nombre</option>
            <option value="profesor">Orden: Profesor</option>
            <option value="fecha">Orden: Fecha</option>
            <option value="tipo">Orden: Tipo</option>
            <option value="created_at">Orden: Creación</option>
          </select>

          <select
            value={dir}
            onChange={(e) => setDir(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="asc">Ascendente</option>
            <option value="desc">Descendente</option>
          </select>

          <button
            onClick={applyFilters}
            className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700"
          >
            Aplicar
          </button>
        </div>

        {/* NUEVO */}
        {trash !== "only" && (
          <button
            onClick={openCreate}
            className="flex items-center justify-center gap-2 rounded-xl bg-gradient-to-r from-primary-600 to-secondary-600 px-4 py-2 text-sm font-semibold text-white shadow hover:opacity-90"
          >
            <FiPlus className="text-lg" /> Nuevo
          </button>
        )}
      </div>

      {/* Tabla */}
      <div className="overflow-x-auto rounded-2xl border border-primary-200/60 bg-white/90 shadow-sm backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/80">
        <table className="min-w-full divide-y divide-primary-200 dark:divide-primary-800">
          <thead className="bg-primary-50 dark:bg-primary-900/40">
            <tr className="text-sm text-primary-700 dark:text-primary-200">
              <th className="px-4 py-2 text-left">UID</th>
              <th className="px-4 py-2 text-left">Slug</th>
              <th className="px-4 py-2 text-left">Nombre</th>
              <th className="px-4 py-2 text-left">Profesor</th>
              <th className="px-4 py-2 text-left">Código</th>
              <th className="px-4 py-2 text-left">Fecha</th>
              <th className="px-4 py-2 text-left">Tipo</th>
              <th className="px-4 py-2 text-center">Estado</th>
              <th className="px-4 py-2 text-center">Acciones</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-primary-100 dark:divide-primary-800">
            {lineas.data.map((l) => (
              <tr key={l.id} className="text-sm hover:bg-primary-50/60 dark:hover:bg-primary-900/30">
                <td className="px-4 py-2 font-mono text-xs flex items-center gap-2">
                  <FiHash className="opacity-60" />
                  {l.uid}
                </td>
                <td className="px-4 py-2 font-mono text-xs">{l.slug}</td>
                <td className="px-4 py-2 font-medium">{l.nombre}</td>
                <td className="px-4 py-2  items-center gap-2">
                    <div className="flex">

                  <FiUser className="opacity-60" />
                  <span>{l.profesor || <span className="text-xs text-gray-400 italic">—</span>}</span>
                    </div>
                </td>
     <td className="px-4 py-2  items-center gap-2">
        <div className="flex">

                  <FiCode className="opacity-60" />
                  <span className="font-mono text-xs">{l.codigo}</span>
        </div>
                </td>
                <td className="px-4 py-2 flex items-center gap-2">
                  <FiCalendar className="opacity-60" />
                  <span>{l.fecha || <span className="text-xs text-gray-400 italic">—</span>}</span>
                </td>
                <td className="px-4 py-2">
                  <span className="inline-flex items-center gap-2 rounded-xl border border-primary-200/60 bg-primary-50/70 px-3 py-1 text-xs font-medium dark:border-primary-800/60 dark:bg-primary-900/40">
                    <FiTag /> {l.tipo}
                  </span>
                </td>
                <td className="px-4 py-2 text-center">
                  <ActiveBadge v={!!l.activo} />
                </td>
                <td className="px-4 py-2">
                  {!l.deleted_at ? (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => openEdit(l)}
                        className="rounded-md p-1.5 text-primary-600 hover:bg-primary-100 dark:text-primary-300 dark:hover:bg-primary-900/40"
                        title="Editar"
                      >
                        <FiEdit />
                      </button>
                      <button
                        onClick={() => destroy(l.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar (papelera)"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  ) : (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => restore(l.id)}
                        className="rounded-md p-1.5 text-green-600 hover:bg-green-100 dark:text-green-300 dark:hover:bg-green-900/40"
                        title="Restaurar"
                      >
                        <FiRefreshCw />
                      </button>
                      <button
                        onClick={() => forceDelete(l.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar definitivo"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  )}
                </td>
              </tr>
            ))}
            {lineas.data.length === 0 && (
              <tr>
                <td colSpan="9" className="px-4 py-6 text-center text-gray-400">
                  No hay líneas de investigación para mostrar.
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Paginación */}
      <div className="mt-4 flex items-center justify-between text-sm">
        <div className="text-primary-700/80 dark:text-primary-200/80">
          Mostrando {lineas.from ?? 0}–{lineas.to ?? 0} de {lineas.total ?? 0}
        </div>
        <div className="flex gap-2">
          <button
            disabled={!lineas.prev_page_url}
            onClick={() => goto(lineas.prev_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Anterior
          </button>
          <button
            disabled={!lineas.next_page_url}
            onClick={() => goto(lineas.next_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Siguiente
          </button>
        </div>
      </div>

      {/* Modal Crear/Editar */}
      <Modal show={showModal} onClose={() => setShowModal(false)} maxWidth="3xl">
        <form onSubmit={submit} className="p-6 space-y-5">
          <h2 className="mb-2 text-lg font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-200 dark:to-secondary-300">
            {editing ? "Editar línea de investigación" : "Nueva línea de investigación"}
          </h2>

          {/* Básicos */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiHash /> UID
              </label>
              <input
                type="text"
                placeholder="LPI-DETRI-2023-04"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 font-mono"
                value={data.uid}
                onChange={(e) => setData("uid", e.target.value)}
              />
              {errors.uid && <p className="text-sm text-red-500">{errors.uid}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiHash /> Slug
              </label>
              <input
                type="text"
                placeholder="sistemas-de-internet-de-las-cosas"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 font-mono"
                value={data.slug}
                onChange={(e) => setData("slug", e.target.value)}
              />
              {errors.slug && <p className="text-sm text-red-500">{errors.slug}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Nombre</label>
              <input
                type="text"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.nombre}
                onChange={(e) => setData("nombre", e.target.value)}
              />
              {errors.nombre && <p className="text-sm text-red-500">{errors.nombre}</p>}
            </div>
          </div>

          {/* Profesor, Código, Fecha */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiUser /> Profesor
              </label>
              <input
                type="text"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.profesor}
                onChange={(e) => setData("profesor", e.target.value)}
              />
              {errors.profesor && <p className="text-sm text-red-500">{errors.profesor}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiCode /> Código
              </label>
              <input
                type="text"
                placeholder="RCIIV-105-2023"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 font-mono"
                value={data.codigo}
                onChange={(e) => setData("codigo", e.target.value)}
              />
              {errors.codigo && <p className="text-sm text-red-500">{errors.codigo}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiCalendar /> Fecha
              </label>
              <input
                type="date"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.fecha}
                onChange={(e) => setData("fecha", e.target.value)}
              />
              {errors.fecha && <p className="text-sm text-red-500">{errors.fecha}</p>}
            </div>
          </div>

          {/* Tipo, Activo */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div className="sm:col-span-2">
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiTag /> Tipo
              </label>
              <select
                value={data.tipo}
                onChange={(e) => setData("tipo", e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              >
                {tipos?.map((t) => (
                  <option key={t} value={t}>{t}</option>
                ))}
              </select>
              {errors.tipo && <p className="text-sm text-red-500">{errors.tipo}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Estado</label>
              <div className="mt-1 flex items-center gap-2">
                <input
                  id="activo"
                  type="checkbox"
                  className="h-4 w-4"
                  checked={!!data.activo}
                  onChange={(e) => setData("activo", e.target.checked)}
                />
                <label htmlFor="activo" className="text-sm">
                  Activa
                </label>
              </div>
            </div>
          </div>

          {/* Descripción */}
          <div>
            <label className="block text-sm font-semibold">Descripción</label>
            <textarea
              rows={3}
              className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              value={data.descripcion}
              onChange={(e) => setData("descripcion", e.target.value)}
            />
            {errors.descripcion && <p className="text-sm text-red-500">{errors.descripcion}</p>}
          </div>

          {/* Footer modal */}
          <div className="mt-2 flex justify-end gap-2">
            <button
              type="button"
              onClick={() => setShowModal(false)}
              className="rounded-xl border border-primary-200/60 px-4 py-2 text-sm dark:border-primary-800/60"
            >
              Cancelar
            </button>
            <button
              type="submit"
              disabled={processing}
              className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700 disabled:opacity-60"
            >
              {processing ? "Guardando..." : editing ? "Actualizar" : "Crear"}
            </button>
          </div>
        </form>
      </Modal>
    </DashboardLayout>
  );
}
