import React, { useEffect, useMemo, useState } from "react";
import { useForm, router } from "@inertiajs/react";
import DashboardLayout from "@/Layouts/AuthenticatedLayout";
import Modal from "@/Components/Modal";
import {
  FiEdit,
  FiPlus,
  FiTrash2,
  FiRefreshCw,
  FiSearch,
  FiImage,
  FiMapPin,
  FiBookOpen,
  FiHash,
  FiX,
  FiTag,
} from "react-icons/fi";

/**
 * props esperadas desde el controlador:
 * { auth, labs, filters, flash }
 * - labs: paginator con data: [ { id, slug, nombre, responsable, descripcion, ubicacion{edificio,piso,oficina}, asignaturas[], imagenes[], deleted_at } ]
 * - filters: { q, trash, perPage, sort, dir }
 */

export default function LaboratoriosIndex({ auth, labs, filters, flash }) {
  // UI state
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState(null);

  // Filtros locales
  const [q, setQ] = useState(filters?.q ?? "");
  const [trash, setTrash] = useState(filters?.trash ?? "none");
  const [sort, setSort] = useState(filters?.sort ?? "nombre");
  const [dir, setDir] = useState(filters?.dir ?? "asc");

  // Form (crear/editar)
  const {
    data,
    setData,
    reset,
    post,
    patch,
    processing,
    errors,
    clearErrors,
  } = useForm({
    slug: "",
    nombre: "",
    responsable: "",
    descripcion: "",

    // Estructuras
    ubicacion: { edificio: "", piso: "", oficina: "" },
    asignaturas: [],

    // Galería:
    imagenes: [],           // (solo lectura para pintar, útil al crear si envías rutas)
    imagenes_files: [],     // FileList->Array para subir nuevas
    imagenes_keep: [],      // rutas existentes a conservar (solo en editar)
  });

  // Preview para nuevas imágenes
  const [newFiles, setNewFiles] = useState([]); // [{file, url}]
  // Manejador de "mantener" para existentes
  const [keepSet, setKeepSet] = useState(new Set()); // rutas existentes marcadas para conservar

  useEffect(() => {
    if (flash?.success) console.log("✅", flash.success);
    if (flash?.error) console.error("❌", flash.error);
  }, [flash]);

  // Modal: Crear
  const openCreate = () => {
    reset();
    setEditing(null);
    setNewFiles([]);
    setKeepSet(new Set()); // no hay existentes
    clearErrors();
    setShowModal(true);
  };

  // Modal: Editar
  const openEdit = (lab) => {
    setEditing(lab);

    const ubic = {
      edificio: lab?.ubicacion?.edificio ?? "",
      piso: lab?.ubicacion?.piso ?? "",
      oficina: lab?.ubicacion?.oficina ?? "",
    };

    const existentes = Array.isArray(lab?.imagenes) ? lab.imagenes : [];
    const initialKeep = new Set(existentes);

    setData({
      slug: lab.slug ?? "",
      nombre: lab.nombre ?? "",
      responsable: lab.responsable ?? "",
      descripcion: lab.descripcion ?? "",
      ubicacion: ubic,
      asignaturas: Array.isArray(lab.asignaturas) ? lab.asignaturas : [],
      imagenes: existentes,
      imagenes_files: [],
      imagenes_keep: Array.from(initialKeep),
    });

    setNewFiles([]);
    setKeepSet(initialKeep);
    clearErrors();
    setShowModal(true);
  };

  // Submit modal
  const submit = (e) => {
    e.preventDefault();

    const payload = {
      ...data,
      // normalizar keep según estado actual
      imagenes_keep: Array.from(keepSet),
    };

    if (editing) {
      patch(route("admin.laboratorios.update", editing.id), {
        data: payload,
        forceFormData: true, // asegura multipart si hay Files
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    } else {
      post(route("admin.laboratorios.store"), {
        data: payload,
        forceFormData: true,
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    }
  };

  // Acciones
  const destroy = (id) => {
    if (confirm("¿Enviar este laboratorio a la papelera?")) {
      router.delete(route("admin.laboratorios.destroy", id), { preserveScroll: true });
    }
  };
  const restore = (id) =>
    router.patch(route("admin.laboratorios.restore", id), {}, { preserveScroll: true });
  const forceDelete = (id) => {
    if (confirm("⚠️ Esto elimina definitivamente. ¿Continuar?")) {
      router.delete(route("admin.laboratorios.force", id), { preserveScroll: true });
    }
  };

  // Filtro/búsqueda
  const applyFilters = () => {
    router.get(
      route("admin.laboratorios.index"),
      { q, trash, sort, dir, page: 1 },
      { preserveScroll: true, replace: true }
    );
  };

  // Paginación
  const goto = (url) => {
    if (!url) return;
    router.visit(url, { preserveScroll: true, replace: true });
  };

  // ------ Asignaturas: campos dinámicos ------
  const [asigInput, setAsigInput] = useState("");
  const addAsignatura = () => {
    const v = asigInput.trim();
    if (!v) return;
    if (data.asignaturas.includes(v)) return;
    setData("asignaturas", [...data.asignaturas, v]);
    setAsigInput("");
  };
  const removeAsignatura = (idx) => {
    const arr = [...data.asignaturas];
    arr.splice(idx, 1);
    setData("asignaturas", arr);
  };

  // ------ Galería: nuevas imágenes (preview + quitar antes de enviar) ------
  const onNewImages = (e) => {
    const files = Array.from(e.target.files ?? []);
    if (files.length === 0) return;

    const mapped = files.map((f) => ({ file: f, url: URL.createObjectURL(f) }));
    setNewFiles((prev) => [...prev, ...mapped]);
    setData("imagenes_files", [...(data.imagenes_files || []), ...files]);
  };

  const removeNewFileAt = (i) => {
    const copy = [...newFiles];
    const removed = copy.splice(i, 1);
    setNewFiles(copy);

    const filesCopy = [...(data.imagenes_files || [])];
    const idx = filesCopy.findIndex((f) => f === removed[0]?.file);
    if (idx > -1) filesCopy.splice(idx, 1);
    setData("imagenes_files", filesCopy);
  };

  // ------ Galería: existentes (mantener/quitar) ------
  const toggleKeepExisting = (path) => {
    const next = new Set(keepSet);
    if (next.has(path)) {
      next.delete(path);
    } else {
      next.add(path);
    }
    setKeepSet(next);
  };

  // Helper UI
  const ubicacionText = (u) => {
    if (!u) return <span className="text-xs text-gray-400 italic">—</span>;
    const p = [u.edificio ? `Edif. ${u.edificio}` : null, u.piso ? `Piso ${u.piso}` : null, u.oficina ? `Of. ${u.oficina}` : null].filter(Boolean);
    return p.length ? p.join(" · ") : <span className="text-xs text-gray-400 italic">—</span>;
  };

  const asignaturasPreview = (arr) =>
    Array.isArray(arr) && arr.length > 0 ? arr.join(" · ") : (
      <span className="text-xs text-gray-400 italic">Sin asignaturas</span>
    );

  return (
    <DashboardLayout auth={auth} titulo="Gestión de Laboratorios">
      {/* Barra superior filtros + NUEVO */}
      <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
        {/* Filtros */}
        <div className="flex flex-col gap-2 sm:flex-row sm:items-center">
          <div className="flex items-center rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 dark:border-primary-800/60 dark:bg-primary-950/80">
            <FiSearch className="opacity-70" />
            <input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              onKeyDown={(e) => e.key === "Enter" && applyFilters()}
              placeholder="Buscar por nombre, slug, responsable, descripción…"
              className="ml-2 w-72 bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            />
          </div>

          <select
            value={trash}
            onChange={(e) => setTrash(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="none">Sin papelera</option>
            <option value="with">Incluir papelera</option>
            <option value="only">Solo papelera</option>
          </select>

          <select
            value={sort}
            onChange={(e) => setSort(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="nombre">Orden: Nombre</option>
            <option value="created_at">Orden: Fecha de creación</option>
          </select>

          <select
            value={dir}
            onChange={(e) => setDir(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="asc">Ascendente</option>
            <option value="desc">Descendente</option>
          </select>

          <button
            onClick={applyFilters}
            className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700"
          >
            Aplicar
          </button>
        </div>

        {/* NUEVO */}
        {trash !== "only" && (
          <button
            onClick={openCreate}
            className="flex items-center justify-center gap-2 rounded-xl bg-gradient-to-r from-primary-600 to-secondary-600 px-4 py-2 text-sm font-semibold text-white shadow hover:opacity-90"
          >
            <FiPlus className="text-lg" /> Nuevo
          </button>
        )}
      </div>

      {/* Tabla */}
      <div className="overflow-x-auto rounded-2xl border border-primary-200/60 bg-white/90 shadow-sm backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/80">
        <table className="min-w-full divide-y divide-primary-200 dark:divide-primary-800">
          <thead className="bg-primary-50 dark:bg-primary-900/40">
            <tr className="text-sm text-primary-700 dark:text-primary-200">
              <th className="px-4 py-2 text-left">Slug</th>
              <th className="px-4 py-2 text-left">Nombre</th>
              <th className="px-4 py-2 text-left">Responsable</th>
              <th className="px-4 py-2 text-left">Ubicación</th>
              <th className="px-4 py-2 text-left">Asignaturas</th>
              <th className="px-4 py-2 text-center">Imágenes</th>
              <th className="px-4 py-2 text-center">Acciones</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-primary-100 dark:divide-primary-800">
            {labs.data.map((l) => (
              <tr key={l.id} className="text-sm hover:bg-primary-50/60 dark:hover:bg-primary-900/30">
                <td className="px-4 py-2 font-mono text-xs flex items-center gap-2">
                  <FiHash className="opacity-60" />
                  {l.slug}
                </td>
                <td className="px-4 py-2 font-medium">{l.nombre}</td>
                <td className="px-4 py-2">{l.responsable || <span className="text-xs text-gray-400 italic">—</span>}</td>
                <td className="px-4 py-2">{ubicacionText(l.ubicacion)}</td>
                <td className="px-4 py-2">{asignaturasPreview(l.asignaturas)}</td>
                <td className="px-4 py-2 text-center">
                  <span className="inline-flex items-center gap-1 rounded-lg bg-primary-50 px-2 py-0.5 text-xs font-semibold text-primary-700 dark:bg-primary-900/40 dark:text-primary-200">
                    <FiImage /> {Array.isArray(l.imagenes) ? l.imagenes.length : 0}
                  </span>
                </td>
                <td className="px-4 py-2">
                  {!l.deleted_at ? (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => openEdit(l)}
                        className="rounded-md p-1.5 text-primary-600 hover:bg-primary-100 dark:text-primary-300 dark:hover:bg-primary-900/40"
                        title="Editar"
                      >
                        <FiEdit />
                      </button>
                      <button
                        onClick={() => destroy(l.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar (papelera)"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  ) : (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => restore(l.id)}
                        className="rounded-md p-1.5 text-green-600 hover:bg-green-100 dark:text-green-300 dark:hover:bg-green-900/40"
                        title="Restaurar"
                      >
                        <FiRefreshCw />
                      </button>
                      <button
                        onClick={() => forceDelete(l.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar definitivo"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  )}
                </td>
              </tr>
            ))}
            {labs.data.length === 0 && (
              <tr>
                <td colSpan="7" className="px-4 py-6 text-center text-gray-400">
                  No hay laboratorios para mostrar.
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Paginación */}
      <div className="mt-4 flex items-center justify-between text-sm">
        <div className="text-primary-700/80 dark:text-primary-200/80">
          Mostrando {labs.from ?? 0}–{labs.to ?? 0} de {labs.total ?? 0}
        </div>
        <div className="flex gap-2">
          <button
            disabled={!labs.prev_page_url}
            onClick={() => goto(labs.prev_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Anterior
          </button>
          <button
            disabled={!labs.next_page_url}
            onClick={() => goto(labs.next_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Siguiente
          </button>
        </div>
      </div>

      {/* Modal Crear/Editar */}
      <Modal show={showModal} onClose={() => setShowModal(false)} maxWidth="3xl">
        <form onSubmit={submit} className="p-6 space-y-5">
          <h2 className="mb-2 text-lg font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-200 dark:to-secondary-300">
            {editing ? "Editar laboratorio" : "Nuevo laboratorio"}
          </h2>

          {/* Básicos */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiHash /> Slug
              </label>
              <input
                type="text"
                placeholder="telecom, redes, electronica…"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 font-mono"
                value={data.slug}
                onChange={(e) => setData("slug", e.target.value)}
              />
              {errors.slug && <p className="text-sm text-red-500">{errors.slug}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Nombre</label>
              <input
                type="text"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.nombre}
                onChange={(e) => setData("nombre", e.target.value)}
              />
              {errors.nombre && <p className="text-sm text-red-500">{errors.nombre}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Responsable</label>
              <input
                type="text"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.responsable}
                onChange={(e) => setData("responsable", e.target.value)}
              />
              {errors.responsable && <p className="text-sm text-red-500">{errors.responsable}</p>}
            </div>
          </div>

          {/* Descripción */}
          <div>
            <label className="block text-sm font-semibold">Descripción</label>
            <textarea
              rows={3}
              className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              value={data.descripcion}
              onChange={(e) => setData("descripcion", e.target.value)}
            />
            {errors.descripcion && <p className="text-sm text-red-500">{errors.descripcion}</p>}
          </div>

          {/* Ubicación */}
          <div>
            <label className="block text-sm font-semibold flex items-center gap-2">
              <FiMapPin /> Ubicación
            </label>
            <div className="mt-1 grid grid-cols-1 gap-3 sm:grid-cols-3">
              <input
                type="text"
                placeholder="Edificio"
                value={data.ubicacion?.edificio ?? ""}
                onChange={(e) => setData("ubicacion", { ...(data.ubicacion || {}), edificio: e.target.value })}
                className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              <input
                type="text"
                placeholder="Piso"
                value={data.ubicacion?.piso ?? ""}
                onChange={(e) => setData("ubicacion", { ...(data.ubicacion || {}), piso: e.target.value })}
                className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              <input
                type="text"
                placeholder="Oficina"
                value={data.ubicacion?.oficina ?? ""}
                onChange={(e) => setData("ubicacion", { ...(data.ubicacion || {}), oficina: e.target.value })}
                className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
            </div>
            {errors.ubicacion && <p className="text-sm text-red-500">{errors.ubicacion}</p>}
          </div>

          {/* Asignaturas */}
          <div>
            <label className="block text-sm font-semibold flex items-center gap-2">
              <FiBookOpen /> Asignaturas
            </label>
            <div className="mt-1 flex gap-2">
              <input
                type="text"
                value={asigInput}
                onChange={(e) => setAsigInput(e.target.value)}
                onKeyDown={(e) => e.key === "Enter" && (e.preventDefault(), addAsignatura())}
                placeholder="Ej: Redes I"
                className="flex-1 rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              <button
                type="button"
                onClick={addAsignatura}
                className="rounded-xl bg-primary-600 px-3 py-2 text-sm font-semibold text-white hover:bg-primary-700"
              >
                Agregar
              </button>
            </div>
            <div className="mt-2 flex flex-wrap gap-2">
              {data.asignaturas?.map((asig, idx) => (
                <span
                  key={`${asig}-${idx}`}
                  className="group inline-flex items-center gap-2 rounded-xl border border-primary-200/60 bg-primary-50/70 px-3 py-1 text-xs font-medium dark:border-primary-800/60 dark:bg-primary-900/40"
                >
                  <FiTag />
                  {asig}
                  <button
                    type="button"
                    onClick={() => removeAsignatura(idx)}
                    className="rounded-md px-1 text-xs opacity-70 hover:bg-primary-100 dark:hover:bg-primary-900/60"
                    title="Quitar"
                  >
                    ×
                  </button>
                </span>
              ))}
              {(!data.asignaturas || data.asignaturas.length === 0) && (
                <span className="text-xs text-gray-400 italic">Aún sin asignaturas</span>
              )}
            </div>
            {errors.asignaturas && <p className="text-sm text-red-500">{errors.asignaturas}</p>}
          </div>

          {/* Galería: existentes (solo en editar) */}
          {editing && (
            <div>
              <label className="block text-sm font-semibold">Imágenes actuales</label>
              <div className="mt-2 grid grid-cols-2 sm:grid-cols-4 md:grid-cols-5 gap-3">
                {(data.imagenes || []).map((url) => {
                  const checked = keepSet.has(url);
                  return (
                    <div key={url} className="relative group">
                      <img
                        src={url}
                        alt="img"
                        className={`h-28 w-full rounded-xl object-cover border ${checked ? "border-primary-400" : "border-primary-200/60 dark:border-primary-800/60 opacity-70"}`}
                      />
                      <button
                        type="button"
                        onClick={() => toggleKeepExisting(url)}
                        className={`absolute top-2 right-2 rounded-full px-2 py-1 text-xs font-semibold shadow
                          ${checked ? "bg-green-600 text-white" : "bg-red-600 text-white"}`}
                        title={checked ? "Mantener" : "Quitar"}
                      >
                        {checked ? "Mantener" : "Quitar"}
                      </button>
                    </div>
                  );
                })}
                {(!data.imagenes || data.imagenes.length === 0) && (
                  <div className="col-span-full text-sm text-gray-400 italic">
                    No hay imágenes almacenadas.
                  </div>
                )}
              </div>
            </div>
          )}

          {/* Galería: agregar nuevas */}
          <div>
            <label className="block text-sm font-semibold flex items-center gap-2">
              <FiImage /> Agregar imágenes
            </label>
            <input
              type="file"
              multiple
              accept="image/*"
              onChange={onNewImages}
              className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm file:mr-3 file:rounded-lg file:border-0 file:bg-primary-100 file:px-3 file:py-1.5 dark:border-primary-800/60 dark:bg-primary-950/80"
            />
            {errors.imagenes_files && (
              <p className="text-sm text-red-500">{errors.imagenes_files}</p>
            )}
            {/* Previews nuevas */}
            {newFiles.length > 0 && (
              <div className="mt-3 grid grid-cols-2 sm:grid-cols-4 md:grid-cols-5 gap-3">
                {newFiles.map((nf, i) => (
                  <div key={nf.url} className="relative group">
                    <img
                      src={nf.url}
                      alt={`new-${i}`}
                      className="h-28 w-full rounded-xl object-cover border border-primary-200/60 dark:border-primary-800/60"
                    />
                    <button
                      type="button"
                      onClick={() => removeNewFileAt(i)}
                      className="absolute top-2 right-2 rounded-full bg-black/60 text-white p-1"
                      title="Quitar"
                    >
                      <FiX />
                    </button>
                  </div>
                ))}
              </div>
            )}
          </div>

          {/* Footer modal */}
          <div className="mt-2 flex justify-end gap-2">
            <button
              type="button"
              onClick={() => setShowModal(false)}
              className="rounded-xl border border-primary-200/60 px-4 py-2 text-sm dark:border-primary-800/60"
            >
              Cancelar
            </button>
            <button
              type="submit"
              disabled={processing}
              className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700 disabled:opacity-60"
            >
              {processing ? "Guardando..." : editing ? "Actualizar" : "Crear"}
            </button>
          </div>
        </form>
      </Modal>
    </DashboardLayout>
  );
}
