import React, { useEffect, useMemo, useState } from "react";
import { useForm, router } from "@inertiajs/react";
import DashboardLayout from "@/Layouts/AuthenticatedLayout";
import Modal from "@/Components/Modal";
import {
  FiEdit,
  FiPlus,
  FiTrash2,
  FiRefreshCw,
  FiSearch,
  FiLink,
  FiBookOpen,
  FiImage,
  FiTag,
} from "react-icons/fi";

/* Tipos de enlace predefinidos (puedes ampliar) */
const LINK_OPTIONS = [
  { value: "web", label: "Sitio / Perfil" },
  { value: "github", label: "GitHub" },
  { value: "scholar", label: "Google Scholar" },
  { value: "orcid", label: "ORCID" },
  { value: "scopus", label: "Scopus" },
  { value: "linkedin", label: "LinkedIn" },
  { value: "otro", label: "Otro…" },
];

export default function DocentesIndex({ auth, docentes, areas, filters, flash }) {
  // UI state
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState(null);

  // Filtros locales
  const [q, setQ] = useState(filters?.q ?? "");
  const [area, setArea] = useState(filters?.area ?? "");
  const [trash, setTrash] = useState(filters?.trash ?? "none");

  // Form (crear/editar)
  const { data, setData, reset, post, patch, processing, errors, clearErrors } = useForm({
    nombre: "",
    grado: "",
    area: "",
    email: "",
    // Foto:
    foto: "",        // string guardada (cuando ya existe)
    foto_file: null, // File a subir (nuevo/actualizado)
    // Contenido dinámico:
    asignaturas: [],
    enlaces: {},     // objeto { key: url }
  });

  // Preview de imagen
  const [fotoPreview, setFotoPreview] = useState(null);

  useEffect(() => {
    if (flash?.success) console.log("✅", flash.success);
    if (flash?.error) console.error("❌", flash.error);
  }, [flash]);

  // Modal: Crear
  const openCreate = () => {
    reset();
    setEditing(null);
    setFotoPreview(null);
    setShowModal(true);
  };

  // Modal: Editar
  const openEdit = (d) => {
    setEditing(d);
    setData({
      nombre: d.nombre ?? "",
      grado: d.grado ?? "",
      area: d.area ?? "",
      email: d.email ?? "",
      foto: d.foto ?? "",
      foto_file: null,
      asignaturas: Array.isArray(d.asignaturas) ? d.asignaturas : [],
      enlaces: d.enlaces && typeof d.enlaces === "object" ? d.enlaces : {},
    });
    setFotoPreview(null);
    clearErrors();
    setShowModal(true);
  };

  // Submit modal
  const submit = (e) => {
    e.preventDefault();
    if (editing) {
      patch(route("admin.docentes.update", editing.id), {
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    } else {
      post(route("admin.docentes.store"), {
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    }
  };

  // Acciones
  const destroy = (id) => {
    if (confirm("¿Enviar este docente a la papelera?")) {
      router.delete(route("admin.docentes.destroy", id), { preserveScroll: true });
    }
  };
  const restore = (id) =>
    router.patch(route("admin.docentes.restore", id), {}, { preserveScroll: true });
  const forceDelete = (id) => {
    if (confirm("⚠️ Esto elimina definitivamente. ¿Continuar?")) {
      router.delete(route("admin.docentes.force", id), { preserveScroll: true });
    }
  };

  // Filtro/búsqueda
  const applyFilters = () => {
    router.get(
      route("admin.docentes.index"),
      { q, area, trash, page: 1 },
      { preserveScroll: true, replace: true }
    );
  };

  // Paginación
  const goto = (url) => {
    if (!url) return;
    router.visit(url, { preserveScroll: true, replace: true });
  };

  // Helpers UI
  const asignaturasPreview = (arr) =>
    Array.isArray(arr) && arr.length > 0 ? arr.join(" · ") : (
      <span className="text-xs text-gray-400 italic">Sin asignaturas</span>
    );

  // ------ Asignaturas: campos dinámicos ------
  const [asigInput, setAsigInput] = useState("");
  const addAsignatura = () => {
    const v = asigInput.trim();
    if (!v) return;
    if (data.asignaturas.includes(v)) return;
    setData("asignaturas", [...data.asignaturas, v]);
    setAsigInput("");
  };
  const removeAsignatura = (idx) => {
    const arr = [...data.asignaturas];
    arr.splice(idx, 1);
    setData("asignaturas", arr);
  };

  // ------ Enlaces: select + input + agregar ------
  const [linkType, setLinkType] = useState("web");
  const [linkKey, setLinkKey] = useState(""); // solo si 'otro'
  const [linkUrl, setLinkUrl] = useState("");

  const addEnlace = () => {
    const key = linkType === "otro" ? linkKey.trim() : linkType;
    const url = linkUrl.trim();
    if (!key || !url) return;

    // Normaliza la key para objeto
    const k = key.toLowerCase().replace(/\s+/g, "_");
    setData("enlaces", { ...(data.enlaces || {}), [k]: url });

    // limpiar inputs
    if (linkType === "otro") setLinkKey("");
    setLinkUrl("");
  };

  const removeEnlace = (k) => {
    const copy = { ...(data.enlaces || {}) };
    delete copy[k];
    setData("enlaces", copy);
  };

  // Sugerencias de áreas (opcional)
  const areaOptions = useMemo(() => {
    const opts = Array.isArray(areas) ? areas : [];
    return opts.filter(Boolean);
  }, [areas]);

  // Imagen: manejo de archivo + preview
  const onFotoChange = (e) => {
    const file = e.target.files?.[0] ?? null;
    setData("foto_file", file);
    if (file) {
      const url = URL.createObjectURL(file);
      setFotoPreview(url);
    } else {
      setFotoPreview(null);
    }
  };

  return (
    <DashboardLayout auth={auth} titulo="Gestión de Docentes">
      {/* Barra superior filtros + NUEVO */}
      <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
        {/* Filtros */}
        <div className="flex flex-col gap-2 sm:flex-row sm:items-center">
          <div className="flex items-center rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 dark:border-primary-800/60 dark:bg-primary-950/80">
            <FiSearch className="opacity-70" />
            <input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              onKeyDown={(e) => e.key === "Enter" && applyFilters()}
              placeholder="Buscar por nombre, correo o área…"
              className="ml-2 w-64 bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            />
          </div>

          <select
            value={area}
            onChange={(e) => setArea(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Área: Todas</option>
            {areaOptions.map((a) => (
              <option key={a} value={a}>
                {a}
              </option>
            ))}
          </select>

          <select
            value={trash}
            onChange={(e) => setTrash(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="none">Sin papelera</option>
            <option value="with">Incluir papelera</option>
            <option value="only">Solo papelera</option>
          </select>

          <button
            onClick={applyFilters}
            className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700"
          >
            Aplicar
          </button>
        </div>

        {/* NUEVO Docente */}
        {trash !== "only" && (
          <button
            onClick={openCreate}
            className="flex items-center justify-center gap-2 rounded-xl bg-gradient-to-r from-primary-600 to-secondary-600 px-4 py-2 text-sm font-semibold text-white shadow hover:opacity-90"
          >
            <FiPlus className="text-lg" /> Nuevo
          </button>
        )}
      </div>

      {/* Tabla */}
      <div className="overflow-x-auto rounded-2xl border border-primary-200/60 bg-white/90 shadow-sm backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/80">
        <table className="min-w-full divide-y divide-primary-200 dark:divide-primary-800">
          <thead className="bg-primary-50 dark:bg-primary-900/40">
            <tr className="text-sm text-primary-700 dark:text-primary-200">
              <th className="px-4 py-2 text-left">Nombre</th>
              <th className="px-4 py-2 text-left">Grado</th>
              <th className="px-4 py-2 text-left">Área</th>
              <th className="px-4 py-2 text-left">Correo</th>
              <th className="px-4 py-2 text-left">Asignaturas</th>
              <th className="px-4 py-2 text-center">Acciones</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-primary-100 dark:divide-primary-800">
            {docentes.data.map((d) => (
              <tr key={d.id} className="text-sm hover:bg-primary-50/60 dark:hover:bg-primary-900/30">
                <td className="px-4 py-2 font-medium flex items-center gap-2">
                  {d.foto && (
                    <img
                      src={d.foto}
                      alt={d.nombre}
                      className="h-8 w-8 rounded-full object-cover border border-primary-200/60 dark:border-primary-800/60"
                    />
                  )}
                  {d.nombre}
                </td>
                <td className="px-4 py-2">{d.grado || <span className="text-xs text-gray-400 italic">—</span>}</td>
                <td className="px-4 py-2">{d.area || <span className="text-xs text-gray-400 italic">—</span>}</td>
                <td className="px-4 py-2">{d.email}</td>
                <td className="px-4 py-2">{asignaturasPreview(d.asignaturas)}</td>
                <td className="px-4 py-2">
                  {!d.deleted_at ? (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => openEdit(d)}
                        className="rounded-md p-1.5 text-primary-600 hover:bg-primary-100 dark:text-primary-300 dark:hover:bg-primary-900/40"
                        title="Editar"
                      >
                        <FiEdit />
                      </button>
                      <button
                        onClick={() => destroy(d.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar (papelera)"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  ) : (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => restore(d.id)}
                        className="rounded-md p-1.5 text-green-600 hover:bg-green-100 dark:text-green-300 dark:hover:bg-green-900/40"
                        title="Restaurar"
                      >
                        <FiRefreshCw />
                      </button>
                      <button
                        onClick={() => forceDelete(d.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar definitivo"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  )}
                </td>
              </tr>
            ))}
            {docentes.data.length === 0 && (
              <tr>
                <td colSpan="6" className="px-4 py-6 text-center text-gray-400">
                  No hay docentes para mostrar.
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Paginación */}
      <div className="mt-4 flex items-center justify-between text-sm">
        <div className="text-primary-700/80 dark:text-primary-200/80">
          Mostrando {docentes.from ?? 0}–{docentes.to ?? 0} de {docentes.total ?? 0}
        </div>
        <div className="flex gap-2">
          <button
            disabled={!docentes.prev_page_url}
            onClick={() => goto(docentes.prev_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Anterior
          </button>
          <button
            disabled={!docentes.next_page_url}
            onClick={() => goto(docentes.next_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Siguiente
          </button>
        </div>
      </div>

      {/* Modal Crear/Editar */}
      <Modal show={showModal} onClose={() => setShowModal(false)} maxWidth="2xl">
        <form onSubmit={submit} className="p-6 space-y-4">
          <h2 className="mb-2 text-lg font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-200 dark:to-secondary-300">
            {editing ? "Editar docente" : "Nuevo docente"}
          </h2>

          {/* Básicos */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-2">
            <div>
              <label className="block text-sm font-semibold">Nombre</label>
              <input
                type="text"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.nombre}
                onChange={(e) => setData("nombre", e.target.value)}
              />
              {errors.nombre && <p className="text-sm text-red-500">{errors.nombre}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Correo</label>
              <input
                type="email"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.email}
                onChange={(e) => setData("email", e.target.value)}
              />
              {errors.email && <p className="text-sm text-red-500">{errors.email}</p>}
            </div>
          </div>

          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div>
              <label className="block text-sm font-semibold">Grado</label>
              <input
                type="text"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.grado}
                onChange={(e) => setData("grado", e.target.value)}
              />
              {errors.grado && <p className="text-sm text-red-500">{errors.grado}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Área</label>
              <input
                list="areaOptions"
                type="text"
                placeholder="Ej: Telecomunicaciones, Electrónica…"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.area}
                onChange={(e) => setData("area", e.target.value)}
              />
              <datalist id="areaOptions">
                {areaOptions.map((a) => (
                  <option key={a} value={a} />
                ))}
              </datalist>
              {errors.area && <p className="text-sm text-red-500">{errors.area}</p>}
            </div>

            {/* Foto: upload + preview */}
            <div>
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiImage /> Foto del docente
              </label>
              <input
                type="file"
                accept="image/*"
                onChange={onFotoChange}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm file:mr-3 file:rounded-lg file:border-0 file:bg-primary-100 file:px-3 file:py-1.5 dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              {(fotoPreview || data.foto) && (
                <div className="mt-2 flex items-center gap-3">
                  <img
                    src={fotoPreview || data.foto}
                    alt="preview"
                    className="h-14 w-14 rounded-xl object-cover border border-primary-200/60 dark:border-primary-800/60"
                  />
                  {fotoPreview && (
                    <span className="text-xs text-primary-700/80 dark:text-primary-200/80">
                      Vista previa (sin guardar)
                    </span>
                  )}
                </div>
              )}
              {errors.foto_file && <p className="text-sm text-red-500">{errors.foto_file}</p>}
            </div>
          </div>

          {/* Asignaturas */}
          <div>
            <label className="block text-sm font-semibold flex items-center gap-2">
              <FiBookOpen /> Asignaturas
            </label>
            <div className="mt-1 flex gap-2">
              <input
                type="text"
                value={asigInput}
                onChange={(e) => setAsigInput(e.target.value)}
                onKeyDown={(e) => e.key === "Enter" && (e.preventDefault(), addAsignatura())}
                placeholder="Ej: Redes I"
                className="flex-1 rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
              <button
                type="button"
                onClick={addAsignatura}
                className="rounded-xl bg-primary-600 px-3 py-2 text-sm font-semibold text-white hover:bg-primary-700"
              >
                Agregar
              </button>
            </div>

            <div className="mt-2 flex flex-wrap gap-2">
              {data.asignaturas?.map((asig, idx) => (
                <span
                  key={`${asig}-${idx}`}
                  className="group inline-flex items-center gap-2 rounded-xl border border-primary-200/60 bg-primary-50/70 px-3 py-1 text-xs font-medium dark:border-primary-800/60 dark:bg-primary-900/40"
                >
                  <FiTag />
                  {asig}
                  <button
                    type="button"
                    onClick={() => removeAsignatura(idx)}
                    className="rounded-md px-1 text-xs opacity-70 hover:bg-primary-100 dark:hover:bg-primary-900/60"
                    title="Quitar"
                  >
                    ×
                  </button>
                </span>
              ))}
              {(!data.asignaturas || data.asignaturas.length === 0) && (
                <span className="text-xs text-gray-400 italic">Aún sin asignaturas</span>
              )}
            </div>
            {errors.asignaturas && <p className="text-sm text-red-500">{errors.asignaturas}</p>}
          </div>

          {/* Enlaces: select + (key si otro) + url + agregar */}
          <div>
            <label className="block text-sm font-semibold flex items-center gap-2">
              <FiLink /> Enlaces
            </label>

            <div className="mt-2 grid grid-cols-1 gap-2 sm:grid-cols-3">
              <select
                value={linkType}
                onChange={(e) => setLinkType(e.target.value)}
                className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              >
                {LINK_OPTIONS.map((o) => (
                  <option key={o.value} value={o.value}>
                    {o.label}
                  </option>
                ))}
              </select>

              {linkType === "otro" && (
                <input
                  type="text"
                  placeholder="Nombre del enlace (ej: portfolio)"
                  value={linkKey}
                  onChange={(e) => setLinkKey(e.target.value)}
                  className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                />
              )}

              <input
                type="text"
                placeholder="https://..."
                value={linkUrl}
                onChange={(e) => setLinkUrl(e.target.value)}
                className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              />
            </div>

            <div className="mt-2">
              <button
                type="button"
                onClick={addEnlace}
                className="rounded-xl bg-primary-600 px-3 py-2 text-sm font-semibold text-white hover:bg-primary-700"
              >
                Agregar enlace
              </button>
            </div>

            {/* Lista de enlaces actuales */}
            <div className="mt-3 space-y-2">
              {Object.entries(data.enlaces || {}).length === 0 && (
                <div className="text-xs text-gray-400 italic">Sin enlaces</div>
              )}
              {Object.entries(data.enlaces || {}).map(([k, url]) => (
                <div
                  key={k}
                  className="flex items-center justify-between rounded-xl border border-primary-200/60 bg-primary-50/50 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-900/40"
                >
                  <div className="flex items-center gap-2">
                    <span className="min-w-[90px] font-semibold">{k}</span>
                    <a
                      href={url}
                      target="_blank"
                      rel="noreferrer"
                      className="truncate text-primary-700 underline dark:text-primary-300"
                      title={url}
                    >
                      {url}
                    </a>
                  </div>
                  <button
                    type="button"
                    onClick={() => removeEnlace(k)}
                    className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                    title="Eliminar enlace"
                  >
                    <FiTrash2 />
                  </button>
                </div>
              ))}
            </div>

            {errors.enlaces && <p className="text-sm text-red-500">{errors.enlaces}</p>}
          </div>

          {/* Footer modal */}
          <div className="mt-4 flex justify-end gap-2">
            <button
              type="button"
              onClick={() => setShowModal(false)}
              className="rounded-xl border border-primary-200/60 px-4 py-2 text-sm dark:border-primary-800/60"
            >
              Cancelar
            </button>
            <button
              type="submit"
              disabled={processing}
              className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700 disabled:opacity-60"
            >
              {processing ? "Guardando..." : editing ? "Actualizar" : "Crear"}
            </button>
          </div>
        </form>
      </Modal>
    </DashboardLayout>
  );
}
