import React, { useMemo } from "react";
import { motion, AnimatePresence } from "framer-motion";
import {
  FiUsers, FiBookOpen, FiBriefcase, FiAperture, FiLayers,
  FiTrendingUp, FiClock, FiAlertCircle, FiCheckCircle
} from "react-icons/fi";
import AuthenticatedLayout from "@/Layouts/AuthenticatedLayout";
import {
  LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer,
  PieChart, Pie, Cell, Legend, BarChart, Bar
} from "recharts";

/* =========================
   Helpers UI
   ========================= */
const cls = (...xs) => xs.filter(Boolean).join(" ");
const fadeIn = { hidden: { opacity: 0, y: 12 }, show: { opacity: 1, y: 0, transition: { duration: .5 } } };
const spring = { type: "spring", stiffness: 120, damping: 18 };

const KPI = ({ icon: Icon, label, value, hint }) => (
  <motion.div variants={fadeIn} className="rounded-2xl p-4 bg-white shadow-sm border border-gray-100">
    <div className="flex items-center justify-between">
      <div className="flex items-center gap-3">
        <span className="inline-flex p-2 rounded-xl bg-primary-50 text-primary-700">
          <Icon size={20} />
        </span>
        <div>
          <div className="text-xs uppercase tracking-wide text-gray-500">{label}</div>
          <div className="text-2xl font-bold text-gray-900">{value ?? 0}</div>
        </div>
      </div>
      {hint && <div className="text-xs text-gray-400">{hint}</div>}
    </div>
  </motion.div>
);

const Card = ({ title, subtitle, right, children, className }) => (
  <motion.div layout transition={spring} className={cls("rounded-2xl bg-white shadow-sm border border-gray-100", className)}>
    <div className="px-4 pt-4 pb-2 flex flex-wrap items-center justify-between gap-3">
      <div>
        <h3 className="text-sm font-semibold text-gray-800">{title}</h3>
        {subtitle && <p className="text-xs text-gray-500">{subtitle}</p>}
      </div>
      {right}
    </div>
    <div className="p-4 pt-0">{children}</div>
  </motion.div>
);

/* =========================
   Recharts color scale (usa tu paleta primary/secondary)
   ========================= */
const COLORS = [
  "#1e3a8a", // primary-800
  "#0ea5e9", // secondary-ish
  "#10b981", // emerald
  "#f59e0b", // amber
  "#ef4444", // red
  "#8b5cf6", // violet
  "#14b8a6", // teal
  "#3b82f6", // blue
];

/* =========================
   Data adapters
   ========================= */
function toLineData(series) {
  if (!series || !series.labels || !series.data) return [];
  return series.labels.map((m, i) => ({ name: m, value: series.data[i] ?? 0 }));
}

function toPieDataFromObject(obj, nameOn = "name", valueOn = "value") {
  if (!obj) return [];
  return Object.entries(obj).map(([k, v]) => ({ [nameOn]: k, [valueOn]: Number(v) || 0 }));
}
function normalizeCountRows(rows, key = "name") {
  if (!rows || !rows.length) return [];
  // rows tipo: [{estado, total}] o [{carrera, total}]
  const entryKey = Object.keys(rows[0] || {}).find(k => k !== "total") || key;
  return rows.map(r => ({ name: r[entryKey] ?? "N/D", value: Number(r.total) || 0 }));
}

/* =========================
   Componente principal
   ========================= */
export default function Index({ auth, range, totals = {}, breakdowns = {}, series = {}, tops = {}, upcoming = {} }) {
  // KPIs principales
  const kpis = useMemo(() => ([
    { icon: FiUsers,      label: "Autoridades",          value: totals.autoridades },
    { icon: FiUsers,      label: "Docentes",             value: totals.docentes },
    { icon: FiAperture,   label: "Laboratorios",         value: totals.laboratorios },
    { icon: FiLayers,     label: "Líneas de investigación", value: totals.lineas_investigacion },
    { icon: FiBriefcase,  label: "Pasantías",            value: totals.pasantias },
    { icon: FiBookOpen,   label: "Pregrados",            value: totals.pregrados },
    { icon: FiBookOpen,   label: "Postgrados",           value: totals.postgrados },
    { icon: FiTrendingUp, label: "Proyectos",            value: totals.proyectos },
    { icon: FiBookOpen,   label: "Publicaciones",        value: totals.publicaciones },
    { icon: FiTrendingUp, label: "TIC Projects",         value: totals.tic_projects },
  ]), [totals]);

  // Series
  const seriePasantias = toLineData(series.pasantias);
  const serieProyectos = toLineData(series.proyectos);
  const seriePublicaciones = toLineData(series.publicaciones);

  // Donut: pasantías por estado
  const donutPasantiasEstado = normalizeCountRows(breakdowns.pasantias_por_estado);

  // Barras: top pasantías por carrera / proyectos por línea
  const barPasantiasCarrera = normalizeCountRows(tops.pasantias_por_carrera, "carrera");
  const barProyectosLinea   = normalizeCountRows(tops.proyectos_por_linea, "linea_investigacion");

  // Publicaciones activas
  const publicacionesActivasData = toPieDataFromObject(breakdowns.publicaciones_activas);

  // Próximos vencimientos
  const proximasPasantias = upcoming?.pasantias_vencen_30d || [];

  return (
    <AuthenticatedLayout auth={auth} titulo="Dashboard">
      <div className="px-4 sm:px-6 lg:px-8 py-6 space-y-6">
        {/* Header simple con rango */}
        <div className="flex flex-wrap items-center justify-between gap-3">
          <div>
            <h1 className="text-xl font-bold text-gray-900">Panel general</h1>
            {range?.from && range?.to && (
              <p className="text-sm text-gray-500">Periodo: {range.from} — {range.to}</p>
            )}
          </div>
        </div>

        {/* KPIs grid */}
        <AnimatePresence>
          <motion.div
            initial="hidden"
            animate="show"
            variants={{ show: { transition: { staggerChildren: 0.05 } } }}
            className="grid grid-cols-1 xs:grid-cols-2 sm:grid-cols-3 lg:grid-cols-5 gap-4"
          >
            {kpis.map((k, i) => <KPI key={i} {...k} />)}
          </motion.div>
        </AnimatePresence>

        {/* Row 1: Series líneas */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
          <Card
            title="Pasantías creadas (últimos 12 meses)"
            subtitle="Total de registros por mes"
            right={<span className="text-xs text-gray-500 flex items-center gap-1"><FiClock /> Último año</span>}
          >
            <div className="h-64">
              {seriePasantias.length ? (
                <ResponsiveContainer width="100%" height="100%">
                  <LineChart data={seriePasantias} margin={{ left: 8, right: 8, top: 10, bottom: 0 }}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" fontSize={12} />
                    <YAxis allowDecimals={false} fontSize={12} />
                    <Tooltip />
                    <Line type="monotone" dataKey="value" stroke="#1e3a8a" strokeWidth={2} dot={false} />
                  </LineChart>
                </ResponsiveContainer>
              ) : (
                <EmptyState text="Sin datos de serie para pasantías." />
              )}
            </div>
          </Card>

          <Card
            title="Proyectos de investigación (últimos 12 meses)"
            subtitle="Total de registros por mes"
            right={<span className="text-xs text-gray-500 flex items-center gap-1"><FiClock /> Último año</span>}
          >
            <div className="h-64">
              {serieProyectos.length ? (
                <ResponsiveContainer width="100%" height="100%">
                  <LineChart data={serieProyectos} margin={{ left: 8, right: 8, top: 10, bottom: 0 }}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" fontSize={12} />
                    <YAxis allowDecimals={false} fontSize={12} />
                    <Tooltip />
                    <Line type="monotone" dataKey="value" stroke="#0ea5e9" strokeWidth={2} dot={false} />
                  </LineChart>
                </ResponsiveContainer>
              ) : (
                <EmptyState text="Sin datos de serie para proyectos." />
              )}
            </div>
          </Card>
        </div>

        {/* Row 2: Donuts / Barras */}
        <div className="grid grid-cols-1 xl:grid-cols-3 gap-4">
          <Card title="Pasantías por estado" subtitle="Distribución actual">
            <div className="h-64">
              {donutPasantiasEstado.length ? (
                <ResponsiveContainer width="100%" height="100%">
                  <PieChart>
                    <Pie data={donutPasantiasEstado} dataKey="value" nameKey="name" innerRadius={50} outerRadius={80} paddingAngle={3}>
                      {donutPasantiasEstado.map((_, i) => <Cell key={i} fill={COLORS[i % COLORS.length]} />)}
                    </Pie>
                    <Tooltip />
                    <Legend />
                  </PieChart>
                </ResponsiveContainer>
              ) : (
                <EmptyState text="Aún no hay estados de pasantías." />
              )}
            </div>
          </Card>

          <Card title="Top pasantías por carrera" subtitle="Top 8">
            <div className="h-64">
              {barPasantiasCarrera.length ? (
                <ResponsiveContainer width="100%" height="100%">
                  <BarChart data={barPasantiasCarrera} margin={{ left: 8, right: 8, top: 10, bottom: 0 }}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" fontSize={12} />
                    <YAxis allowDecimals={false} fontSize={12} />
                    <Tooltip />
                    <Bar dataKey="value" fill="#1e3a8a" radius={[6, 6, 0, 0]} />
                  </BarChart>
                </ResponsiveContainer>
              ) : (
                <EmptyState text="Sin datos por carrera." />
              )}
            </div>
          </Card>

          <Card title="Proyectos por línea de investigación" subtitle="Top 8">
            <div className="h-64">
              {barProyectosLinea.length ? (
                <ResponsiveContainer width="100%" height="100%">
                  <BarChart data={barProyectosLinea} margin={{ left: 8, right: 8, top: 10, bottom: 0 }}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" fontSize={12} />
                    <YAxis allowDecimals={false} fontSize={12} />
                    <Tooltip />
                    <Bar dataKey="value" fill="#0ea5e9" radius={[6, 6, 0, 0]} />
                  </BarChart>
                </ResponsiveContainer>
              ) : (
                <EmptyState text="Sin datos por línea." />
              )}
            </div>
          </Card>
        </div>

        {/* Row 3: Publicaciones + Serie Publicaciones + Próximos vencimientos */}
        <div className="grid grid-cols-1 xl:grid-cols-3 gap-4">
          <Card title="Publicaciones (activas vs inactivas)">
            <div className="h-64">
              {publicacionesActivasData.length ? (
                <ResponsiveContainer width="100%" height="100%">
                  <PieChart>
                    <Pie data={publicacionesActivasData} dataKey="value" nameKey="name" outerRadius={80} label>
                      {publicacionesActivasData.map((_, i) => <Cell key={i} fill={COLORS[i % COLORS.length]} />)}
                    </Pie>
                    <Tooltip />
                    <Legend />
                  </PieChart>
                </ResponsiveContainer>
              ) : (
                <EmptyState text="No hay desglose de publicaciones." />
              )}
            </div>
          </Card>

          <Card title="Publicaciones por mes (últimos 12 meses)">
            <div className="h-64">
              {seriePublicaciones.length ? (
                <ResponsiveContainer width="100%" height="100%">
                  <LineChart data={seriePublicaciones} margin={{ left: 8, right: 8, top: 10, bottom: 0 }}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" fontSize={12} />
                    <YAxis allowDecimals={false} fontSize={12} />
                    <Tooltip />
                    <Line type="monotone" dataKey="value" stroke="#10b981" strokeWidth={2} dot={false} />
                  </LineChart>
                </ResponsiveContainer>
              ) : (
                <EmptyState text="Sin datos de serie para publicaciones." />
              )}
            </div>
          </Card>

          <Card title="Pasantías: plazos próximos (30 días)">
            <div className="space-y-3">
              {proximasPasantias.length ? (
                <ul className="divide-y divide-gray-100">
                  {proximasPasantias.map(item => (
                    <li key={item.id} className="py-3 flex items-start gap-3">
                      <span className="mt-0.5 inline-flex p-2 rounded-xl bg-amber-50 text-amber-700">
                        <FiAlertCircle />
                      </span>
                      <div className="min-w-0">
                        <div className="font-medium text-gray-800 truncate">{item.titulo}</div>
                        <div className="text-xs text-gray-500">
                          {item.empresa ? `${item.empresa} · ` : ""}{item.carrera || "—"}
                        </div>
                        <div className="text-xs text-gray-500">
                          Vence: <span className="font-medium">{item.fecha_maxima_postulacion?.slice(0,10)}</span>
                          {item.estado ? <> · Estado: <span className="uppercase">{item.estado}</span></> : null}
                        </div>
                      </div>
                    </li>
                  ))}
                </ul>
              ) : (
                <EmptyState text="No hay pasantías por vencer en los próximos 30 días." icon={<FiCheckCircle />} />
              )}
            </div>
          </Card>
        </div>
      </div>
    </AuthenticatedLayout>
  );
}

/* =========================
   Empty state compacto
   ========================= */
function EmptyState({ text = "Sin información", icon }) {
  return (
    <div className="h-full w-full flex items-center justify-center">
      <div className="text-center text-gray-400">
        <div className="mx-auto mb-2 inline-flex p-2 rounded-xl bg-gray-50">{icon || <FiAlertCircle />}</div>
        <p className="text-sm">{text}</p>
      </div>
    </div>
  );
}
