import React, { useEffect, useMemo, useState } from "react";
import { useForm, router } from "@inertiajs/react";
import DashboardLayout from "@/Layouts/AuthenticatedLayout";
import Modal from "@/Components/Modal";
import {
  FiEdit,
  FiPlus,
  FiTrash2,
  FiRefreshCw,
  FiSearch,
  FiImage,
  FiMapPin,
  FiMail,
  FiUser,
  FiHash,
  FiX,
  FiClock,
  FiAtSign,
  FiPhone,
  FiTag,
} from "react-icons/fi";

/**
 * props esperadas desde el controlador:
 * {
 *   auth,
 *   autoridades,  // paginator
 *   filters,      // { q, grupoKey, area, trash, perPage, sort, dir }
 *   areas,        // string[]
 *   grupoKeys,    // string[]
 *   labels,       // Record<grupoKey, label>
 *   flash
 * }
 */

export default function AutoridadesIndex({ auth, autoridades, filters, areas, grupoKeys, labels, flash }) {
  // UI state
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState(null);

  // Filtros locales
  const [q, setQ] = useState(filters?.q ?? "");
  const [grupoKey, setGrupoKey] = useState(filters?.grupoKey ?? "");
  const [area, setArea] = useState(filters?.area ?? "");
  const [trash, setTrash] = useState(filters?.trash ?? "none");
  const [sort, setSort] = useState(filters?.sort ?? "orden");
  const [dir, setDir] = useState(filters?.dir ?? "asc");

  // Form (crear/editar)
  const { data, setData, reset, post, patch, processing, errors, clearErrors } = useForm({
    slug: "",
    grupo_key: "",
    grupo: "",
    nombre: "",
    cargo: "",
    area: "",
    email: "",
    telefono: "",
    horario_dias: "",
    horario_rango: "",
    ubicacion: "",
    foto: "",       // ruta actual (string) si existe
    foto_file: null // File para subida
  });

  // Preview para nueva foto
  const [fotoPreview, setFotoPreview] = useState(null);

  useEffect(() => {
    if (flash?.success) console.log("✅", flash.success);
    if (flash?.error) console.error("❌", flash.error);
  }, [flash]);

  // Modal: Crear
  const openCreate = () => {
    reset();
    setEditing(null);
    setFotoPreview(null);
    clearErrors();
    setShowModal(true);
  };

  // Modal: Editar
  const openEdit = (a) => {
    setEditing(a);
    setData({
      slug: a.slug || "",
      grupo_key: a.grupo_key || "",
      grupo: a.grupo || (labels?.[a.grupo_key] ?? ""),
      nombre: a.nombre || "",
      cargo: a.cargo || "",
      area: a.area || "",
      email: a.email || "",
      telefono: a.telefono || "",
      horario_dias: a.horario_dias || "",
      horario_rango: a.horario_rango || "",
      ubicacion: a.ubicacion || "",
      foto: a.foto || "",
      foto_file: null,
    });
    setFotoPreview(null);
    clearErrors();
    setShowModal(true);
  };

  // Submit modal
  const submit = (e) => {
    e.preventDefault();

    const payload = { ...data };

    // Si grupo está vacío, derive desde grupo_key y labels (servidor también lo hace; esto es UX)
    if (!payload.grupo && payload.grupo_key && labels?.[payload.grupo_key]) {
      payload.grupo = labels[payload.grupo_key];
    }

    if (editing) {
      patch(route("admin.autoridades.update", editing.id), {
        data: payload,
        forceFormData: true, // asegura multipart si hay File
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    } else {
      post(route("admin.autoridades.store"), {
        data: payload,
        forceFormData: true,
        preserveScroll: true,
        onSuccess: () => setShowModal(false),
      });
    }
  };

  // Acciones
  const destroy = (id) => {
    if (confirm("¿Enviar esta autoridad a la papelera?")) {
      router.delete(route("admin.autoridades.destroy", id), { preserveScroll: true });
    }
  };
  const restore = (id) =>
    router.patch(route("admin.autoridades.restore", id), {}, { preserveScroll: true });
  const forceDelete = (id) => {
    if (confirm("⚠️ Esto elimina definitivamente. ¿Continuar?")) {
      router.delete(route("admin.autoridades.force", id), { preserveScroll: true });
    }
  };

  // Filtro/búsqueda
  const applyFilters = () => {
    router.get(
      route("admin.autoridades.index"),
      { q, grupoKey, area, trash, sort, dir, page: 1 },
      { preserveScroll: true, replace: true }
    );
  };

  // Paginación
  const goto = (url) => {
    if (!url) return;
    router.visit(url, { preserveScroll: true, replace: true });
  };

  // Helpers UI
  const horarioText = (a) => {
    const d = a.horario_dias?.trim();
    const r = a.horario_rango?.trim();
    if (d && r) return `${d} · ${r}`;
    return d || r || <span className="text-xs text-gray-400 italic">—</span>;
  };

  const contactoText = (a) => {
    const mail = a.email ? <span className="inline-flex items-center gap-1"><FiAtSign className="opacity-70" />{a.email}</span> : null;
    const tel = a.telefono ? <span className="inline-flex items-center gap-1"><FiPhone className="opacity-70" />{a.telefono}</span> : null;
    return (
      <div className="flex flex-col gap-1">
        {mail || <span className="text-xs text-gray-400 italic">sin email</span>}
        {tel || <span className="text-xs text-gray-400 italic">sin teléfono</span>}
      </div>
    );
  };

  // ------ Foto: seleccionar / preview / limpiar ------
  const onFotoFile = (e) => {
    const file = e.target.files?.[0];
    setData("foto_file", file || null);
    if (file) {
      setFotoPreview(URL.createObjectURL(file));
    } else {
      setFotoPreview(null);
    }
  };
  const clearFoto = () => {
    setData("foto", "");
    setData("foto_file", null);
    setFotoPreview(null);
  };

  // Auto-derivar etiqueta "grupo" al cambiar "grupo_key" (si grupo está vacío o coincide con el label anterior)
  const onChangeGrupoKey = (newKey) => {
    const currentLabel = labels?.[data.grupo_key] ?? "";
    const nextLabel = labels?.[newKey] ?? "";
    const groupIsDefault = !data.grupo || data.grupo === currentLabel;
    setData("grupo_key", newKey);
    if (groupIsDefault) setData("grupo", nextLabel);
  };

  return (
    <DashboardLayout auth={auth} titulo="Gestión de Autoridades">
      {/* Barra superior filtros + NUEVO */}
      <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
        {/* Filtros */}
        <div className="flex flex-col gap-2 sm:flex-row sm:items-center">
          <div className="flex items-center rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 dark:border-primary-800/60 dark:bg-primary-950/80">
            <FiSearch className="opacity-70" />
            <input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              onKeyDown={(e) => e.key === "Enter" && applyFilters()}
              placeholder="Buscar por nombre, cargo, email, ubicación…"
              className="ml-2 w-72 bg-transparent text-sm outline-none placeholder-primary-400 dark:placeholder-primary-500"
            />
          </div>

          <select
            value={grupoKey}
            onChange={(e) => setGrupoKey(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Grupo: Todos</option>
            {grupoKeys?.map((gk) => (
              <option key={gk} value={gk}>
                {labels?.[gk] ?? gk}
              </option>
            ))}
          </select>

          <select
            value={area}
            onChange={(e) => setArea(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="">Área: Todas</option>
            {areas?.map((a) => (
              <option key={a} value={a}>
                {a}
              </option>
            ))}
          </select>

          <select
            value={trash}
            onChange={(e) => setTrash(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="none">Sin papelera</option>
            <option value="with">Incluir papelera</option>
            <option value="only">Solo papelera</option>
          </select>

          <select
            value={sort}
            onChange={(e) => setSort(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="orden">Orden: Orden</option>
            <option value="nombre">Orden: Nombre</option>
            <option value="grupo_key">Orden: Grupo</option>
            <option value="email">Orden: Email</option>
            <option value="created_at">Orden: Fecha de creación</option>
          </select>

          <select
            value={dir}
            onChange={(e) => setDir(e.target.value)}
            className="rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
          >
            <option value="asc">Ascendente</option>
            <option value="desc">Descendente</option>
          </select>

          <button
            onClick={applyFilters}
            className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700"
          >
            Aplicar
          </button>
        </div>

        {/* NUEVO */}
        {trash !== "only" && (
          <button
            onClick={openCreate}
            className="flex items-center justify-center gap-2 rounded-xl bg-gradient-to-r from-primary-600 to-secondary-600 px-4 py-2 text-sm font-semibold text-white shadow hover:opacity-90"
          >
            <FiPlus className="text-lg" /> Nuevo
          </button>
        )}
      </div>

      {/* Tabla */}
      <div className="overflow-x-auto rounded-2xl border border-primary-200/60 bg-white/90 shadow-sm backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/80">
        <table className="min-w-full divide-y divide-primary-200 dark:divide-primary-800">
          <thead className="bg-primary-50 dark:bg-primary-900/40">
            <tr className="text-sm text-primary-700 dark:text-primary-200">
              <th className="px-4 py-2 text-left">Slug</th>
              <th className="px-4 py-2 text-left">Nombre</th>
              <th className="px-4 py-2 text-left">Cargo</th>
              <th className="px-4 py-2 text-left">Grupo</th>
              <th className="px-4 py-2 text-left">Área</th>
              <th className="px-4 py-2 text-left">Contacto</th>
              <th className="px-4 py-2 text-left">Horario</th>
              <th className="px-4 py-2 text-left">Ubicación</th>
              <th className="px-4 py-2 text-center">Foto</th>
              <th className="px-4 py-2 text-center">Acciones</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-primary-100 dark:divide-primary-800">
            {autoridades.data.map((a) => (
              <tr key={a.id} className="text-sm hover:bg-primary-50/60 dark:hover:bg-primary-900/30">
                <td className="px-4 py-2 font-mono text-xs flex items-center gap-2">
                  <FiHash className="opacity-60" />
                  {a.slug}
                </td>
                <td className="px-4 py-2 font-medium flex items-center gap-2">
                  <FiUser className="opacity-60" />
                  {a.nombre}
                </td>
                <td className="px-4 py-2">{a.cargo}</td>
                <td className="px-4 py-2">
                  <span className="inline-flex items-center gap-1 rounded-lg bg-primary-50 px-2 py-0.5 text-xs font-semibold text-primary-700 dark:bg-primary-900/40 dark:text-primary-200">
                    {labels?.[a.grupo_key] ?? a.grupo ?? a.grupo_key}
                  </span>
                </td>
                <td className="px-4 py-2">{a.area || <span className="text-xs text-gray-400 italic">—</span>}</td>
                <td className="px-4 py-2">{contactoText(a)}</td>
                <td className="px-4 py-2">
                  <span className="inline-flex items-center gap-1">
                    <FiClock className="opacity-70" />
                    {horarioText(a)}
                  </span>
                </td>
                <td className="px-4 py-2">
                  <span className="inline-flex items-center gap-1">
                    <FiMapPin className="opacity-70" />
                    {a.ubicacion || <span className="text-xs text-gray-400 italic">—</span>}
                  </span>
                </td>
                <td className="px-4 py-2 text-center">
                  {a.foto ? (
                    <img
                      src={a.foto}
                      alt="foto"
                      className="mx-auto h-10 w-10 rounded-full object-cover ring-1 ring-primary-200/60 dark:ring-primary-800/60"
                    />
                  ) : (
                    <span className="inline-flex items-center gap-1 rounded-lg bg-primary-50 px-2 py-0.5 text-xs font-semibold text-primary-700 dark:bg-primary-900/40 dark:text-primary-200">
                      <FiImage /> Sin foto
                    </span>
                  )}
                </td>
                <td className="px-4 py-2">
                  {!a.deleted_at ? (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => openEdit(a)}
                        className="rounded-md p-1.5 text-primary-600 hover:bg-primary-100 dark:text-primary-300 dark:hover:bg-primary-900/40"
                        title="Editar"
                      >
                        <FiEdit />
                      </button>
                      <button
                        onClick={() => destroy(a.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar (papelera)"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  ) : (
                    <div className="flex items-center justify-center gap-2">
                      <button
                        onClick={() => restore(a.id)}
                        className="rounded-md p-1.5 text-green-600 hover:bg-green-100 dark:text-green-300 dark:hover:bg-green-900/40"
                        title="Restaurar"
                      >
                        <FiRefreshCw />
                      </button>
                      <button
                        onClick={() => forceDelete(a.id)}
                        className="rounded-md p-1.5 text-red-600 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900/40"
                        title="Eliminar definitivo"
                      >
                        <FiTrash2 />
                      </button>
                    </div>
                  )}
                </td>
              </tr>
            ))}
            {autoridades.data.length === 0 && (
              <tr>
                <td colSpan="10" className="px-4 py-6 text-center text-gray-400">
                  No hay autoridades para mostrar.
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Paginación */}
      <div className="mt-4 flex items-center justify-between text-sm">
        <div className="text-primary-700/80 dark:text-primary-200/80">
          Mostrando {autoridades.from ?? 0}–{autoridades.to ?? 0} de {autoridades.total ?? 0}
        </div>
        <div className="flex gap-2">
          <button
            disabled={!autoridades.prev_page_url}
            onClick={() => goto(autoridades.prev_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Anterior
          </button>
          <button
            disabled={!autoridades.next_page_url}
            onClick={() => goto(autoridades.next_page_url)}
            className="rounded-xl border border-primary-200/60 px-3 py-1 disabled:opacity-50 dark:border-primary-800/60"
          >
            Siguiente
          </button>
        </div>
      </div>

      {/* Modal Crear/Editar */}
      <Modal show={showModal} onClose={() => setShowModal(false)} maxWidth="4xl">
        <form onSubmit={submit} className="p-6 space-y-5">
          <h2 className="mb-2 text-lg font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-200 dark:to-secondary-300">
            {editing ? "Editar autoridad" : "Nueva autoridad"}
          </h2>

          {/* Básicos */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiHash /> Slug
              </label>
              <input
                type="text"
                placeholder="jefe-detri, coord-pregrado-telecom…"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80 font-mono"
                value={data.slug}
                onChange={(e) => setData("slug", e.target.value)}
              />
              {errors.slug && <p className="text-sm text-red-500">{errors.slug}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Nombre</label>
              <input
                type="text"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.nombre}
                onChange={(e) => setData("nombre", e.target.value)}
              />
              {errors.nombre && <p className="text-sm text-red-500">{errors.nombre}</p>}
            </div>

            <div className="sm:col-span-1">
              <label className="block text-sm font-semibold">Cargo</label>
              <input
                type="text"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.cargo}
                onChange={(e) => setData("cargo", e.target.value)}
              />
              {errors.cargo && <p className="text-sm text-red-500">{errors.cargo}</p>}
            </div>
          </div>

          {/* Grupo / Área */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div>
              <label className="block text-sm font-semibold">Grupo (clave)</label>
              <select
                value={data.grupo_key}
                onChange={(e) => onChangeGrupoKey(e.target.value)}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
              >
                <option value="">Selecciona…</option>
                {grupoKeys?.map((gk) => (
                  <option key={gk} value={gk}>
                    {labels?.[gk] ?? gk}
                  </option>
                ))}
              </select>
              {errors.grupo_key && <p className="text-sm text-red-500">{errors.grupo_key}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Grupo (etiqueta)</label>
              <input
                type="text"
                placeholder={data.grupo_key ? labels?.[data.grupo_key] : "p.ej. Coordinadores de Posgrado"}
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.grupo}
                onChange={(e) => setData("grupo", e.target.value)}
              />
              {errors.grupo && <p className="text-sm text-red-500">{errors.grupo}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold">Área</label>
              <input
                type="text"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.area}
                onChange={(e) => setData("area", e.target.value)}
              />
              {errors.area && <p className="text-sm text-red-500">{errors.area}</p>}
            </div>
          </div>

          {/* Contacto */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div>
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiMail /> Email
              </label>
              <input
                type="email"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.email}
                onChange={(e) => setData("email", e.target.value)}
              />
              {errors.email && <p className="text-sm text-red-500">{errors.email}</p>}
            </div>

            <div>
              <label className="block text-sm font-semibold flex items-center gap-2">
                <FiPhone /> Teléfono
              </label>
              <input
                type="text"
                className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                value={data.telefono}
                onChange={(e) => setData("telefono", e.target.value)}
              />
              {errors.telefono && <p className="text-sm text-red-500">{errors.telefono}</p>}
            </div>

            <div className="grid grid-cols-2 gap-3">
              <div>
                <label className="block text-sm font-semibold">Horario (días)</label>
                <input
                  type="text"
                  placeholder="Lunes a Viernes"
                  className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                  value={data.horario_dias}
                  onChange={(e) => setData("horario_dias", e.target.value)}
                />
              </div>
              <div>
                <label className="block text-sm font-semibold">Horario (rango)</label>
                <input
                  type="text"
                  placeholder="09h00 – 12h00"
                  className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
                  value={data.horario_rango}
                  onChange={(e) => setData("horario_rango", e.target.value)}
                />
              </div>
            </div>
          </div>

          {/* Ubicación */}
          <div>
            <label className="block text-sm font-semibold flex items-center gap-2">
              <FiMapPin /> Ubicación
            </label>
            <input
              type="text"
              placeholder="Edificio ESFOT, Oficina 301"
              value={data.ubicacion}
              onChange={(e) => setData("ubicacion", e.target.value)}
              className="mt-1 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm dark:border-primary-800/60 dark:bg-primary-950/80"
            />
            {errors.ubicacion && <p className="text-sm text-red-500">{errors.ubicacion}</p>}
          </div>

          {/* Foto */}
          <div>
            <label className="block text-sm font-semibold flex items-center gap-2">
              <FiImage /> Foto
            </label>

            {/* Actual */}
            {(data.foto || fotoPreview) && (
              <div className="mt-2 flex items-center gap-3">
                <img
                  src={fotoPreview || data.foto}
                  alt="foto"
                  className="h-16 w-16 rounded-full object-cover ring-1 ring-primary-200/60 dark:ring-primary-800/60"
                />
                <button
                  type="button"
                  onClick={clearFoto}
                  className="rounded-md bg-red-600 px-3 py-1.5 text-xs font-semibold text-white hover:bg-red-700"
                >
                  Quitar foto
                </button>
              </div>
            )}

            {/* Nueva */}
            <input
              type="file"
              accept="image/*"
              onChange={onFotoFile}
              className="mt-2 w-full rounded-xl border border-primary-200/60 bg-white/90 px-3 py-2 text-sm file:mr-3 file:rounded-lg file:border-0 file:bg-primary-100 file:px-3 file:py-1.5 dark:border-primary-800/60 dark:bg-primary-950/80"
            />
            {errors.foto_file && <p className="text-sm text-red-500">{errors.foto_file}</p>}
          </div>

          {/* Footer modal */}
          <div className="mt-2 flex justify-end gap-2">
            <button
              type="button"
              onClick={() => setShowModal(false)}
              className="rounded-xl border border-primary-200/60 px-4 py-2 text-sm dark:border-primary-800/60"
            >
              Cancelar
            </button>
            <button
              type="submit"
              disabled={processing}
              className="rounded-xl bg-primary-600 px-4 py-2 text-sm font-semibold text-white transition hover:bg-primary-700 disabled:opacity-60"
            >
              {processing ? "Guardando..." : editing ? "Actualizar" : "Crear"}
            </button>
          </div>
        </form>
      </Modal>
    </DashboardLayout>
  );
}
