// resources/js/Layouts/DashboardLayout.jsx
import React, { useMemo, useState, useRef, useEffect } from "react";
import { Head, Link } from "@inertiajs/react";
import {
  FiMenu,
  FiX,
  FiChevronDown,
  FiLogOut,
  FiUser,
  FiGrid,
  FiHome,
} from "react-icons/fi";

/* ===== Menú por rol (rutas directas) ===== */
const MENU_BY_ROLE = {
  Administrador: [
    { label: "Inicio", icon: <FiHome />, href: "/admin/dashboard" },
    { label: "Usuarios", icon: <FiGrid />, href: "/admin/usuarios" },
    { label: "Docentes", icon: <FiGrid />, href: "/admin/docentes" },
    { label: "Autoridades", icon: <FiGrid />, href: "/admin/autoridades" },
    { label: "Laboratorios", icon: <FiGrid />, href: "/admin/laboratorios" },
    { label: "Proyectos", icon: <FiGrid />, href: "/admin/proyectos" },
    { label: "Pregrado", icon: <FiGrid />, href: "/admin/pregrados" },
    { label: "Posgrado", icon: <FiGrid />, href: "/admin/postgrados" },
    { label: "Lineas de Investigación", icon: <FiGrid />, href: "/admin/lineas_investigacion" },
    { label: "Pasantias", icon: <FiGrid />, href: "/admin/pasantias" },
    { label: "Publicaciones", icon: <FiGrid />, href: "/admin/publicaciones" },
    //Trabajos de Integracion curricular
    { label: "Trabajos de Integración Curricular", icon: <FiGrid />, href: "/admin/tic" },
  ],
  Secretaria: [
    { label: "Inicio", icon: <FiHome />, href: "/dashboard" },
    { label: "Matrículas", icon: <FiGrid />, href: "/secretaria/matriculas" },
    { label: "Certificados", icon: <FiGrid />, href: "/secretaria/certificados" },
    { label: "Agenda", icon: <FiGrid />, href: "/secretaria/agenda" },
  ],
  Docente: [
    { label: "Inicio", icon: <FiHome />, href: "/dashboard" },
    { label: "Yo, Docente", icon: <FiGrid />, href: "/docente/perfil" },
  ],
};

/* ===== Helpers de ruta activa (sin Ziggy) ===== */
const getCurrentPath = () =>
  typeof window !== "undefined"
    ? window.location.pathname.replace(/\/+$/, "") || "/"
    : "/";

const isActiveHref = (href) => {
  const path = getCurrentPath();
  const base = href.replace(/\/+$/, "") || "/";
  if (base === "/") return path === "/";
  return path === base || path.startsWith(base + "/");
};

/* ===== Link lateral con estado activo ===== */
function SideLink({ item }) {
  const active = isActiveHref(item.href);

  return (
    <Link
      href={item.href}
      aria-current={active ? "page" : undefined}
      className={`group relative flex items-center gap-3 rounded-xl px-3 py-2 text-sm transition-all
        ${
          active
            ? "bg-primary-50 text-primary-700 dark:bg-primary-900/40 dark:text-primary-200"
            : "text-primary-900/80 hover:bg-primary-50 hover:text-primary-700 dark:text-primary-100/80 dark:hover:bg-primary-900/30 dark:hover:text-primary-100"
        }`}
    >
      <span
        className={`absolute left-0 top-1/2 -translate-y-1/2 h-6 w-1 rounded-r
          ${
            active
              ? "bg-gradient-to-b from-primary-500 to-secondary-500"
              : "bg-transparent group-hover:bg-gradient-to-b group-hover:from-primary-300 group-hover:to-secondary-300"
          }`}
      />
      <span className="text-base opacity-90 group-hover:opacity-100">
        {item.icon}
      </span>
      <span className="font-medium">{item.label}</span>
      <span className="ml-auto h-0.5 w-0 bg-gradient-to-r from-primary-500 to-secondary-500 transition-all group-hover:w-4" />
    </Link>
  );
}

/* ===== Layout principal ===== */
export default function DashboardLayout({ auth, titulo, children }) {
  const user = auth?.user ?? {};
  const role = user?.role || "Docente";
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [userOpen, setUserOpen] = useState(false);
  const userRef = useRef(null);

  useEffect(() => {
    const onDoc = (e) => {
      if (userRef.current && !userRef.current.contains(e.target)) setUserOpen(false);
    };
    document.addEventListener("mousedown", onDoc);
    return () => document.removeEventListener("mousedown", onDoc);
  }, []);

  const menuItems = useMemo(
    () => MENU_BY_ROLE[role] ?? MENU_BY_ROLE["Docente"],
    [role]
  );

  return (
    <div className="min-h-screen bg-white text-primary-900 dark:bg-primary-950 dark:text-primary-50">
      <Head title={titulo} />

      {/* Topbar */}
      <header className="sticky top-0 z-40 border-b border-primary-200/60 bg-white/95 backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/90">
        <div
          className="h-1 bg-gradient-to-r from-primary-500 via-secondary-500 to-primary-600"
          style={{
            backgroundSize: "200% 100%",
            animation: "gradient-x 3s ease infinite",
          }}
        />
        <div className="mx-auto flex h-14 max-w-7xl items-center justify-between px-4 sm:px-6 lg:px-8">
          {/* Izquierda */}
          <div className="flex items-center gap-3">
            <button
              className="lg:hidden rounded-xl p-2 text-primary-900 hover:bg-primary-100/60 dark:text-primary-100 dark:hover:bg-primary-900/40"
              onClick={() => setSidebarOpen(true)}
              aria-label="Abrir menú"
            >
              <FiMenu className="h-5 w-5" />
            </button>
            <Link href="/" className="flex items-center gap-2">
              <img src="/assets/images/logo.png" alt="DETRI" className="h-8 w-8" />
              <span className="hidden font-bold tracking-tight sm:inline bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-300 dark:to-secondary-400">
                DETRI
              </span>
              <span className="ml-2 rounded-full bg-primary-100 px-2 py-0.5 text-xs font-semibold text-primary-700 dark:bg-primary-900/50 dark:text-primary-200">
                {role}
              </span>
            </Link>
          </div>

          {/* Usuario */}
          <div className="relative" ref={userRef}>
            <button
              onClick={() => setUserOpen((v) => !v)}
              className="flex items-center gap-2 rounded-full pl-2 pr-3 py-1.5 bg-white/70 dark:bg-primary-900/50 border border-primary-200/60 dark:border-primary-800/60 shadow-sm hover:shadow transition-all"
              aria-haspopup="menu"
              aria-expanded={userOpen}
            >
              <img
                src={
                  user.avatar_url ??
                  `https://ui-avatars.com/api/?background=0D8ABC&color=fff&name=${encodeURIComponent(
                    user.name || "U"
                  )}`
                }
                alt={user.name || "Usuario"}
                className="h-8 w-8 rounded-full object-cover"
              />
              <span className="hidden max-w-[10rem] truncate text-sm sm:inline">
                {user.name || "Usuario"}
              </span>
              <FiChevronDown
                className={`transition-transform ${userOpen ? "rotate-180" : ""}`}
              />
            </button>

            <div
              className={`absolute right-0 mt-2 w-64 transition-all duration-150 ${
                userOpen
                  ? "opacity-100 translate-y-0"
                  : "pointer-events-none -translate-y-1 opacity-0"
              }`}
            >
              <div className="overflow-hidden rounded-xl border border-primary-200/60 bg-white shadow-xl dark:border-primary-800/60 dark:bg-primary-950">
                <div className="px-4 py-3">
                  <p className="truncate text-sm font-semibold">{user.email}</p>
                </div>
                <div className="h-[1px] bg-gradient-to-r from-primary-500 via-secondary-500 to-primary-600" />
                <ul className="py-1 text-sm">
                  <li>
                    <Link
                      href="/perfil"
                      className="flex items-center gap-2 px-4 py-2 hover:bg-primary-50 dark:hover:bg-primary-900/40"
                    >
                      <FiUser /> Perfil
                    </Link>
                  </li>
                  <li>
                    <Link
                      href="/dashboard"
                      className="flex items-center gap-2 px-4 py-2 hover:bg-primary-50 dark:hover:bg-primary-900/40"
                    >
                      <FiGrid /> Panel
                    </Link>
                  </li>
                  <li className="mt-1 border-t border-primary-200/60 dark:border-primary-800/60">
                    <Link
                      href={route("logout")}
                      method="post"
                      as="button"
                      className="flex w-full items-center gap-2 px-4 py-2 text-left hover:bg-primary-50 dark:hover:bg-primary-900/40"
                    >
                      <FiLogOut /> Cerrar sesión
                    </Link>
                  </li>
                </ul>
              </div>
            </div>
          </div>
        </div>
      </header>

      {/* Cuerpo */}
      <div className="mx-auto grid max-w-screen-3xl grid-cols-1 gap-0 px-4 sm:px-6 lg:grid-cols-[260px_1fr] lg:px-8">
        {/* Sidebar (desktop) */}
        <aside className="sticky top-14 hidden h-[calc(100vh-3.5rem)] lg:block">
          <div className="mt-4 rounded-2xl border border-primary-200/60 bg-white/90 p-3 shadow-lg backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/80">
            <div className="mb-2 h-1 w-full bg-gradient-to-r from-primary-500 via-secondary-500 to-primary-600" />
            <nav className="space-y-1">
              {menuItems.map((item) => (
                <SideLink key={item.href} item={item} />
              ))}
            </nav>
          </div>
        </aside>

        {/* Contenido */}
        <main className="py-6 lg:py-8 px-0 lg:px-6">
          {titulo && (
            <div className="mb-4">
              <h1 className="text-xl font-extrabold tracking-tight bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent dark:from-primary-200 dark:to-secondary-300">
                {titulo}
              </h1>
            </div>
          )}
          <div className="rounded-2xl border border-primary-200/60 bg-white/90 p-4 shadow-sm backdrop-blur dark:border-primary-800/60 dark:bg-primary-950/80">
            {children}
          </div>
        </main>
      </div>

      {/* Sidebar móvil */}
      <div
        className={`fixed inset-0 z-50 lg:hidden transition-all duration-300 ${
          sidebarOpen ? "pointer-events-auto" : "pointer-events-none"
        }`}
      >
        {/* Fondo */}
        <div
          className={`absolute inset-0 bg-black/30 backdrop-blur-sm transition-opacity duration-300 ${
            sidebarOpen ? "opacity-100" : "opacity-0"
          }`}
          onClick={() => setSidebarOpen(false)}
        />

        {/* Panel lateral */}
        <aside
          className={`absolute left-0 top-0 h-full w-[80%] max-w-[260px]
                      bg-white dark:bg-primary-950 shadow-2xl border-r
                      border-primary-200/60 dark:border-primary-800/60
                      transform transition-transform duration-300 ease-out
                      ${sidebarOpen ? "translate-x-0" : "-translate-x-full"}`}
        >
          <div className="flex items-center justify-between px-4 py-3 border-b border-primary-200/60 dark:border-primary-800/60">
            <div className="flex items-center gap-2">
              <img src="/assets/images/logo.png" alt="DETRI" className="h-8 w-8" />
              <span className="font-bold bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent">
                DETRI
              </span>
            </div>
            <button
              onClick={() => setSidebarOpen(false)}
              className="rounded-xl p-2 text-primary-700 hover:bg-primary-100 dark:text-primary-100 dark:hover:bg-primary-900/40"
              aria-label="Cerrar menú"
            >
              <FiX className="h-5 w-5" />
            </button>
          </div>

          <nav className="p-4 space-y-1">
            {menuItems.map((item) => (
              <Link
                key={item.href}
                href={item.href}
                onClick={() => setSidebarOpen(false)}
                className={`flex items-center gap-3 rounded-xl px-3 py-2 text-sm transition-all
                  ${
                    isActiveHref(item.href)
                      ? "bg-primary-50 text-primary-700 dark:bg-primary-900/40 dark:text-primary-200"
                      : "text-primary-900/80 hover:bg-primary-50 hover:text-primary-700 dark:text-primary-100/80 dark:hover:bg-primary-900/30 dark:hover:text-primary-100"
                  }`}
              >
                {item.icon}
                <span className="font-medium">{item.label}</span>
              </Link>
            ))}
          </nav>
        </aside>
      </div>

      {/* Animación gradiente */}
      <style>{`
        @keyframes gradient-x {
          0%, 100% { background-position: 0% 50%; }
          50% { background-position: 100% 50%; }
        }
      `}</style>
    </div>
  );
}
