import React, { useEffect, useRef, useState } from 'react';
import { Link, usePage } from '@inertiajs/react';
import { FiChevronDown, FiMenu, FiX, FiUser, FiLogIn, FiGrid, FiLogOut } from 'react-icons/fi';

/* ========= NAV DATA ORIGINAL ========= */
const ES = [
  {
    label: "DETRI",
    link: "#",
    subItems: [
      { label: "Misión", link: "detri.mision" },
      { label: "Autoridades", link: "detri.autoridades" },
      { label: "Personal Docente", link: "detri.personal-docente" },
      { label: "Ubicación", link: "detri.ubicacion" },
    ]
  },
  {
    label: "OFERTA ACADÉMICA",
    link: "#",
    subItems: [
      { label: "Grado", link: "oferta.grado" },
      { label: "Posgrado", link: "oferta.posgrado" },
    ]
  },
  {
    label: "INVESTIGACIÓN",
    link: "#",
    subItems: [
      { label: "Centros y Laboratorios", link: "investigacion.centros-laboratorios" },
      { label: "Proyectos", link: "investigacion.proyectos-investigacion" },
      { label: "Publicaciones", link: "investigacion.publicaciones" },
      { label: "Líneas de Investigación", link: "investigacion.lineas-investigacion" },
    ]
  },
  {
    label: "VIDA ESTUDIANTIL",
    link: "#",
    subItems: [
      { label: "Pasantías", link: "vida-estudiantil.pasantias" },
      { label: "TIC", link: "vida-estudiantil.tic" },
    ]
  },
];

/* ========= Helpers ========= */
function resolveHref(nameOrUrl) {
  if (!nameOrUrl) return '#';
  if (nameOrUrl.startsWith('/') || nameOrUrl.startsWith('http')) return nameOrUrl;

  try {
    if (typeof window !== 'undefined' && window.route && typeof window.route === 'function') {
      return window.route(nameOrUrl);
    }
  } catch (_) {}

  return '/' + nameOrUrl.replace(/\.+/g, '/');
}

const NAV = [
  ...ES.map(g => ({
    label: g.label,
    children: (g.subItems || []).map(s => ({ label: s.label, href: resolveHref(s.link) })),
  })),
];

export default function SiteNavbar() {
  const { url, props } = usePage();
  const user = props?.auth?.user ?? null;

  const [openMobile, setOpenMobile] = useState(false);
  const [openIndex, setOpenIndex] = useState(null);
  const [openAccordions, setOpenAccordions] = useState({});
  const [scrolled, setScrolled] = useState(false);
  const [userOpen, setUserOpen] = useState(false);

  const navRef = useRef(null);
  const userRef = useRef(null);

  useEffect(() => {
    const handleScroll = () => setScrolled(window.scrollY > 20);
    handleScroll();
    window.addEventListener('scroll', handleScroll, { passive: true });
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);

  useEffect(() => {
    function onClick(e) {
      if (navRef.current && !navRef.current.contains(e.target)) setOpenIndex(null);
      if (userRef.current && !userRef.current.contains(e.target)) setUserOpen(false);
    }
    document.addEventListener('mousedown', onClick);
    return () => document.removeEventListener('mousedown', onClick);
  }, []);

  const toggleAccordion = (i) => setOpenAccordions((p) => ({ ...p, [i]: !p[i] }));
  const isActive = (href) => href && url === href;

  /* Rutas de auth */
  const loginHref = resolveHref('login');
  const profileHref = resolveHref('profile.edit'); // ajusta si tu ruta es distinta
          // POST

          /* Dashboard según tipo de usuario */
let dashboardHref = '';

if (user) {
  switch ((user.role || '')) {
    case 'Administrador':
      dashboardHref = resolveHref('admin.index');
      break;
    case 'Docente':
      dashboardHref = resolveHref('docente.index');
      break;
    case 'Secretaria':
      dashboardHref = resolveHref('secretaria.index');
      break;

    default:
      dashboardHref = resolveHref('dashboard');
  }
}

  return (
    <>
      <header
        ref={navRef}
        className={`fixed top-0 left-0 right-0 z-40 transition-all duration-500
          ${scrolled
            ? 'bg-white/95 dark:bg-primary-950/95 backdrop-blur-xl shadow-lg shadow-primary-500/10'
            : 'bg-white/80 dark:bg-primary-950/80 backdrop-blur-md'
          }
          border-b ${scrolled ? 'border-primary-200/50' : 'border-primary-100/30'}
          dark:border-primary-800/30`}
        role="banner"
      >
        {/* Barra superior animada */}
        <div
          className="h-1 bg-gradient-to-r from-primary-500 via-secondary-500 to-primary-600 animate-gradient-x"
          style={{ backgroundSize: '200% 100%' }}
        />

        <nav className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8" aria-label="Primary">
          <div className={`flex items-center justify-between gap-3 transition-all duration-300 ${scrolled ? 'h-14' : 'h-16'}`}>

            {/* Logo */}
            <Link href="/" className="flex items-center gap-3 group relative">
              <div className="relative">
                <img
                  src="/assets/images/logo.png"
                  alt="Logo DETRI"
                  className={`transition-all duration-300 group-hover:scale-110 group-hover:rotate-6
                    ${scrolled ? 'h-9 w-9' : 'h-11 w-11'}`}
                />
                <div className="absolute inset-0 bg-primary-400/20 rounded-full blur-lg opacity-0 group-hover:opacity-100 transition-opacity duration-300" />
              </div>
              <span className={`font-bold bg-gradient-to-r from-primary-700 to-secondary-600 bg-clip-text text-transparent
                dark:from-primary-300 dark:to-secondary-400 tracking-tight transition-all duration-300
                ${scrolled ? 'text-lg' : 'text-xl'}`}>
                DETRI
              </span>
            </Link>

            {/* Desktop nav */}
            <div className="hidden lg:flex items-center gap-1">
              {NAV.map((item, i) =>
                item.children ? (
                  <div
                    key={item.label}
                    className="relative group"
                    onMouseEnter={() => setOpenIndex(i)}
                    onMouseLeave={() => setOpenIndex((idx) => (idx === i ? null : idx))}
                  >
                    <button
                      className={`inline-flex items-center gap-1.5 px-3.5 py-2 rounded-xl text-sm font-medium
                        transition-all duration-300 relative overflow-hidden group
                        ${openIndex === i
                          ? 'text-primary-700 dark:text-primary-300'
                          : 'text-primary-900/80 dark:text-primary-100/80 hover:text-primary-700 dark:hover:text-primary-300'
                        }`}
                      aria-expanded={openIndex === i}
                    >
                      <span className="absolute inset-0 bg-gradient-to-r from-primary-50 to-secondary-50
                        dark:from-primary-900/30 dark:to-secondary-900/30 opacity-0 group-hover:opacity-100 transition-opacity duration-300" />
                      <span className="relative z-10">{item.label}</span>
                      <FiChevronDown className={`relative z-10 transition-transform duration-300 ${openIndex === i ? 'rotate-180' : 'group-hover:translate-y-0.5'}`} />
                    </button>

                    <div
                      className={`absolute left-1/2 -translate-x-1/2 pt-3 transition-all duration-300
                        ${openIndex === i ? 'opacity-100 translate-y-0 pointer-events-auto' : 'opacity-0 -translate-y-2 pointer-events-none'}`}
                      role="menu"
                      aria-label={item.label}
                    >
                      <div className="relative min-w-64 rounded-2xl bg-white dark:bg-primary-950 shadow-2xl shadow-primary-500/20 dark:shadow-primary-900/40 border border-primary-200/50 dark:border-primary-800/50 overflow-hidden backdrop-blur-xl">
                        <div className="h-1 bg-gradient-to-r from-primary-500 via-secondary-500 to-primary-600" />
                        <ul className="py-2">
                          {item.children.map((sub, idx) => (
                            <li key={sub.label} style={{ animationDelay: `${idx * 50}ms`, animation: openIndex === i ? 'slideInLeft 0.3s ease-out forwards' : 'none' }}>
                              <Link
                                href={sub.href}
                                className="group/item flex items-center justify-between px-4 py-3 text-sm
                                  text-primary-900/80 dark:text-primary-100/80 hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50
                                  dark:hover:from-primary-900/50 dark:hover:to-secondary-900/50 hover:text-primary-700 dark:hover:text-primary-300
                                  transition-all duration-200 relative overflow-hidden"
                                role="menuitem"
                              >
                                <span className="relative z-10 font-medium">{sub.label}</span>
                                <span className="w-0 h-0.5 bg-gradient-to-r from-primary-500 to-secondary-500 group-hover/item:w-4 transition-all duration-300 ml-2" />
                              </Link>
                            </li>
                          ))}
                        </ul>
                      </div>
                    </div>
                  </div>
                ) : (
                  <Link
                    key={item.label}
                    href={item.href}
                    className={`px-3.5 py-2 rounded-xl text-sm font-medium transition-all duration-300 relative overflow-hidden group
                      ${isActive(item.href) ? 'text-primary-700 dark:text-primary-300' : 'text-primary-900/80 dark:text-primary-100/80 hover:text-primary-700 dark:hover:text-primary-300'}`}
                  >
                    <span className="absolute inset-0 bg-gradient-to-r from-primary-50 to-secondary-50 dark:from-primary-900/30 dark:to-secondary-900/30 opacity-0 group-hover:opacity-100 transition-opacity duration-300" />
                    <span className="relative z-10">{item.label}</span>
                  </Link>
                )
              )}
            </div>

            {/* === ACCESO / USUARIO (Desktop) === */}
            <div className="hidden lg:flex items-center">
              {!user ? (
                <Link
                  href={loginHref}
                  className="relative group inline-flex items-center gap-2 rounded-xl px-4 py-2 text-sm font-semibold
                    text-primary-800 dark:text-primary-100 transition-all
                    focus:outline-none focus-visible:ring-2 focus-visible:ring-secondary-400"
                >
                  {/* borde degradado distintivo */}
                  <span className="absolute inset-0 rounded-xl p-[1px] bg-gradient-to-r from-primary-500 via-secondary-500 to-primary-600" />
                  <span className="absolute inset-[1px] rounded-[11px] bg-white/90 dark:bg-primary-950/90 backdrop-blur" />
                  <FiLogIn className="relative z-10" />
                  {/* shimmer */}
                  <span className="pointer-events-none absolute -inset-1 rounded-xl opacity-0 group-hover:opacity-100 transition-opacity duration-500"
                        style={{ background: 'radial-gradient(120px 60px at var(--x,80%) 50%, rgba(255,255,255,0.25), transparent 55%)' }}/>
                </Link>
              ) : (
                <div className="relative" ref={userRef}>
                  <button
                    onClick={() => setUserOpen((v) => !v)}
                    className="flex items-center gap-2 rounded-full pl-2 pr-3 py-1.5 bg-white/70 dark:bg-primary-900/50
                      border border-primary-200/60 dark:border-primary-800/60 shadow-sm hover:shadow transition-all"
                    aria-haspopup="menu"
                    aria-expanded={userOpen}
                  >
                    <img
                      src={user.avatar_url ?? `https://ui-avatars.com/api/?background=0D8ABC&color=fff&name=${encodeURIComponent(user.name || 'U')}`}
                      alt={user.name || 'Usuario'}
                      className="h-8 w-8 rounded-full object-cover"
                    />
                    <span className="max-w-[10rem] truncate text-sm text-primary-900 dark:text-primary-100">{user.name || 'Usuario'}</span>
                    <FiChevronDown className={`transition-transform ${userOpen ? 'rotate-180' : ''}`} />
                  </button>

                  {/* dropdown usuario */}
                  <div
                    className={`absolute right-0 mt-2 w-60 transition-all duration-200 ${userOpen ? 'opacity-100 translate-y-0' : 'opacity-0 -translate-y-1 pointer-events-none'}`}
                    role="menu"
                  >
                    <div className="overflow-hidden rounded-xl border border-primary-200/60 dark:border-primary-800/60 bg-white dark:bg-primary-950 shadow-xl">
                      <div className="px-4 py-3">
                        <p className="text-sm font-semibold text-primary-900 dark:text-primary-100 truncate">{user.email}</p>
                      </div>
                      <div className="h-[1px] bg-gradient-to-r from-primary-500 via-secondary-500 to-primary-600" />
                      <ul className="py-1 text-sm">
                        <li>
                          <Link href={dashboardHref} className="flex items-center gap-2 px-4 py-2 hover:bg-primary-50 dark:hover:bg-primary-900/40">
                            <FiGrid /> Panel
                          </Link>
                        </li>
                        <li>
                          <Link href='/perfil' className="flex items-center gap-2 px-4 py-2 hover:bg-primary-50 dark:hover:bg-primary-900/40">
                            <FiUser /> Perfil
                          </Link>
                        </li>
                        <li className="mt-1 border-t border-primary-200/60 dark:border-primary-800/60">

                            {/* Enertia/Laravel: csrf si lo necesitas:
                                <input type="hidden" name="_token" value={props.csrf_token} />
                             */}
                             <Link href={route("logout")} method="post" as='button' className="w-full text-left flex items-center gap-2 px-4 py-2 hover:bg-primary-50 dark:hover:bg-primary-900/40">
                              <FiLogOut /> Cerrar sesión
                            </Link>

                        </li>
                      </ul>
                    </div>
                  </div>
                </div>
              )}
            </div>

            {/* Mobile toggle */}
            <button
              className="lg:hidden relative p-2.5 rounded-xl overflow-hidden group
                text-primary-900 dark:text-primary-100 transition-all duration-300 hover:scale-110"
              aria-label="Abrir menú"
              onClick={() => setOpenMobile(true)}
            >
              <span className="absolute inset-0 bg-gradient-to-r from-primary-100 to-secondary-100
                dark:from-primary-900/50 dark:to-secondary-900/50 opacity-0 group-hover:opacity-100 transition-opacity duration-300" />
              <FiMenu className="relative z-10 w-5 h-5" />
            </button>
          </div>
        </nav>
      </header>

      {/* Spacer */}
      <div className={`transition-all duration-300 ${scrolled ? 'h-[57px]' : 'h-[69px]'}`} />

      {/* Mobile menu */}
      <div className={`lg:hidden fixed inset-0 z-50 transition-all duration-300 ${openMobile ? 'pointer-events-auto' : 'pointer-events-none'}`} aria-hidden={!openMobile}>
        <div
          className={`absolute inset-0 backdrop-blur-sm bg-gradient-to-br from-primary-950/60 via-secondary-950/60 to-primary-950/60
            transition-opacity duration-300 ${openMobile ? 'opacity-100' : 'opacity-0'}`}
          onClick={() => setOpenMobile(false)}
        />
        <aside
          className={`absolute left-0 top-0 h-full w-[85%] max-w-sm bg-white dark:bg-primary-950 shadow-2xl shadow-primary-500/20
            transition-transform duration-500 ease-out ${openMobile ? 'translate-x-0' : '-translate-x-full'}`}
          role="dialog"
          aria-label="Menú de navegación"
        >
          {/* Header panel */}
          <div className="relative flex items-center justify-between h-16 px-5 bg-gradient-to-r from-primary-600 to-secondary-600 shadow-lg">
            <Link href="/" className="flex items-center gap-3 group" onClick={() => setOpenMobile(false)}>
              <img src="/assets/images/logo.png" alt="Logo DETRI" className="h-10 w-10 transition-transform duration-300 group-hover:rotate-12" />
              <span className="font-bold text-white text-lg tracking-tight">DETRI</span>
            </Link>
            <button
              className="p-2 rounded-xl text-white hover:bg-white/20 transition-all duration-300 hover:rotate-90"
              aria-label="Cerrar menú"
              onClick={() => setOpenMobile(false)}
            >
              <FiX className="w-5 h-5" />
            </button>
          </div>

          {/* Navegación */}
          <div className="overflow-y-auto h-[calc(100%-4rem)] px-3 pt-3 pb-28">
            {NAV.map((item, i) =>
              item.children ? (
                <div key={item.label} className="mb-2" style={{ animationDelay: `${i * 80}ms`, animation: openMobile ? 'slideInLeft 0.4s ease-out forwards' : 'none' }}>
                  <button
                    className="w-full flex items-center justify-between px-4 py-3.5 rounded-xl text-left font-semibold text-primary-900 dark:text-primary-100
                      hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50 dark:hover:from-primary-900/30 dark:hover:to-secondary-900/30 transition-all duration-300 group"
                    aria-expanded={!!openAccordions[i]}
                    onClick={() => toggleAccordion(i)}
                  >
                    <span>{item.label}</span>
                    <FiChevronDown className={`transition-all duration-300 ${openAccordions[i] ? 'rotate-180 text-primary-600' : 'group-hover:translate-y-0.5'}`} />
                  </button>
                  <div className={`grid overflow-hidden transition-all duration-300 ${openAccordions[i] ? 'grid-rows-[1fr] opacity-100 mt-1' : 'grid-rows-[0fr] opacity-0'}`}>
                    <ul className="min-h-0 space-y-1">
                      {item.children.map((sub) => (
                        <li key={sub.label}>
                          <Link
                            href={sub.href}
                            onClick={() => setOpenMobile(false)}
                            className={`block px-6 py-2.5 text-sm rounded-lg transition-all duration-200 relative overflow-hidden group/sub
                              ${isActive(sub.href) ? 'text-primary-700 dark:text-primary-300 font-medium bg-primary-50 dark:bg-primary-900/30'
                                : 'text-primary-900/70 dark:text-primary-100/70 hover:text-primary-700 dark:hover:text-primary-300'}`}
                          >
                            <span className="absolute left-0 top-1/2 -translate-y-1/2 w-1 h-0 bg-gradient-to-b from-primary-500 to-secondary-500 group-hover/sub:h-full transition-all duration-300" />
                            <span className="relative z-10 ml-2">{sub.label}</span>
                          </Link>
                        </li>
                      ))}
                    </ul>
                  </div>
                </div>
              ) : (
                <Link
                  key={item.label}
                  href={item.href}
                  onClick={() => setOpenMobile(false)}
                  className={`block px-4 py-3.5 rounded-xl font-semibold mb-2 transition-all duration-300 relative overflow-hidden group
                    ${isActive(item.href) ? 'text-primary-700 dark:text-primary-300 bg-primary-50 dark:bg-primary-900/30'
                      : 'text-primary-900 dark:text-primary-100 hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50 dark:hover:from-primary-900/30 dark:hover:to-secondary-900/30'}`}
                  style={{ animationDelay: `${i * 80}ms`, animation: openMobile ? 'slideInLeft 0.4s ease-out forwards' : 'none' }}
                >
                  <span className="relative z-10">{item.label}</span>
                </Link>
              )
            )}
          </div>

          {/* === ACCESO / USUARIO (Mobile, bottom bar) === */}
          <div className="absolute bottom-0 left-0 right-0 p-4 bg-white/85 dark:bg-primary-950/85 backdrop-blur border-t border-primary-200/60 dark:border-primary-800/60">
            {!user ? (
              <Link
                href={loginHref}
                onClick={() => setOpenMobile(false)}
                className="relative group flex items-center justify-center gap-2 rounded-xl px-4 py-3 text-sm font-semibold
                  text-primary-800 dark:text-primary-100 focus:outline-none focus-visible:ring-2 focus-visible:ring-secondary-400"
              >
                <span className="absolute inset-0 rounded-xl p-[1px] bg-gradient-to-r from-primary-500 via-secondary-500 to-primary-600" />
                <span className="absolute inset-[1px] rounded-[11px] bg-white/95 dark:bg-primary-950/95 backdrop-blur" />
                <FiLogIn className="relative z-10" />
                <span className="relative z-10">Acceder</span>
              </Link>
            ) : (
              <div className="flex items-center justify-between gap-3">
                <div className="flex items-center gap-3">
                  <img
                    src={user.avatar_url ?? `https://ui-avatars.com/api/?background=0D8ABC&color=fff&name=${encodeURIComponent(user.name || 'U')}`}
                    alt={user.name || 'Usuario'}
                    className="h-9 w-9 rounded-full object-cover"
                  />
                  <div className="min-w-0">
                    <p className="text-sm font-semibold text-primary-900 dark:text-primary-100 truncate">{user.name}</p>
                    <p className="text-xs text-primary-700/70 dark:text-primary-200/70 truncate">{user.email}</p>
                  </div>
                </div>
                <div className="flex items-center gap-2">
                  <Link href={dashboardHref} onClick={() => setOpenMobile(false)} className="px-3 py-2 rounded-lg bg-primary-50 dark:bg-primary-900/40 text-sm font-medium">
                    Panel
                  </Link>
                  <Link href={route("logout")} method='post' as='button' className='px-3 py-2 rounded-lg bg-primary-50 dark:bg-primary-900/40 text-sm font-medium flex items-center gap-1.5'>
                    <FiLogOut />
                    <span>Salir</span>
                  </Link>
                </div>
              </div>
            )}
          </div>
        </aside>
      </div>

      <style>{`
        @keyframes gradient-x {
          0%, 100% { background-position: 0% 50%; }
          50% { background-position: 100% 50%; }
        }
        @keyframes slideInLeft {
          from { opacity: 0; transform: translateX(-20px); }
          to { opacity: 1; transform: translateX(0); }
        }
        .animate-gradient-x { animation: gradient-x 3s ease infinite; }
      `}</style>
    </>
  );
}
