import React, { useEffect, useState } from "react";
import { Link } from "@inertiajs/react";
import {
  FiFacebook,
  FiInstagram,
  FiLinkedin,
  FiYoutube,
  FiMail,
  FiArrowUp,
  FiPhone,
  FiMapPin,
} from "react-icons/fi";

export default function FooterDetri() {
  const [showTop, setShowTop] = useState(false);

  useEffect(() => {
    const onScroll = () => setShowTop(window.scrollY > 300);
    onScroll();
    window.addEventListener("scroll", onScroll, { passive: true });
    return () => window.removeEventListener("scroll", onScroll);
  }, []);

  const onBackToTop = () => window.scrollTo({ top: 0, behavior: "smooth" });

  const socialLinks = [
    { icon: FiFacebook, href: "#", label: "Facebook" },
    { icon: FiInstagram, href: "#", label: "Instagram" },
    { icon: FiLinkedin, href: "#", label: "LinkedIn" },
    { icon: FiYoutube, href: "#", label: "YouTube" },
  ];

  const footerLinks = [
    {
      title: "Académico",
      links: [
        { name: "Programas", href: "/academico/programas" },
        { name: "Laboratorios", href: "/academico/laboratorios" },
        { name: "Horarios", href: "/academico/horarios" },
      ],
    },
    {
      title: "Investigación",
      links: [
        { name: "Proyectos", href: "/investigacion/proyectos" },
        { name: "Publicaciones", href: "/investigacion/publicaciones" },
        { name: "Grupos", href: "/investigacion/grupos" },
      ],
    },
    {
      title: "Recursos",
      links: [
        { name: "Biblioteca", href: "/recursos/biblioteca" },
        { name: "Documentos", href: "/recursos/documentos" },
        { name: "Enlaces", href: "/recursos/enlaces" },
      ],
    },
  ];

  return (
    <footer className="relative mt-20">
      {/* Gradiente superior con animación */}
      <div className="pointer-events-none absolute inset-x-0 -top-12 h-24 bg-gradient-to-b from-primary-100/30 via-secondary-100/20 to-transparent animate-pulse"
           style={{ animationDuration: '4s' }} />

      {/* Fondo principal */}
      <div className="relative overflow-hidden bg-gradient-to-br from-primary-900 via-primary-800 to-secondary-900">
        {/* Pattern de fondo animado */}
        <div className="absolute inset-0 opacity-10">
          <div className="absolute inset-0 bg-[radial-gradient(circle_at_30%_20%,rgba(255,255,255,0.1)_0%,transparent_50%)] animate-pulse"
               style={{ animationDuration: '6s' }} />
          <div className="absolute inset-0 bg-[radial-gradient(circle_at_70%_80%,rgba(255,255,255,0.08)_0%,transparent_50%)] animate-pulse"
               style={{ animationDuration: '8s', animationDelay: '1s' }} />
        </div>

        {/* Grid decorativo */}
        <div className="absolute inset-0 bg-grid-pattern opacity-5"
             style={{
               backgroundImage: 'linear-gradient(rgba(255,255,255,0.05) 1px, transparent 1px), linear-gradient(90deg, rgba(255,255,255,0.05) 1px, transparent 1px)',
               backgroundSize: '50px 50px'
             }} />

        {/* Contenido principal */}
        <div className="relative mx-auto max-w-7xl px-4 sm:px-6 lg:px-8 py-12 sm:py-16">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-12 gap-8 lg:gap-12">
            {/* Marca y descripción - Col 1-5 */}
            <div className="lg:col-span-5 space-y-6">
              <Link
                href="/"
                className="inline-flex items-center gap-3 group transition-transform duration-300 hover:scale-105"
              >
                <div className="relative">
                  <img
                    src="/assets/images/logo.png"
                    alt="Logo DETRI"
                    className="h-14 w-14 transition-transform duration-300 group-hover:rotate-12"
                  />
                  <div className="absolute inset-0 bg-primary-400/20 rounded-full blur-xl group-hover:bg-primary-300/30 transition-all duration-300" />
                </div>
                <span className="text-2xl font-bold text-white tracking-tight">
                  DETRI
                </span>
              </Link>

              <p className="text-sm leading-relaxed text-primary-100/90">
                Departamento de Electrónica, Telecomunicaciones y Redes de
                Información. Innovación, investigación y formación de excelencia
                al servicio del Ecuador.
              </p>

              {/* Información de contacto */}
              <div className="space-y-3">
                <a
                  href="mailto:contacto@detri.epn.edu.ec"
                  className="flex items-center gap-3 text-sm text-primary-100/80 hover:text-white transition-colors duration-300 group"
                >
                  <div className="p-2 rounded-lg bg-primary-800/50 group-hover:bg-primary-700/50 transition-colors duration-300">
                    <FiMail className="text-primary-200" />
                  </div>
                  <span>contacto@detri.epn.edu.ec</span>
                </a>

                <div className="flex items-center gap-3 text-sm text-primary-100/80">
                  <div className="p-2 rounded-lg bg-primary-800/50">
                    <FiMapPin className="text-primary-200" />
                  </div>
                  <span>Escuela Politécnica Nacional, Quito - Ecuador</span>
                </div>
              </div>

              {/* Redes sociales */}
              <div className="flex gap-3 pt-2">
                {socialLinks.map((social, idx) => (
                  <a
                    key={social.label}
                    aria-label={social.label}
                    href={social.href}
                    className="group relative p-2.5 rounded-lg bg-primary-800/40 hover:bg-secondary-700/60
                             text-primary-100 hover:text-white border border-primary-700/30 hover:border-secondary-600/50
                             transition-all duration-300 hover:scale-110 hover:-translate-y-1"
                    style={{
                      animationDelay: `${idx * 100}ms`,
                      animation: 'fadeInUp 0.6s ease-out forwards'
                    }}
                  >
                    <social.icon className="w-4 h-4" />
                    <div className="absolute inset-0 rounded-lg bg-secondary-500/0 group-hover:bg-secondary-500/10 transition-all duration-300" />
                  </a>
                ))}
              </div>
            </div>

            {/* Enlaces - Col 6-12 */}
            <div className="lg:col-span-7 grid grid-cols-1 sm:grid-cols-3 gap-8">
              {footerLinks.map((section, idx) => (
                <div
                  key={section.title}
                  className="space-y-4"
                  style={{
                    animationDelay: `${(idx + 1) * 150}ms`,
                    animation: 'fadeInUp 0.6s ease-out forwards',
                    opacity: 0
                  }}
                >
                  <h3 className="text-sm font-semibold text-white uppercase tracking-wider">
                    {section.title}
                  </h3>
                  <ul className="space-y-2.5">
                    {section.links.map((link) => (
                      <li key={link.name}>
                        <Link
                          href={link.href}
                          className="group flex items-center text-sm text-primary-200/80 hover:text-white
                                   transition-all duration-300 hover:translate-x-1"
                        >
                          <span className="w-0 h-px bg-secondary-400 group-hover:w-4 transition-all duration-300 mr-0 group-hover:mr-2" />
                          {link.name}
                        </Link>
                      </li>
                    ))}
                  </ul>
                </div>
              ))}
            </div>
          </div>

          {/* Línea divisoria animada */}
          <div className="mt-12 mb-6 relative h-px bg-gradient-to-r from-transparent via-primary-600/50 to-transparent">
            <div className="absolute inset-0 bg-gradient-to-r from-transparent via-secondary-500/50 to-transparent animate-pulse" />
          </div>

          {/* Barra inferior */}
          <div className="flex flex-col sm:flex-row items-center justify-between gap-4 text-sm">
            <p className="text-primary-200/70">
              © {new Date().getFullYear()} DETRI — EPN. Todos los derechos reservados.
            </p>
            <div className="flex flex-wrap gap-6 text-primary-200/70">
              <Link
                href="/politicas/privacidad"
                className="hover:text-white transition-colors duration-300 relative group"
              >
                Privacidad
                <span className="absolute -bottom-1 left-0 w-0 h-0.5 bg-secondary-400 group-hover:w-full transition-all duration-300" />
              </Link>
              <Link
                href="/politicas/terminos"
                className="hover:text-white transition-colors duration-300 relative group"
              >
                Términos
                <span className="absolute -bottom-1 left-0 w-0 h-0.5 bg-secondary-400 group-hover:w-full transition-all duration-300" />
              </Link>
              <Link
                href="/contacto"
                className="hover:text-white transition-colors duration-300 relative group"
              >
                Contacto
                <span className="absolute -bottom-1 left-0 h-0.5 bg-secondary-400 group-hover:w-full transition-all duration-300" />
              </Link>
              <a
                href="#"
                className="hover:text-white transition-colors duration-300 relative group"
              >
                Accesibilidad
                <span className="absolute -bottom-1 left-0 w-0 h-0.5 bg-secondary-400 group-hover:w-full transition-all duration-300" />
              </a>
            </div>
          </div>
        </div>
      </div>

      {/* Botón volver arriba mejorado */}
      <button
        onClick={onBackToTop}
        aria-label="Volver arriba"
        className={`fixed right-6 bottom-6 z-50 p-3.5 rounded-full shadow-2xl
                    bg-gradient-to-br from-primary-600 to-secondary-600
                    text-white border-2 border-primary-400/30
                    hover:from-primary-500 hover:to-secondary-500
                    hover:scale-110 hover:shadow-primary-500/50
                    transition-all duration-300 group
                    ${showTop ? "opacity-100 translate-y-0" : "opacity-0 translate-y-4 pointer-events-none"}`}
      >
        <FiArrowUp className="w-5 h-5 group-hover:-translate-y-1 transition-transform duration-300" />
        <div className="absolute inset-0 rounded-full bg-white/20 opacity-0 group-hover:opacity-100 group-hover:animate-ping transition-opacity" />
      </button>

      <style>{`
        @keyframes fadeInUp {
          from {
            opacity: 0;
            transform: translateY(20px);
          }
          to {
            opacity: 1;
            transform: translateY(0);
          }
        }
      `}</style>
    </footer>
  );
}
