<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Publicacion extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'publicaciones';

    protected $fillable = [
        'anio',
        'estado',
        'titulo',
        'autores',
        'revista',
        'editorial',
        'extracto',
        'citado_por',
        'versiones',
        'url',
        'activo',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'anio'        => 'integer',
        'citado_por'  => 'integer',
        'versiones'   => 'integer',
        'activo'      => 'boolean',
    ];

    public const ESTADO_PUBLICADO = 'publicado';
    public const ESTADO_EN_PRENSA  = 'en-prensa';

    /* ---------- Scopes para filtros ---------- */

    public function scopeSearch(Builder $q, ?string $term): Builder
    {
        $term = trim((string)$term);
        if ($term === '') return $q;

        return $q->where(function ($qq) use ($term) {
            $qq->where('titulo', 'like', "%{$term}%")
               ->orWhere('autores', 'like', "%{$term}%")
               ->orWhere('revista', 'like', "%{$term}%")
               ->orWhere('editorial', 'like', "%{$term}%")
               ->orWhere('extracto', 'like', "%{$term}%");
        });
    }

    public function scopeEstado(Builder $q, ?string $estado): Builder
    {
        if (!$estado) return $q;
        return $q->where('estado', $estado);
    }

    public function scopeActivas(Builder $q, $activo): Builder
    {
        if ($activo === '' || is_null($activo)) return $q;
        return $q->where('activo', (bool)$activo);
    }

    /* ---------- Relaciones trazabilidad ---------- */
    public function creador()
    {
        return $this->belongsTo(\App\Models\User::class, 'created_by');
    }

    public function editor()
    {
        return $this->belongsTo(\App\Models\User::class, 'updated_by');
    }
}
