<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class LineaInvestigacion extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'lineas_investigacion';

    protected $fillable = [
        'uid',
        'slug',
        'nombre',
        'profesor',
        'codigo',
        'fecha',
        'tipo',
        'descripcion',
        'activo',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'fecha'  => 'date',
        'activo' => 'boolean',
    ];

    // Tipos estándar
    public const TIPO_PRIORITARIA = 'Prioritaria';
    public const TIPO_LINEA       = 'Linea';

    /* ---------------------------
     |  Boot: generar slug único
     |----------------------------*/
    protected static function booted(): void
    {
        static::creating(function (self $m) {
            // slug si no viene
            if (empty($m->slug) && !empty($m->nombre)) {
                $m->slug = static::uniqueSlug($m->nombre);
            }
        });

        static::updating(function (self $m) {
            // Si cambia el nombre y no se forzó slug manual, actualiza slug
            if ($m->isDirty('nombre') && !$m->isDirty('slug')) {
                $m->slug = static::uniqueSlug($m->nombre, $m->id);
            }
        });
    }

    protected static function uniqueSlug(string $base, $ignoreId = null): string
    {
        $slug = Str::slug($base);
        $original = $slug;
        $i = 1;

        while (static::query()
            ->when($ignoreId, fn($q) => $q->where('id', '!=', $ignoreId))
            ->where('slug', $slug)->exists()) {
            $slug = $original.'-'.($i++);
        }

        return $slug;
    }

    /* ---------------------------
     |  Scopes para tus filtros UI
     |----------------------------*/
    public function scopeSearch(Builder $q, ?string $term): Builder
    {
        $term = trim((string)$term);
        if ($term === '') return $q;

        return $q->where(function ($q) use ($term) {
            $q->where('nombre', 'like', "%{$term}%")
              ->orWhere('descripcion', 'like', "%{$term}%")
              ->orWhere('profesor', 'like', "%{$term}%")
              ->orWhere('codigo', 'like', "%{$term}%")
              ->orWhere('uid', 'like', "%{$term}%");
        });
    }

    public function scopeTipo(Builder $q, ?string $tipo): Builder
    {
        if (!$tipo || $tipo === 'Todos') return $q;
        return $q->where('tipo', $tipo);
    }

    public function scopeActivas(Builder $q): Builder
    {
        return $q->where('activo', true);
    }

    /* ---------------------------
     |  Relaciones de trazabilidad
     |----------------------------*/
    public function creador()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function editor()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }
}
