<?php

namespace App\Http\Controllers\Public;

use App\Models\Pregrado;
use App\Models\Docente;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Laboratorio;
use App\Models\Autoridad;
use App\Models\Postgrado;
use App\Models\Publicacion;
use App\Models\ProyectoInvestigacion;
use App\Models\LineaInvestigacion;
use App\Models\Pasantia;
use App\Models\TicProject;
use Str;


class LandingController extends Controller
{

   public function home()
{
    // ============================================================
    // 🔹 PREGRADOS — solo visibles
    // ============================================================
    $pregrados = Pregrado::where('is_visible', true)
        ->orderBy('orden')
        ->orderBy('nombre')
        ->get()
        ->map(function ($p) {
            $url = fn($path) => $path
                ? (Str::startsWith($path, ['http://', 'https://', '/'])
                    ? $path
                    : '/storage/' . ltrim($path, '/'))
                : null;

            return [
                'id'          => $p->id,
                'slug'        => $p->slug,
                'nombre'      => $p->nombre,
                'titulo'      => $p->titulo,
                'modalidad'   => $p->modalidad,
                'duracion'    => $p->duracion,
                'fotoTarjeta' => $url($p->foto_tarjeta),
                'imagenHero'  => $url($p->imagen_hero),
            ];
        });

    // ============================================================
    // 🔹 POSTGRADOS — todos (no hay campo is_visible)
    // ============================================================
    $postgrados = Postgrado::orderBy('nombre')
        ->get()
        ->map(function ($p) {
            $url = fn($path) => $path
                ? (Str::startsWith($path, ['http://', 'https://', '/'])
                    ? $path
                    : '/storage/' . ltrim($path, '/'))
                : null;

            return [
                'id'          => $p->id,
                'slug'        => $p->slug,
                'nombre'      => $p->nombre,
                'titulo'      => $p->titulo,
                'modalidad'   => $p->modalidad,
                'duracion'    => $p->duracion,
                'tipo'        => $p->tipo,
                'area'        => $p->area,
                'fotoTarjeta' => $url($p->foto_tarjeta),
                'imagenHero'  => $url($p->imagen_hero),
            ];
        });

    // ============================================================
    // 🔹 Renderizar vista
    // ============================================================
    return Inertia::render('Landing/Home/Index', [
        'pregrados'  => $pregrados,
        'postgrados' => $postgrados,
    ]);
}

    public function mision()
    {
        return inertia('Landing/Detri/Mision/Index');
    }

public function autoridades()
{
    // Etiquetas legibles (opcional, por si en BD falta "grupo")
    $labels = [
        'jefatura'       => 'Jefatura del Departamento',
        'pregrado'       => 'Coordinadores de Carrera de Pregrado',
        'posgrado'       => 'Coordinadores de Posgrado',
        'administrativo' => 'Apoyo Administrativo',
    ];

    $autoridades = Autoridad::query()
        ->select([
            'id','slug','grupo_key','grupo','nombre','cargo','area','email','telefono',
            'horario_dias','horario_rango','ubicacion','foto','orden',
        ])
        // orden lógico por grupo + orden configurable + nombre
        ->orderByRaw("FIELD(grupo_key,'jefatura','pregrado','posgrado','administrativo')")
        ->orderBy('orden')
        ->orderBy('nombre')
        ->get()
        // formateo ligero para el front
        ->map(function ($a) use ($labels) {
            return [
                'id'            => $a->id,
                'slug'          => $a->slug,
                'grupo_key'     => $a->grupo_key,
                'grupo'         => $a->grupo ?: ($labels[$a->grupo_key] ?? $a->grupo_key),
                'nombre'        => $a->nombre,
                'cargo'         => $a->cargo,
                'area'          => $a->area,
                'email'         => $a->email,
                'telefono'      => $a->telefono,
                'horario_dias'  => $a->horario_dias,
                'horario_rango' => $a->horario_rango,
                'ubicacion'     => $a->ubicacion,
                'foto'          => $a->foto,
                'orden'         => (int) $a->orden,
            ];
        });

    return Inertia::render('Landing/Detri/Autoridades/Index', [
        'autoridades' => $autoridades,
        'labels'      => $labels,
    ]);
}



public function personalDocente()
{
    // Trae todos los docentes activos (sin papelera)
    $docentes = Docente::query()
        ->whereNull('deleted_at')
        ->orderBy('area')
        ->orderBy('nombre')
        ->get([
            'id',
            'nombre',
            'grado',
            'area',
            'foto',
            'email',
            'asignaturas',
            'enlaces',
        ])
        ->map(function ($d) {
            return [
                'id'          => $d->id,
                'nombre'      => $d->nombre,
                'grado'       => $d->grado,
                'area'        => $d->area,
                'foto'        => $d->foto ? asset($d->foto) : null,
                'email'       => $d->email,
                'asignaturas' => $d->asignaturas ?? [],
                'enlaces'     => $d->enlaces ?? [],
            ];
        });

    return Inertia::render('Landing/Detri/PersonalDocente/Index', [
        'docentes' => $docentes,
    ]);
}

    public function ubicacion()
    {
        return inertia('Landing/Detri/Ubicacion/Index');
    }

    public function ofertaGrado()
    {
        // 🔹 Traer todas las carreras visibles (puedes quitar el where si quieres mostrar todas)
        $pregrados = Pregrado::where('is_visible', true)
            ->orderBy('orden')
            ->orderBy('nombre')
            ->get()
            ->map(function ($p) {
                $url = fn($path) => $path
                    ? (str_starts_with($path, 'http') ? $path : Storage::url($path))
                    : null;

                return [
                    'id'          => $p->id,
                    'slug'        => $p->slug,
                    'nombre'      => $p->nombre,
                    'titulo'      => $p->titulo,
                    'modalidad'   => $p->modalidad,
                    'duracion'    => $p->duracion,
                    'fotoTarjeta' => $url($p->foto_tarjeta),
                    'imagenHero'  => $url($p->imagen_hero),
                    'coordinador' => [
                        'nombre' => $p->coordinador_nombre,
                        'email'  => $p->coordinador_email,
                        'foto'   => $url($p->coordinador_foto),
                    ],
                ];
            });

        return Inertia::render('Landing/Oferta/Pregrado/Index', [
            'pregrados' => $pregrados,
        ]);
    }




public function ofertaPosgrado()
{
    $postgrados = Postgrado::query()
        ->orderBy('nombre')
        ->get()
        ->map(function ($p) {
            // 🔑 Genera URL relativa cuando no viene absoluta
            $asset = fn($path) => $path
                ? (Str::startsWith($path, ['http://','https://','/'])
                    ? $path
                    : '/storage/' . ltrim($path, '/'))
                : null;

            return [
                'id'           => $p->id,
                'slug'         => $p->slug,
                'nombre'       => $p->nombre,
                'tipo'         => $p->tipo,
                'area'         => $p->area,
                'resumen'      => $p->resumen,
                'duracion'     => $p->duracion,
                'modalidad'    => $p->modalidad,
                'titulo'       => $p->titulo,
                'resolucion'   => $p->resolucion_senescyt,
                'imagen_hero'  => $asset($p->imagen_hero),
                'foto_tarjeta' => $asset($p->foto_tarjeta),
                'brochure'     => $asset($p->brochure_url),
                'malla'        => $asset($p->malla_url),
                'coordinador'  => [
                    'nombre' => $p->coordinador_nombre,
                    'email'  => $p->coordinador_email,
                    'foto'   => $asset($p->coordinador_foto),
                ],
                'colorFrom' => 'from-primary-500',
                'colorTo'   => 'to-secondary-600',
            ];
        });

    return inertia('Landing/Oferta/Posgrado/Index', [
        'postgrados' => $postgrados,
        'titulo'     => 'Programas de Posgrado',
    ]);
}


public function centrosLaboratorios()
{
    // Trae solo lo necesario y en un orden amigable
    $labs = Laboratorio::query()
        ->orderBy('nombre', 'asc')
        ->get([
            'id', 'slug', 'nombre', 'responsable', 'descripcion',
            'ubicacion', 'asignaturas', 'imagenes',
            'created_at', 'updated_at',
        ])
        ->map(function (Laboratorio $lab) {
            return [
                'id'          => $lab->id,
                'slug'        => $lab->slug,
                'nombre'      => $lab->nombre,
                'responsable' => $lab->responsable,
                'descripcion' => $lab->descripcion,
                'ubicacion'   => $lab->ubicacion ?? [],     // {edificio,piso,oficina}
                'asignaturas' => $lab->asignaturas ?? [],   // ["Redes I", ...]
                'imagenes'    => $lab->imagenes ?? [],      // ["/storage/labs/..", ...]
                'created_at'  => optional($lab->created_at)->toISOString(),
                'updated_at'  => optional($lab->updated_at)->toISOString(),
            ];
        });

    return Inertia::render('Landing/Investigacion/CentrosLaboratorios/Index', [
        'laboratorios' => $labs,
    ]);
}

public function proyectosInvestigacion(Request $request)
    {
        $q       = $request->string('q')->toString();
        $linea   = $request->string('linea')->toString();
        $estado  = $request->string('estado')->toString();
        $prof    = $request->string('profesor')->toString();
        $ano     = $request->integer('ano');
        $perPage = (int) max(6, min(24, $request->integer('per_page', 12)));
        $sort    = $request->string('sort', 'ano')->toString(); // 'ano'|'titulo'|'created_at'
        $dir     = $request->string('dir', 'desc')->toString(); // 'asc'|'desc'

        $builder = ProyectoInvestigacion::query(); // público: sin papelera

        // Búsqueda general
        $builder->when($q, function ($query, $q) {
            $query->where(function ($q2) use ($q) {
                $q2->where('titulo', 'like', "%{$q}%")
                   ->orWhere('profesor_a_cargo', 'like', "%{$q}%")
                   ->orWhere('descripcion', 'like', "%{$q}%")
                   ->orWhereJsonContains('keywords', $q);
            });
        });

        // Filtros específicos
        $builder->when($linea,  fn($q2) => $q2->where('linea_investigacion', $linea));
        $builder->when($estado, fn($q2) => $q2->where('estado', $estado));
        $builder->when($prof,   fn($q2) => $q2->where('profesor_a_cargo', 'like', "%{$prof}%"));
        $builder->when($ano,    fn($q2) => $q2->where('ano', $ano));

        // Orden
        $allowed = ['ano','titulo','created_at'];
        if (!in_array($sort, $allowed, true)) $sort = 'ano';
        $builder->orderBy($sort, $dir === 'asc' ? 'asc' : 'desc');

        $proyectos = $builder
            ->paginate($perPage)
            ->withQueryString()
            ->through(function (ProyectoInvestigacion $p) {
                return [
                    'id'                  => $p->id,
                    'titulo'              => $p->titulo,
                    'linea_investigacion' => $p->linea_investigacion,
                    'profesor_a_cargo'    => $p->profesor_a_cargo,
                    'ano'                 => $p->ano,
                    'estado'              => $p->estado,
                    'descripcion'         => str($p->descripcion)->limit(220)->toString(),
                    'keywords'            => $p->keywords,
                ];
            });

        return Inertia::render('Landing/Investigacion/ProyectosInvestigacion/Index', [
            'filters'  => [
                'q' => $q, 'linea' => $linea, 'estado' => $estado,
                'profesor' => $prof, 'ano' => $ano, 'per_page' => $perPage,
                'sort' => $sort, 'dir' => $dir,
            ],
            'lineas'   => ProyectoInvestigacion::LINEAS_INVESTIGACION,
            'estados'  => ProyectoInvestigacion::ESTADOS,
            'anos'     => ProyectoInvestigacion::select('ano')->distinct()->orderBy('ano','desc')->pluck('ano')->values(),
            'proyectos'=> $proyectos,
        ]);
    }
 public function publicaciones(Request $request)
    {
        // Filtros
        $q        = $request->string('q')->toString();
        $estado   = $request->string('estado')->toString();      // 'publicado' | 'en-prensa' | ''
        $anio     = $request->integer('anio');                   // exacto
        $anioFrom = $request->integer('anio_from');
        $anioTo   = $request->integer('anio_to');

        // Orden y tamaño
        $perPage  = (int) max(4, min(24, $request->integer('per_page', 8)));
        $sort     = $request->string('sort', 'anio')->toString();     // 'anio'|'citado_por'|'titulo'
        $dir      = $request->string('dir', 'desc')->toString();      // 'asc'|'desc'

        $allowedSorts = ['anio','citado_por','titulo'];
        if (!in_array($sort, $allowedSorts, true)) $sort = 'anio';
        $dir = $dir === 'asc' ? 'asc' : 'desc';

        $builder = Publicacion::query()->where('activo', true);

        // Búsqueda
        if ($q) {
            $builder->where(function ($qq) use ($q) {
                $qq->where('titulo','like',"%{$q}%")
                   ->orWhere('autores','like',"%{$q}%")
                   ->orWhere('revista','like',"%{$q}%")
                   ->orWhere('editorial','like',"%{$q}%")
                   ->orWhere('extracto','like',"%{$q}%");
            });
        }

        // Filtros
        if ($estado) $builder->where('estado', $estado);
        if ($anio)   $builder->where('anio', $anio);
        if ($anioFrom && $anioTo)   $builder->whereBetween('anio', [$anioFrom,$anioTo]);
        elseif ($anioFrom)          $builder->where('anio','>=',$anioFrom);
        elseif ($anioTo)            $builder->where('anio','<=',$anioTo);

        // Orden
        $builder->orderBy($sort, $dir)->orderBy('titulo');

        // Paginación + mapeo a camelCase para el front
        $rows = $builder->paginate($perPage)->withQueryString()->through(function (Publicacion $p) {
            return [
                'id'         => $p->id,
                'anio'       => (int)$p->anio,
                'estado'     => $p->estado,
                'titulo'     => $p->titulo,
                'autores'    => $p->autores,
                'revista'    => (string)($p->revista ?? ''),
                'editorial'  => (string)($p->editorial ?? ''),
                'extracto'   => (string)($p->extracto ?? ''),
                'citadoPor'  => (int)($p->citado_por ?? 0),
                'url'        => $p->url,
                'versiones'  => (int)($p->versiones ?? 0),
            ];
        });

        // Catálogos
        $years = Publicacion::query()
            ->select('anio')->where('activo',true)
            ->distinct()->orderByDesc('anio')->pluck('anio')->map(fn($y)=>(string)$y);

        $estados = ['publicado','en-prensa'];

        // Métricas (para el hero)
        $totalCitas = (int) Publicacion::query()->where('activo',true)->sum('citado_por');
        $recientes  = (int) Publicacion::query()->where('activo',true)->where('anio','>=', now()->year - 5)->count();
        $total      = (int) Publicacion::query()->where('activo',true)->count();

        return Inertia::render('Landing/Investigacion/Publicaciones/Index', [
            'filters' => [
                'q'        => $q,
                'estado'   => $estado,
                'anio'     => $anio,
                'anio_from'=> $anioFrom,
                'anio_to'  => $anioTo,
                'sort'     => $sort,
                'dir'      => $dir,
                'per_page' => $perPage,
            ],
            'catalogs' => [
                'years'   => $years,
                'estados' => $estados,
            ],
            'metrics' => [
                'total'        => $total,
                'totalCitas'   => $totalCitas,
                'recientes5an' => $recientes,
            ],
            'publicaciones' => $rows,
        ]);
    }
    public function lineasInvestigacion()
    {

        $lineasInvestigacion=LineaInvestigacion::all();

        return inertia('Landing/Investigacion/LineasInvestigacion/Index',[
            'lineas' => $lineasInvestigacion,
        ]);
    }
   public function pasantias(Request $request)
    {
        $modalidad = $request->string('modalidad')->toString();
        $carrera   = $request->string('carrera')->toString();
        $q         = trim($request->string('q')->toString());

        $query = Pasantia::query()
            ->where('estado', 'publicado')
            ->select([
                'id','titulo','empresa','modalidad','duracion','carrera',
                'fecha_inicio','fecha_maxima_postulacion',
                'tipo','remunerado','monto_remuneracion','cupos',
                // si tienes JSON en texto:
                'requisitos','beneficios',
            ]);

        if ($modalidad && $modalidad !== 'Todas') {
            $query->where('modalidad', $modalidad);
        }
        if ($carrera && $carrera !== 'Todas') {
            $query->where('carrera', $carrera);
        }
        if ($q !== '') {
            $query->where(function($qq) use ($q) {
                $qq->where('titulo', 'like', "%{$q}%")
                   ->orWhere('empresa', 'like', "%{$q}%")
                   ->orWhere('carrera', 'like', "%{$q}%");
            });
        }

        $pasantias = $query
            ->orderByDesc('id')
            ->simplePaginate(8)
            ->appends($request->query()) // para que next_page_url mantenga los filtros
            ->through(function ($p) {
                // Normalizaciones y formatos (sin colores/gradientes):
                $p->fecha_inicio_legible = optional($p->fecha_inicio)->format('d M Y');
                $p->fecha_maxima_postulacion_legible = optional($p->fecha_maxima_postulacion)->format('d M Y');
                $p->requisitos = is_array($p->requisitos) ? $p->requisitos : json_decode($p->requisitos ?? '[]', true);
                $p->beneficios = is_array($p->beneficios) ? $p->beneficios : json_decode($p->beneficios ?? '[]', true);
                return $p;
            });

        return Inertia::render('Landing/VidaEstudiantil/Pasantias/Index', [
            'pasantias'   => $pasantias,
            'modalidades' => ['Todas','Remoto','Presencial','Híbrido'],
            'carreras'    => array_values(array_unique(array_merge(
                ['Todas'],
                Pasantia::where('estado','publicado')->pluck('carrera')->filter()->all()
            ))),
            'empresas_count' => Pasantia::where('estado','publicado')->distinct('empresa')->count('empresa'),
            'filters' => [
                'modalidad' => $modalidad ?: 'Todas',
                'carrera'   => $carrera ?: 'Todas',
                'q'         => $q ?: '',
            ],
        ]);
    }
 public function tic(Request $request)
    {
        // Filtros entrantes
        $carrera   = $request->string('carrera')->toString();
        $profesor  = $request->string('profesorCargo')->toString();
        $semestre  = $request->string('semestre')->toString();
        $asignatura= $request->string('asignaturaRelacionada')->toString();
        $linea     = $request->string('lineaInvestigacion')->toString();
        $q         = trim($request->string('q')->toString());

        // Query base
        $query = TicProject::query()
            ->where('estado', 'publicado')
            ->select([
                'id',
                'carrera',
                'nombre_proyecto as nombreProyecto',
                'profesor_cargo as profesorCargo',
                'areas_trabajo',                // JSON
                'resumen',
                'semestre',
                'asignaturas_relacionadas',     // JSON
                'linea_investigacion as lineaInvestigacion',
            ]);

        // Filtros
        if ($carrera !== '') {
            $query->where('carrera', $carrera);
        }
        if ($profesor !== '') {
            $query->where('profesor_cargo', $profesor);
        }
        if ($semestre !== '') {
            $query->where('semestre', $semestre);
        }
        if ($linea !== '') {
            $query->where('linea_investigacion', $linea);
        }
        if ($asignatura !== '') {
            // Si guardas JSON en MySQL
            $query->whereJsonContains('asignaturas_relacionadas', $asignatura);
        }
        if ($q !== '') {
            $query->where(function ($qq) use ($q) {
                $qq->where('nombre_proyecto', 'like', "%{$q}%")
                   ->orWhere('profesor_cargo', 'like', "%{$q}%")
                   ->orWhere('carrera', 'like', "%{$q}%")
                   ->orWhere('resumen', 'like', "%{$q}%")
                   ->orWhere('linea_investigacion', 'like', "%{$q}%");
            });
        }

        // Paginación + mapeo de JSON a arrays
        $projects = $query->orderByDesc('id')
            ->simplePaginate(9)
            ->appends($request->query())
            ->through(function ($p) {
                // Normaliza JSON → array
                $p->areasTrabajo = is_array($p->areas_trabajo)
                    ? $p->areas_trabajo
                    : json_decode($p->areas_trabajo ?? '[]', true);
                unset($p->areas_trabajo);

                $p->asignaturasRelacionadas = is_array($p->asignaturas_relacionadas)
                    ? $p->asignaturas_relacionadas
                    : json_decode($p->asignaturas_relacionadas ?? '[]', true);
                unset($p->asignaturas_relacionadas);

                return $p;
            });

        // Catálogos (para los selects)
        $base = TicProject::where('estado', 'publicado');

        $carreras = $base->clone()->distinct()->orderBy('carrera')->pluck('carrera')->filter()->values()->all();
        $profesores = $base->clone()->distinct()->orderBy('profesor_cargo')->pluck('profesor_cargo')->filter()->values()->all();
        $semestres = $base->clone()->distinct()->orderByDesc('semestre')->pluck('semestre')->filter()->values()->all();
        $lineas = $base->clone()->distinct()->orderBy('linea_investigacion')->pluck('linea_investigacion')->filter()->values()->all();

        // Asignaturas desde JSON (aplana y deduplica)
        $asignaturasCol = $base->clone()->pluck('asignaturas_relacionadas')->filter();
        $asignaturas = $asignaturasCol
            ->flatMap(function ($v) {
                if (is_array($v)) return $v;
                $arr = json_decode($v ?? '[]', true);
                return is_array($arr) ? $arr : [];
            })
            ->filter()
            ->unique()
            ->values()
            ->all();

        // Total (opcional)
        $total_count = (int) $base->clone()->count();

        return Inertia::render('Landing/VidaEstudiantil/TIC/Index', [
            'projects' => $projects,
            'carreras' => $carreras,
            'profesores' => $profesores,
            'semestres' => $semestres,
            'asignaturas' => $asignaturas,
            'lineasInvestigacion' => $lineas,
            'filters' => [
                'carrera' => $carrera,
                'profesorCargo' => $profesor,
                'semestre' => $semestre,
                'asignaturaRelacionada' => $asignatura,
                'lineaInvestigacion' => $linea,
                'q' => $q,
            ],
            'total_count' => $total_count,
        ]);
    }

    public function pregrado($slug)
    {
        // Buscar por slug
        $career = Pregrado::where('slug', $slug)
            ->where('is_visible', true)
            ->first();

        if (!$career) {
            abort(404, 'Carrera no encontrada');
        }

        // Normalizar rutas a /storage
        $url = fn($path) => $path
            ? (str_starts_with($path, 'http') ? $path : Storage::url($path))
            : null;

        // Adaptar estructura a lo que ya usa tu vista
        $careerData = [
            'nombre' => $career->nombre,
            'coordinador' => [
                'nombre' => $career->coordinador_nombre,
                'email'  => $career->coordinador_email,
                'foto'   => $url($career->coordinador_foto),
            ],
            'duracion' => $career->duracion,
            'modalidad' => $career->modalidad,
            'titulo' => $career->titulo,
            'imagenHero' => $url($career->imagen_hero),
            'fotoTarjeta' => $url($career->foto_tarjeta),
            'informacion' => $career->informacion,
            'perfil' => $career->perfil,
            'mallaUrl' => $url($career->malla_url), // o $career->malla_archivo
            'requisitos' => $career->requisitos ?? [],
        ];

        return Inertia::render('Landing/Oferta/CareerDetail', [
            'career' => $careerData,
        ]);
    }

    // app/Http/Controllers/Public/AcademiaController.php

 public function postgrado(string $slug)
    {
        // Busca visible y no eliminado
        $p = Postgrado::query()
            ->where('slug', $slug)
            ->firstOrFail();

        // Helper para resolver /storage
        $asset = function (?string $path) {
            if (!$path) return null;
            return str_starts_with($path, 'http')
                ? $path
                : Storage::disk('public')->url($path);
        };

        // Arreglo normalizado + alias para tu vista actual
        $programa = [
            // ====== Campos base ======
            'slug'       => $p->slug,
            'nombre'     => $p->nombre,
            'tipo'       => $p->tipo,            // p.ej. Maestría profesional / Doctorado
            'area'       => $p->area,            // p.ej. IoT / Seguridad / Telecom
            'resumen'    => $p->resumen,
            'duracion'   => $p->duracion,
            'modalidad'  => $p->modalidad,
            'titulo'     => $p->titulo,
            'resolucion' => $p->resolucion_senescyt, // alias simple para tu vista

            // ====== Recursos (resueltos a URL pública) ======
            'imagen_hero'  => $asset($p->imagen_hero),
            'malla_url'    => $asset($p->malla_url),
            'brochure_url' => $asset($p->brochure_url),
            'inscripciones_url' => $p->inscripciones_url,

            // ====== Estructurados ======
            'objetivos'      => $p->objetivos ?? [],
            'perfil_egreso'  => $p->perfil_egreso ?? [],
            'requisitos'     => $p->requisitos ?? [],
            'calendario'     => $p->calendario ?? null,  // ej: ['convocatoria'=>'...','clases'=>'...']

            // ====== Contacto / Coordinación ======
            'contacto' => [
                'coordinador' => $p->coordinador_nombre,
                'email'       => $p->coordinador_email,
                'foto'        => $asset($p->coordinador_foto),
            ],

            // ====== Aliases para compat con tu vista actual ======
            'banner'  => $asset($p->imagen_hero), // antes usabas 'banner'
            'enlaces' => [
                'brochure' => $asset($p->brochure_url), // alias
                'plan'     => $asset($p->malla_url),    // alias (plan de estudios)
                'detalle'  => url()->current(),         // o route('postgrado.show', $p->slug)
            ],

            // Opcionales (si tu UI usa gradientes por programa)
            'colorFrom' => 'from-primary-500',
            'colorTo'   => 'to-secondary-600',
        ];

        return inertia('Landing/Oferta/PosgradoDetail', [
            'programa' => $programa,
            'titulo'   => $p->nombre,
        ]);
    }
}
