<?php

namespace App\Http\Controllers\Docente;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Carbon\Carbon;

use App\Models\Docente;
use App\Models\ProyectoInvestigacion;
use App\Models\Publicacion;
use App\Models\TicProject;

class DocenteDashboardController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();

        // Rango por defecto: últimos 12 meses
        $to   = Carbon::parse($request->input('to', now()->toDateString()))->endOfDay();
        $from = Carbon::parse($request->input('from', $to->copy()->subDays(364)->toDateString()))->startOfDay();

        // ---- Resolver Docente actual ----------------------------------------
        $docente = null;

        // 1) por docente_id explícito (útil para pruebas)
        if ($request->filled('docente_id')) {
            $docente = Docente::find($request->integer('docente_id'));
        }

        // 2) por relación con usuario (user_id) si existe la columna
        if (!$docente) {
            $docTable = (new Docente)->getTable();
            if (Schema::hasColumn($docTable, 'user_id') && $user) {
                $docente = Docente::where('user_id', $user->id)->first();
            }
        }

        // 3) por correo == email del usuario (si existe columna correo)
        if (!$docente && $user && Schema::hasColumn((new Docente)->getTable(), 'correo')) {
            $docente = Docente::where('correo', $user->email)->first();
        }

        // 4) fallback (primer docente) — evita crashear en ambientes vacíos
        if (!$docente) {
            $docente = Docente::first();
        }

        if (!$docente) {
            // No hay docentes en el sistema
            return inertia('Docente/Dashboard/Index', [
                'range' => ['from' => $from->toDateString(), 'to' => $to->toDateString()],
                'profile' => null,
                'totals' => ['proyectos' => 0, 'publicaciones' => 0, 'tic_projects' => 0],
                'series' => ['proyectos' => [], 'publicaciones' => [], 'tic_projects' => []],
                'tops' => ['proyectos_por_linea' => []],
                'latest' => ['publicaciones' => []],
            ]);
        }

        // ---- Helpers ---------------------------------------------------------
        $makeEmptyMonths = function (Carbon $from, Carbon $to) {
            $cursor = $from->copy()->startOfMonth();
            $end    = $to->copy()->startOfMonth();
            $res = [];
            while ($cursor <= $end) {
                $res[$cursor->format('Y-m')] = 0;
                $cursor->addMonth();
            }
            return $res;
        };

        $monthlySeries = function ($builder, $from, $to, $dateCol = 'created_at') use ($makeEmptyMonths) {
            $table = $builder->getModel()->getTable();
            if (!Schema::hasColumn($table, $dateCol)) return [];

            $base = $makeEmptyMonths($from, $to);
            $rows = (clone $builder)
                ->whereBetween($dateCol, [$from, $to])
                ->selectRaw("DATE_FORMAT($dateCol, '%Y-%m') as ym, COUNT(*) as c")
                ->groupBy('ym')
                ->orderBy('ym')
                ->pluck('c', 'ym')
                ->toArray();

            foreach ($rows as $ym => $c) {
                if (array_key_exists($ym, $base)) $base[$ym] = (int) $c;
            }

            return [
                'labels' => array_keys($base),
                'data'   => array_values($base),
            ];
        };

        // Construye un builder filtrado por el docente según columnas disponibles
        $filterByDocente = function ($modelClass) use ($docente) {
            $table = (new $modelClass)->getTable();
            $q = $modelClass::query();

            $hasIdCols = [
                'docente_id'  => Schema::hasColumn($table, 'docente_id'),
                'profesor_id' => Schema::hasColumn($table, 'profesor_id'),
            ];
            $hasTxtCols = [
                'docente'        => Schema::hasColumn($table, 'docente'),
                'profesor'       => Schema::hasColumn($table, 'profesor'),
                'profesor_cargo' => Schema::hasColumn($table, 'profesor_cargo'),
                'autores'        => Schema::hasColumn($table, 'autores'),
            ];

            // Aplica condiciones OR seguras: por ID y por nombre del docente
            $q->where(function ($w) use ($hasIdCols, $hasTxtCols, $docente) {
                $nombre = trim(($docente->nombre ?? '').' '.($docente->apellido ?? ''));
                // ID-based
                if ($hasIdCols['docente_id'])  $w->orWhere('docente_id',  $docente->id);
                if ($hasIdCols['profesor_id']) $w->orWhere('profesor_id', $docente->id);
                // Text-based
                if ($hasTxtCols['docente'] && $nombre)        $w->orWhere('docente', 'like', "%{$nombre}%");
                if ($hasTxtCols['profesor'] && $nombre)       $w->orWhere('profesor', 'like', "%{$nombre}%");
                if ($hasTxtCols['profesor_cargo'] && $nombre) $w->orWhere('profesor_cargo', 'like', "%{$nombre}%");
                if ($hasTxtCols['autores'] && $nombre)        $w->orWhere('autores', 'like', "%{$nombre}%");
            });

            return $q;
        };

        // ---- Builders por entidad del docente -------------------------------
        $qProyectos    = $filterByDocente(ProyectoInvestigacion::class);
        $qPublicaciones= $filterByDocente(Publicacion::class);
        $qTic          = $filterByDocente(TicProject::class);

        // ---- Totales ---------------------------------------------------------
        $totals = [
            'proyectos'     => (clone $qProyectos)->count(),
            'publicaciones' => (clone $qPublicaciones)->count(),
            'tic_projects'  => (clone $qTic)->count(),
        ];

        // ---- Series (últimos 12 meses) --------------------------------------
        $series = [
            'proyectos'     => $monthlySeries($qProyectos, $from, $to),
            'publicaciones' => $monthlySeries($qPublicaciones, $from, $to),
            'tic_projects'  => $monthlySeries($qTic, $from, $to),
        ];

        // ---- Tops: proyectos por línea de investigación ---------------------
        $tops = ['proyectos_por_linea' => []];
        $projTable = (new ProyectoInvestigacion)->getTable();
        if (Schema::hasColumn($projTable, 'linea_investigacion')) {
            $tops['proyectos_por_linea'] = (clone $qProyectos)
                ->whereNotNull('linea_investigacion')
                ->select('linea_investigacion', DB::raw('COUNT(*) as total'))
                ->groupBy('linea_investigacion')
                ->orderByDesc('total')
                ->limit(8)
                ->get();
        }

        // ---- Últimas publicaciones ------------------------------------------
        $latest = ['publicaciones' => []];
        $pubTable = (new Publicacion)->getTable();
        $cols = Schema::getColumnListing($pubTable);
        $select = array_values(array_intersect($cols, [
            'id','titulo','anio','revista','editorial','autores','created_at'
        ]));
        if (!empty($select)) {
            $latest['publicaciones'] = (clone $qPublicaciones)
                ->orderByDesc('created_at')
                ->limit(8)
                ->get($select);
        } else {
            $latest['publicaciones'] = (clone $qPublicaciones)
                ->orderByDesc('id')
                ->limit(8)
                ->get();
        }

        // ---- Perfil compacto del docente ------------------------------------
        $profile = [
            'id'      => $docente->id,
            'nombre'  => trim(($docente->nombre ?? '').' '.($docente->apellido ?? '')) ?: ($docente->nombre ?? null),
            'correo'  => $docente->correo ?? null,
            'area'    => $docente->area ?? ($docente->departamento ?? null),
            'foto'    => $docente->foto ?? null,
        ];

        return inertia('Docente/Dashboard/Index', [
            'range'  => ['from' => $from->toDateString(), 'to' => $to->toDateString()],
            'profile'=> $profile,
            'totals' => $totals,
            'series' => $series,
            'tops'   => $tops,
            'latest' => $latest,
        ]);
    }
}
