<?php

namespace App\Http\Controllers\Docente;

use App\Http\Controllers\Controller;
use App\Models\Docente;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class DocenteController extends Controller
{

  /**
     * Obtiene (o crea si no existe) el docente vinculado al usuario autenticado.
     * Se vincula por email (tu modelo Docente no tiene user_id).
     */
    protected function docenteFromAuth(): Docente
    {
        $user = auth()->user();

        return Docente::firstOrCreate(
            ['email' => $user->email],
            [
                'nombre'      => $user->name,
                'grado'       => null,
                'area'        => null,
                'foto'        => null,
                'asignaturas' => [],
                'enlaces'     => [],
            ]
        );
    }

    /**
     * GET /docente/perfil
     * Editar solo mi perfil de docente.
     */
    public function edit(Request $request)
    {
        $docente = $this->docenteFromAuth();

        return Inertia::render('Docente/Perfil/Index', [
            'perfil' => [
                'id'          => $docente->id,
                'nombre'      => $docente->nombre,
                'grado'       => $docente->grado,
                'area'        => $docente->area,
                'foto'        => $docente->foto,
                'email'       => $docente->email,
                'asignaturas' => $docente->asignaturas ?? [],
                'enlaces'     => $docente->enlaces ?? [],
                'updated_at'  => $docente->updated_at?->toISOString(),
            ],
        ]);
    }

    /**
     * PATCH /docente/perfil
     * Actualiza SOLO mi perfil. Permite cambiar email y nombre; sincroniza con User.
     * Soporta subir nueva foto (opcional) y arrays (asignaturas/enlaces).
     */
    public function update(Request $request)
    {
        $user    = $request->user();
        $docente = $this->docenteFromAuth();

        $data = $request->validate([
            'nombre'       => ['required','string','max:255'],
            'grado'        => ['nullable','string','max:255'],
            'area'         => ['nullable','string','max:255'],
            'email'        => [
                'required','email','max:255',
                // Unicidad en docentes ignorando mi propio registro
                Rule::unique('docentes','email')->whereNull('deleted_at')->ignore($docente->id),
            ],
            'asignaturas'  => ['nullable','array'],
            'enlaces'      => ['nullable','array'],
            'foto_file'    => ['nullable','image','max:2048'], // 2MB
        ]);

        // Procesar imagen si viene
        if ($request->hasFile('foto_file')) {
            // Elimina previa si quieres:
            // if ($docente->foto && str_starts_with($docente->foto, '/storage/')) { ... }
            $path = $request->file('foto_file')->store('docentes', 'public');
            $data['foto'] = "/storage/{$path}";
        }
        unset($data['foto_file']);

        // Normaliza arrays
        $data['asignaturas'] = $data['asignaturas'] ?? [];
        $data['enlaces']     = $data['enlaces'] ?? [];

        // Actualiza Docente
        $docente->update($data);

        // Sincroniza también con el usuario (name + email)
        // Valida unicidad en users (ignorando el propio id)
        $request->validate([
            'email' => [Rule::unique('users','email')->whereNull('deleted_at')->ignore($user->id)],
        ]);

        $user->update([
            'name'  => $data['nombre'],
            'email' => $data['email'],
        ]);

        return back()->with('success', 'Perfil actualizado correctamente.');
    }
}
