<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Inertia\Inertia;

class AdminUsuarioController extends Controller
{
    /**
     * GET /admin/usuarios
     * Filtros: q, role, status(active|inactive), trash(none|with|only), per_page, sort(name|email|created_at), dir(asc|desc)
     */
    public function index(Request $request)
    {
        $q        = $request->string('q')->toString();
        $role     = $request->string('role')->toString();
        $status   = $request->string('status')->toString();         // 'active' | 'inactive' | ''
        $trash    = $request->string('trash', 'none')->toString();  // 'none' | 'with' | 'only'
        $perPage  = (int) max(5, min(100, $request->integer('per_page', 10)));
        $sort     = $request->string('sort', 'name')->toString();   // 'name'|'email'|'created_at'
        $dir      = $request->string('dir', 'asc')->toString();     // 'asc'|'desc'

        $builder = User::query();

        // Papelera / SoftDeletes
        if ($trash === 'with') {
            $builder->withTrashed();
        } elseif ($trash === 'only') {
            $builder->onlyTrashed();
        }

        // Búsqueda
        $builder->when($q, function ($query, $q) {
            $query->where(function ($q2) use ($q) {
                $q2->where('name', 'like', "%{$q}%")
                   ->orWhere('email', 'like', "%{$q}%");
            });
        });

        // Rol
        $builder->when($role, fn($query) => $query->where('role', $role));

        // Estado
        $builder->when($status === 'active', fn($query) => $query->where('is_active', true));
        $builder->when($status === 'inactive', fn($query) => $query->where('is_active', false));

        // Orden
        if (!in_array($sort, ['name','email','created_at'], true)) {
            $sort = 'name';
        }
        $builder->orderBy($sort, $dir === 'desc' ? 'desc' : 'asc');

        $users = $builder
            ->paginate($perPage)
            ->withQueryString()
            ->through(function (User $user) {
                return [
                    'id'         => $user->id,
                    'name'       => $user->name,
                    'email'      => $user->email,
                    'role'       => $user->role,
                    'is_active'  => (bool) $user->is_active,
                    'deleted_at' => $user->deleted_at?->toISOString(),
                    'created_at' => $user->created_at?->toISOString(),
                    'updated_at' => $user->updated_at?->toISOString(),
                ];
            });

        return Inertia::render('Admin/Usuarios/Index', [
            'filters' => compact('q','role','status','trash','perPage','sort','dir'),
            'roles'   => ['Administrador','Secretaria','Docente'],
            'users'   => $users,
        ]);
    }

    /**
     * POST /admin/usuarios
     * Crea un usuario. Password opcional; si no se envía, se genera aleatoria.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'      => ['required','string','max:120'],
            'email'     => [
                'required','email','max:150',
                Rule::unique('users','email')->whereNull('deleted_at'),
            ],
            'role'      => ['required', Rule::in(['Administrador','Secretaria','Docente'])],
            'is_active' => ['nullable','boolean'],
            'password'  => ['nullable','string','min:8','max:100'],
        ]);

        $plain = $data['password'] ?? Str::password(12);

        User::create([
            'name'      => $data['name'],
            'email'     => $data['email'],
            'role'      => $data['role'],
            'is_active' => $data['is_active'] ?? true,
            'password'  => Hash::make($plain),
        ]);

        return back()->with('success', 'Usuario creado correctamente.');
    }

    /**
     * PATCH /admin/usuarios/{user}
     * Actualiza datos. Si viene 'password', se cambia; si no, se conserva.
     */
    public function update(Request $request, User $user)
    {
        $data = $request->validate([
            'name'      => ['required','string','max:120'],
            'email'     => [
                'required','email','max:150',
                Rule::unique('users','email')->whereNull('deleted_at')->ignore($user->id),
            ],
            'role'      => ['required', Rule::in(['Administrador','Secretaria','Docente'])],
            'is_active' => ['nullable','boolean'],
            'password'  => ['nullable','string','min:8','max:100'],
        ]);

        $payload = [
            'name'      => $data['name'],
            'email'     => $data['email'],
            'role'      => $data['role'],
            'is_active' => array_key_exists('is_active', $data) ? (bool)$data['is_active'] : $user->is_active,
        ];

        if (!empty($data['password'])) {
            $payload['password'] = Hash::make($data['password']);
            // Opcional: invalidar tokens "remember me"
            $user->setRememberToken(Str::random(60));
        }

        $user->update($payload);

        return back()->with('success', 'Usuario actualizado correctamente.');
    }

    /**
     * DELETE /admin/usuarios/{user}
     * Soft delete.
     */
    public function destroy(User $user)
    {
        $user->delete();

        return back()->with('success', 'Usuario enviado a la papelera.');
    }

    /**
     * PATCH /admin/usuarios/{id}/restore
     * Restaura un usuario eliminado (soft-deleted).
     */
    public function restore($id)
    {
        $user = User::withTrashed()->findOrFail($id);
        if ($user->trashed()) {
            $user->restore();
        }

        return back()->with('success', 'Usuario restaurado.');
    }

    /**
     * DELETE /admin/usuarios/{id}/force
     * Eliminación definitiva.
     */
    public function forceDestroy($id)
    {
        $user = User::withTrashed()->findOrFail($id);
        $user->forceDelete();

        return back()->with('success', 'Usuario eliminado definitivamente.');
    }

    /**
     * PATCH /admin/usuarios/{user}/toggle-status
     * Activa/Desactiva (switch rápido).
     */
    public function toggleStatus(User $user)
    {
        $user->is_active = !$user->is_active;
        $user->save();

        return back()->with('success', $user->is_active ? 'Usuario activado.' : 'Usuario desactivado.');
    }
}
