<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class AdminUserController extends Controller
{
    /**
     * GET /admin/mi-perfil
     * Devuelve SOLO el usuario autenticado (sin rol/activo para edición).
     */
    public function index(Request $request)
    {
        $u = $request->user();

        return Inertia::render('Admin/Users/Index', [
            'user' => [
                'id'         => $u->id,
                'name'       => $u->name,
                'email'      => $u->email,
                'created_at' => $u->created_at?->toISOString(),
                'updated_at' => $u->updated_at?->toISOString(),
            ],
        ]);
    }

    /**
     * PATCH /admin/mi-perfil
     * Actualiza name, email y (opcional) password con verificación de contraseña actual.
     */
    public function update(Request $request)
    {
        $user = $request->user();

        $data = $request->validate([
            'name'               => ['required','string','max:255'],
            'email'              => ['required','email','max:255', Rule::unique('users','email')->ignore($user->id)],
            // Si se va a cambiar password: requiere current_password y confirmación
            'current_password'   => ['required_with:password','current_password'],
            'password'           => ['nullable','string','min:8','max:100','confirmed'],
            // 'password_confirmation' viene del helper 'confirmed'
        ]);

        // Actualiza nombre y correo
        $user->name  = $data['name'];
        $user->email = $data['email'];

        // Si envían nueva contraseña, se setea (el cast 'hashed' en el modelo la hashea)
        if (!empty($data['password'])) {
            $user->password = $data['password']; // se hashea por $casts['password' => 'hashed']
        }

        $user->save();

        return back()->with('success', 'Perfil actualizado correctamente.');
    }
}
