<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\TicProject;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class AdminTicController extends Controller
{
    /**
     * GET /admin/tic-projects
     * Listado con filtros, orden, paginación y papelera.
     */
    public function index(Request $request)
    {
        $q          = $request->string('q')->toString();           // texto libre
        $carrera    = $request->string('carrera')->toString();
        $profesor   = $request->string('profesor')->toString();    // like
        $semestre   = $request->string('semestre')->toString();
        $linea      = $request->string('linea')->toString();
        $asignatura = $request->string('asignatura')->toString();  // json contains
        $estado     = $request->string('estado')->toString();      // 'borrador'|'publicado'|''

        $trash      = $request->string('trash', 'none')->toString(); // 'none'|'with'|'only'
        $perPage    = (int) max(5, min(100, (int)$request->integer('per_page', 12)));

        // sort: nombre|profesor|carrera|semestre|linea|estado|created_at
        $sort       = $request->string('sort', 'created_at')->toString();
        $dir        = $request->string('dir', 'desc')->toString(); // asc|desc

        // Mapeo seguro de sort → columnas reales
        $sortMap = [
            'nombre'     => 'nombre_proyecto',
            'profesor'   => 'profesor_cargo',
            'carrera'    => 'carrera',
            'semestre'   => 'semestre',
            'linea'      => 'linea_investigacion',
            'estado'     => 'estado',
            'created_at' => 'created_at',
        ];
        $sortCol = $sortMap[$sort] ?? 'created_at';
        $dir     = $dir === 'asc' ? 'asc' : 'desc';

        $builder = TicProject::query();

        // Papelera
        if ($trash === 'with') {
            $builder->withTrashed();
        } elseif ($trash === 'only') {
            $builder->onlyTrashed();
        }

        // Búsqueda general
        if ($q !== '') {
            $builder->where(function ($qb) use ($q) {
                $qb->where('nombre_proyecto', 'like', "%{$q}%")
                   ->orWhere('resumen', 'like', "%{$q}%")
                   ->orWhere('profesor_cargo', 'like', "%{$q}%")
                   ->orWhere('carrera', 'like', "%{$q}%")
                   ->orWhere('linea_investigacion', 'like', "%{$q}%");
            });
        }

        // Filtros específicos
        $builder->when($carrera,  fn($qb) => $qb->where('carrera', $carrera));
        $builder->when($profesor, fn($qb) => $qb->where('profesor_cargo', 'like', "%{$profesor}%"));
        $builder->when($semestre, fn($qb) => $qb->where('semestre', $semestre));
        $builder->when($linea,    fn($qb) => $qb->where('linea_investigacion', $linea));
        if ($asignatura !== '') {
            $builder->whereJsonContains('asignaturas_relacionadas', $asignatura);
        }
        if ($estado !== '') {
            $builder->where('estado', $estado);
        }

        // Orden
        $builder->orderBy($sortCol, $dir);

        // Paginación + mapeo para frontend (camelCase + arrays)
        $projects = $builder
            ->paginate($perPage)
            ->withQueryString()
            ->through(function (TicProject $p) {
                return [
                    'id'                     => $p->id,
                    'nombreProyecto'         => $p->nombre_proyecto,
                    'carrera'                => $p->carrera,
                    'profesorCargo'          => $p->profesor_cargo,
                    'resumen'                => $p->resumen,
                    'semestre'               => $p->semestre,
                    'lineaInvestigacion'     => $p->linea_investigacion,
                    'areasTrabajo'           => is_array($p->areas_trabajo) ? $p->areas_trabajo : json_decode($p->areas_trabajo ?? '[]', true),
                    'asignaturasRelacionadas'=> is_array($p->asignaturas_relacionadas) ? $p->asignaturas_relacionadas : json_decode($p->asignaturas_relacionadas ?? '[]', true),
                    'estado'                 => $p->estado,
                    'deleted_at'             => $p->deleted_at?->toISOString(),
                    'created_at'             => $p->created_at?->toISOString(),
                    'updated_at'             => $p->updated_at?->toISOString(),
                ];
            });

        // Catálogos dinámicos (para selects)
        $base = TicProject::query();
        $carreras   = (clone $base)->distinct()->orderBy('carrera')->pluck('carrera')->filter()->values()->all();
        $profesores = (clone $base)->distinct()->orderBy('profesor_cargo')->pluck('profesor_cargo')->filter()->values()->all();
        $semestres  = (clone $base)->distinct()->orderByDesc('semestre')->pluck('semestre')->filter()->values()->all();
        $lineas     = (clone $base)->distinct()->orderBy('linea_investigacion')->pluck('linea_investigacion')->filter()->values()->all();

        // Asignaturas desde JSON
        $asigCol = (clone $base)->pluck('asignaturas_relacionadas')->filter();
        $asignaturas = $asigCol
            ->flatMap(function ($v) {
                if (is_array($v)) return $v;
                $arr = json_decode($v ?? '[]', true);
                return is_array($arr) ? $arr : [];
            })
            ->filter()
            ->unique()
            ->values()
            ->all();

        return Inertia::render('Admin/TicProjects/Index', [
            'filters' => [
                'q'          => $q,
                'carrera'    => $carrera,
                'profesor'   => $profesor,
                'semestre'   => $semestre,
                'linea'      => $linea,
                'asignatura' => $asignatura,
                'estado'     => $estado,
                'trash'      => $trash,
                'perPage'    => $perPage,
                'sort'       => $sort,
                'dir'        => $dir,
            ],
            'catalogs' => [
                'carreras'   => $carreras,
                'profesores' => $profesores,
                'semestres'  => $semestres,
                'lineas'     => $lineas,
                'asignaturas'=> $asignaturas,
                'estados'    => ['borrador', 'publicado'],
            ],
            'projects' => $projects,
        ]);
    }

    /**
     * POST /admin/tic-projects
     * Crear un nuevo proyecto TIC.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'nombre_proyecto'           => ['required','string','max:255'],
            'carrera'                   => ['nullable','string','max:150'],
            'profesor_cargo'            => ['nullable','string','max:150'],
            'resumen'                   => ['nullable','string'],
            'semestre'                  => ['nullable','string','max:20'],
            'linea_investigacion'       => ['nullable','string','max:255'],
            'areas_trabajo'             => ['nullable','array'],
            'areas_trabajo.*'           => ['string','max:255'],
            'asignaturas_relacionadas'  => ['nullable','array'],
            'asignaturas_relacionadas.*'=> ['string','max:255'],
            'estado'                    => ['required','string', Rule::in(['borrador','publicado'])],
        ]);

        // Trazabilidad
        if (auth()->check()) {
            $data['created_by'] = auth()->id();
            $data['updated_by'] = auth()->id();
        }

        TicProject::create($data);

        return back()->with('success', 'Proyecto TIC creado correctamente.');
    }

    /**
     * PATCH /admin/tic-projects/{project}
     * Actualizar proyecto TIC.
     */
    public function update(Request $request, TicProject $project)
    {
        $data = $request->validate([
            'nombre_proyecto'           => ['required','string','max:255'],
            'carrera'                   => ['nullable','string','max:150'],
            'profesor_cargo'            => ['nullable','string','max:150'],
            'resumen'                   => ['nullable','string'],
            'semestre'                  => ['nullable','string','max:20'],
            'linea_investigacion'       => ['nullable','string','max:255'],
            'areas_trabajo'             => ['nullable','array'],
            'areas_trabajo.*'           => ['string','max:255'],
            'asignaturas_relacionadas'  => ['nullable','array'],
            'asignaturas_relacionadas.*'=> ['string','max:255'],
            'estado'                    => ['required','string', Rule::in(['borrador','publicado'])],
        ]);

        if (auth()->check()) {
            $data['updated_by'] = auth()->id();
        }

        $project->update($data);

        return back()->with('success', 'Proyecto TIC actualizado correctamente.');
    }

    /**
     * DELETE /admin/tic-projects/{project}
     * Enviar a papelera (soft delete).
     */
    public function destroy(TicProject $project)
    {
        $project->delete();
        return back()->with('success', 'Proyecto enviado a la papelera.');
    }

    /**
     * PATCH /admin/tic-projects/{id}/restore
     * Restaurar desde papelera.
     */
    public function restore($id)
    {
        $project = TicProject::withTrashed()->findOrFail($id);
        if ($project->trashed()) {
            $project->restore();
        }
        return back()->with('success', 'Proyecto restaurado.');
    }

    /**
     * DELETE /admin/tic-projects/{id}/force
     * Eliminación definitiva.
     */
    public function forceDestroy($id)
    {
        $project = TicProject::withTrashed()->findOrFail($id);
        $project->forceDelete();

        return back()->with('success', 'Proyecto eliminado definitivamente.');
    }
}
