<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Publicacion;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class AdminPublicacionController extends Controller
{
    /**
     * GET /admin/publicaciones
     */
    public function index(Request $request)
    {
        $q        = $request->string('q')->toString();
        $estado   = $request->string('estado')->toString();      // 'publicado'|'en-prensa'
        $revista  = $request->string('revista')->toString();     // like
        $editorial= $request->string('editorial')->toString();   // like
        $anio     = $request->integer('anio');                   // exacto
        $anioFrom = $request->integer('anio_from');
        $anioTo   = $request->integer('anio_to');
        $citadoMin= $request->integer('citado_min');
        $citadoMax= $request->integer('citado_max');
        $activo   = $request->string('activo')->toString();      // ''|'1'|'0'
        $trash    = $request->string('trash','none')->toString();// 'none'|'with'|'only'

        $perPage  = (int) max(5, min(100, $request->integer('per_page', 12)));
        $sort     = $request->string('sort','anio')->toString(); // 'anio'|'titulo'|'revista'|'editorial'|'citado_por'|'created_at'
        $dir      = $request->string('dir','desc')->toString();  // 'asc'|'desc'

        $builder = Publicacion::query();

        // Papelera
        if ($trash === 'with')      $builder->withTrashed();
        elseif ($trash === 'only')  $builder->onlyTrashed();

        // Búsqueda general
        $builder->search($q);

        // Filtros
        $builder->estado($estado);
        if ($revista)   $builder->where('revista','like',"%{$revista}%");
        if ($editorial) $builder->where('editorial','like',"%{$editorial}%");
        if ($anio)      $builder->where('anio',$anio);
        if ($anioFrom && $anioTo)   $builder->whereBetween('anio', [$anioFrom,$anioTo]);
        elseif ($anioFrom)          $builder->where('anio','>=',$anioFrom);
        elseif ($anioTo)            $builder->where('anio','<=',$anioTo);

        if (!is_null($citadoMin))   $builder->where('citado_por','>=',$citadoMin);
        if (!is_null($citadoMax))   $builder->where('citado_por','<=',$citadoMax);

        $builder->activas($activo);

        // Orden permitido
        $allowed = ['anio','titulo','revista','editorial','citado_por','created_at'];
        if (!in_array($sort,$allowed,true)) $sort = 'anio';
        $builder->orderBy($sort, $dir === 'asc' ? 'asc' : 'desc');

        $rows = $builder
            ->paginate($perPage)
            ->withQueryString()
            ->through(function (Publicacion $p) {
                return [
                    'id'         => $p->id,
                    'anio'       => $p->anio,
                    'estado'     => $p->estado,
                    'titulo'     => $p->titulo,
                    'autores'    => $p->autores,
                    'revista'    => $p->revista,
                    'editorial'  => $p->editorial,
                    'extracto'   => $p->extracto,
                    'citado_por' => $p->citado_por,
                    'versiones'  => $p->versiones,
                    'url'        => $p->url,
                    'activo'     => (bool)$p->activo,
                    'deleted_at' => $p->deleted_at?->toISOString(),
                    'created_at' => $p->created_at?->toISOString(),
                    'updated_at' => $p->updated_at?->toISOString(),
                ];
            });

        // Catálogos simples
        $estados = [Publicacion::ESTADO_PUBLICADO, Publicacion::ESTADO_EN_PRENSA];
        $years = range((int)date('Y')+1, (int)date('Y')-25); // últimos 25 años + prox.
        sort($years);
        $years = array_reverse($years); // descendente
        $years = array_map('strval', $years);

        return Inertia::render('Admin/Publicaciones/Index', [
            'filters' => [
                'q'          => $q,
                'estado'     => $estado,
                'revista'    => $revista,
                'editorial'  => $editorial,
                'anio'       => $anio,
                'anio_from'  => $anioFrom,
                'anio_to'    => $anioTo,
                'citado_min' => $citadoMin,
                'citado_max' => $citadoMax,
                'activo'     => $activo,
                'trash'      => $trash,
                'perPage'    => $perPage,
                'sort'       => $sort,
                'dir'        => $dir,
            ],
            'estados' => $estados,
            'years'   => $years,
            'publicaciones' => $rows,
        ]);
    }

    /**
     * POST /admin/publicaciones
     */
    public function store(Request $request)
    {
        $currentYear = (int) now()->year + 1; // permitir "en prensa" del próximo año

        $data = $request->validate([
            'anio'       => ['required','integer','min:1900','max:'.$currentYear],
            'estado'     => ['required','string', Rule::in([Publicacion::ESTADO_PUBLICADO, Publicacion::ESTADO_EN_PRENSA])],
            'titulo'     => ['required','string','max:255'],
            'autores'    => ['required','string','max:500'],
            'revista'    => ['nullable','string','max:255'],
            'editorial'  => ['nullable','string','max:255'],
            'extracto'   => ['nullable','string'],
            'citado_por' => ['nullable','integer','min:0'],
            'versiones'  => ['nullable','integer','min:1','max:65535'],
            'url'        => ['nullable','url','max:255','unique:publicaciones,url'],
            'activo'     => ['sometimes','boolean'],
        ]);

        // unicidad compuesta título+anio
        // (ya la asegura la db; aquí sólo mensaje limpio)
        if (Publicacion::where('titulo',$data['titulo'])->where('anio',$data['anio'])->exists()) {
            return back()->withErrors(['titulo' => 'Ya existe una publicación con ese título en el año seleccionado.'])->withInput();
        }

        $data['activo'] = $data['activo'] ?? true;

        if (auth()->check()) {
            $data['created_by'] = auth()->id();
            $data['updated_by'] = auth()->id();
        }

        Publicacion::create($data);

        return back()->with('success', 'Publicación creada correctamente.');
    }

    /**
     * PATCH /admin/publicaciones/{publicacion}
     */
    public function update(Request $request, Publicacion $publicacion)
    {
        $currentYear = (int) now()->year + 1;

        $data = $request->validate([
            'anio'       => ['required','integer','min:1900','max:'.$currentYear],
            'estado'     => ['required','string', Rule::in([Publicacion::ESTADO_PUBLICADO, Publicacion::ESTADO_EN_PRENSA])],
            'titulo'     => ['required','string','max:255'],
            'autores'    => ['required','string','max:500'],
            'revista'    => ['nullable','string','max:255'],
            'editorial'  => ['nullable','string','max:255'],
            'extracto'   => ['nullable','string'],
            'citado_por' => ['nullable','integer','min:0'],
            'versiones'  => ['nullable','integer','min:1','max:65535'],
            'url'        => ['nullable','url','max:255', Rule::unique('publicaciones','url')->ignore($publicacion->id)],
            'activo'     => ['sometimes','boolean'],
        ]);

        // unicidad compuesta título+anio
        $exists = Publicacion::where('titulo',$data['titulo'])
            ->where('anio',$data['anio'])
            ->where('id','!=',$publicacion->id)
            ->exists();
        if ($exists) {
            return back()->withErrors(['titulo' => 'Ya existe otra publicación con ese título en el año seleccionado.'])->withInput();
        }

        if (auth()->check()) {
            $data['updated_by'] = auth()->id();
        }

        $publicacion->update($data);

        return back()->with('success', 'Publicación actualizada correctamente.');
    }

    /**
     * DELETE /admin/publicaciones/{publicacion}
     */
    public function destroy(Publicacion $publicacion)
    {
        $publicacion->delete();
        return back()->with('success', 'Publicación enviada a la papelera.');
    }

    /**
     * PATCH /admin/publicaciones/{id}/restore
     */
    public function restore($id)
    {
        $p = Publicacion::withTrashed()->findOrFail($id);
        if ($p->trashed()) $p->restore();
        return back()->with('success', 'Publicación restaurada.');
    }

    /**
     * DELETE /admin/publicaciones/{id}/force
     */
    public function forceDestroy($id)
    {
        $p = Publicacion::withTrashed()->findOrFail($id);
        $p->forceDelete();
        return back()->with('success', 'Publicación eliminada definitivamente.');
    }
}
