<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ProyectoInvestigacion;
use Illuminate\Support\Facades\Schema;
use Illuminate\Http\Request;
use App\Models\Docente;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class AdminProyectoInvestigacionController extends Controller
{
    /**
     * GET /admin/proyectos
     * Listado con filtros, orden y paginación (similar a AdminDocenteController@index).
     */

public function index(Request $request)
{
    $q        = $request->string('q')->toString();
    $linea    = $request->string('linea')->toString();  // enum linea_investigacion
    $estado   = $request->string('estado')->toString(); // enum estado
    $prof     = $request->string('profesor')->toString();
    $ano      = $request->integer('ano');               // año exacto
    $anoFrom  = $request->integer('ano_from');
    $anoTo    = $request->integer('ano_to');
    $trash    = $request->string('trash', 'none')->toString(); // 'none' | 'with' | 'only'

    $perPage  = (int) max(5, min(100, $request->integer('per_page', 10)));
    $sort     = $request->string('sort', 'titulo')->toString(); // 'titulo'|'profesor_a_cargo'|'ano'|'estado'|'created_at'
    $dir      = $request->string('dir', 'asc')->toString();     // 'asc'|'desc'

    $builder = ProyectoInvestigacion::query();

    // Papelera / SoftDeletes
    if ($trash === 'with') {
        $builder->withTrashed();
    } elseif ($trash === 'only') {
        $builder->onlyTrashed();
    }

    // Búsqueda general (título, profesor, descripción)
    $builder->when($q, function ($query, $q) {
        $query->where(function ($q2) use ($q) {
            $q2->where('titulo', 'like', "%{$q}%")
               ->orWhere('profesor_a_cargo', 'like', "%{$q}%")
               ->orWhere('descripcion', 'like', "%{$q}%");
        });
    });

    // Filtros específicos
    $builder->when($linea, fn($q2) => $q2->where('linea_investigacion', $linea));
    $builder->when($estado, fn($q2) => $q2->where('estado', $estado));
    $builder->when($prof,   fn($q2) => $q2->where('profesor_a_cargo', 'like', "%{$prof}%"));
    $builder->when($ano,    fn($q2) => $q2->where('ano', $ano));

    // Rango de años
    if ($anoFrom && $anoTo) {
        $builder->whereBetween('ano', [$anoFrom, $anoTo]);
    } elseif ($anoFrom) {
        $builder->where('ano', '>=', $anoFrom);
    } elseif ($anoTo) {
        $builder->where('ano', '<=', $anoTo);
    }

    // Orden permitido
    $allowedSorts = ['titulo','profesor_a_cargo','ano','estado','created_at'];
    if (!in_array($sort, $allowedSorts, true)) {
        $sort = 'titulo';
    }
    $builder->orderBy($sort, $dir === 'desc' ? 'desc' : 'asc');

    // Paginación + mapeo para el frontend
    $proyectos = $builder
        ->paginate($perPage)
        ->withQueryString()
        ->through(function (ProyectoInvestigacion $p) {
            return [
                'id'                  => $p->id,
                'titulo'              => $p->titulo,
                'linea_investigacion' => $p->linea_investigacion,
                'profesor_a_cargo'    => $p->profesor_a_cargo,
                'ano'                 => $p->ano,
                'estado'              => $p->estado,
                'descripcion'         => $p->descripcion,
                'keywords'            => $p->keywords,
                'deleted_at'          => $p->deleted_at?->toISOString(),
                'created_at'          => $p->created_at?->toISOString(),
                'updated_at'          => $p->updated_at?->toISOString(),
            ];
        });

    // Catálogos
    $lineas  = collect(ProyectoInvestigacion::LINEAS_INVESTIGACION)->values();
    $estados = collect(ProyectoInvestigacion::ESTADOS)->values();
    $anos    = ['2024','2025','2026','2027','2028','2029','2030','2031','2032','2033','2034','2035'];

    // ======= Listado de Docentes (compacto para selects) ====================
    // Tolerante a tus posibles columnas: nombre/apellido, nombres/apellidos, nombre_completo
    $docTable = (new Docente)->getTable();
    $docCols  = Schema::getColumnListing($docTable);

    $hasNombre      = in_array('nombre', $docCols, true);
    $hasApellido    = in_array('apellido', $docCols, true);
    $hasNombres     = in_array('nombres', $docCols, true);
    $hasApellidos   = in_array('apellidos', $docCols, true);
    $hasNombreFull  = in_array('nombre_completo', $docCols, true);
    $hasCorreo      = in_array('correo', $docCols, true);

    $docentesRaw = Docente::query()->get(); // ordenaremos después por el nombre construido

    $docentes = $docentesRaw->map(function ($d) use ($hasNombre, $hasApellido, $hasNombres, $hasApellidos, $hasNombreFull, $hasCorreo) {
        $primero = $hasNombre   ? ($d->nombre   ?? null) : ($hasNombres ? ($d->nombres ?? null) : null);
        $ultimo  = $hasApellido ? ($d->apellido ?? null) : ($hasApellidos ? ($d->apellidos ?? null) : null);

        $nombre = trim(implode(' ', array_filter([$primero, $ultimo]))) ?: (
            $hasNombreFull ? ($d->nombre_completo ?? null) : null
        );

        if (!$nombre) {
            $nombre = $hasCorreo ? ($d->correo ?? null) : null;
        }
        if (!$nombre) {
            $nombre = "Docente #{$d->id}";
        }

        return [
            'id'     => $d->id,
            'nombre' => $nombre,
            'correo' => $hasCorreo ? ($d->correo ?? null) : null,
        ];
    })->sortBy('nombre', SORT_NATURAL | SORT_FLAG_CASE)->values();

    return Inertia::render('Admin/Proyectos/Index', [
        'filters'   => [
            'q'       => $q,
            'linea'   => $linea,
            'estado'  => $estado,
            'profesor'=> $prof,
            'ano'     => $ano,
            'ano_from'=> $anoFrom,
            'ano_to'  => $anoTo,
            'trash'   => $trash,
            'perPage' => $perPage,
            'sort'    => $sort,
            'dir'     => $dir,
        ],
        'lineas'    => $lineas,
        'estados'   => $estados,
        'anos'      => $anos,
        'proyectos' => $proyectos,
        'docentes'  => $docentes, // ← listado para tu select de profesores
    ]);
}


    /**
     * POST /admin/proyectos
     * Crear un nuevo proyecto de investigación.
     */
    public function store(Request $request)
    {
        $currentYear = (int) now()->year;

        $data = $request->validate([
            'titulo'              => ['required','string','max:255'],
            'linea_investigacion' => ['required','string', Rule::in(ProyectoInvestigacion::LINEAS_INVESTIGACION)],
            'profesor_a_cargo'    => ['required','string','max:255'],
            'ano'                 => ['required','integer','min:1900','max:'.$currentYear],
            'estado'              => ['required','string', Rule::in(ProyectoInvestigacion::ESTADOS)],
            'descripcion'         => ['required','string'],
            'keywords'            => ['nullable','array'],
            'keywords.*'          => ['string','max:100'],
        ]);

        // Normaliza keywords a array vacío si no viene
        $data['keywords'] = $data['keywords'] ?? [];

        ProyectoInvestigacion::create($data);

        return back()->with('success', 'Proyecto creado correctamente.');
    }

    /**
     * PATCH /admin/proyectos/{proyecto}
     * Actualizar un proyecto de investigación.
     */
    public function update(Request $request, ProyectoInvestigacion $proyecto)
    {
        $currentYear = (int) now()->year;

        $data = $request->validate([
            'titulo'              => ['required','string','max:255'],
            'linea_investigacion' => ['required','string', Rule::in(ProyectoInvestigacion::LINEAS_INVESTIGACION)],
            'profesor_a_cargo'    => ['required','string','max:255'],
            'ano'                 => ['required','integer','min:1900','max:'.$currentYear],
            'estado'              => ['required','string', Rule::in(ProyectoInvestigacion::ESTADOS)],
            'descripcion'         => ['required','string'],
            'keywords'            => ['nullable','array'],
            'keywords.*'          => ['string','max:100'],
        ]);

        $proyecto->update([
            ...$data,
            'keywords' => $data['keywords'] ?? [],
        ]);

        return back()->with('success', 'Proyecto actualizado correctamente.');
    }

    /**
     * DELETE /admin/proyectos/{proyecto}
     * Enviar a papelera (soft delete).
     */
    public function destroy(ProyectoInvestigacion $proyecto)
    {
        $proyecto->delete();
        return back()->with('success', 'Proyecto enviado a la papelera.');
    }

    /**
     * PATCH /admin/proyectos/{id}/restore
     * Restaurar un proyecto eliminado.
     */
    public function restore($id)
    {
        $proyecto = ProyectoInvestigacion::withTrashed()->findOrFail($id);
        if ($proyecto->trashed()) {
            $proyecto->restore();
        }

        return back()->with('success', 'Proyecto restaurado.');
    }

    /**
     * DELETE /admin/proyectos/{id}/force
     * Eliminación definitiva.
     */
    public function forceDestroy($id)
    {
        $proyecto = ProyectoInvestigacion::withTrashed()->findOrFail($id);
        $proyecto->forceDelete();

        return back()->with('success', 'Proyecto eliminado definitivamente.');
    }
}
