<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Pregrado;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Inertia\Inertia;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class AdminPregradoController extends Controller
{
    /**
     * GET /admin/pregrados
     * Filtros:
     *  - q: búsqueda por nombre, slug, título, coordinador
     *  - modalidad: exacta (opcional)
     *  - visible: 'all' | '1' | '0'
     *  - trash: 'none' | 'with' | 'only'
     *  - per_page: tamaño página
     *  - sort: orden|nombre|created_at
     *  - dir: asc|desc
     */
    public function index(Request $request)
    {
        $q         = $request->string('q')->toString();
        $modalidad = $request->string('modalidad')->toString();
        $visible   = $request->string('visible', 'all')->toString(); // all|1|0
        $trash     = $request->string('trash', 'none')->toString();  // none|with|only
        $perPage   = (int) max(5, min(100, $request->integer('per_page', 10)));
        $sort      = $request->string('sort', 'orden')->toString();  // orden|nombre|created_at
        $dir       = $request->string('dir', 'asc')->toString();

        $builder = Pregrado::query();

        // Papelera
        if ($trash === 'with') {
            $builder->withTrashed();
        } elseif ($trash === 'only') {
            $builder->onlyTrashed();
        }

        // Búsqueda
        $builder->when($q, function ($query, $q) {
            $query->where(function ($q2) use ($q) {
                $q2->where('nombre', 'like', "%{$q}%")
                   ->orWhere('slug', 'like', "%{$q}%")
                   ->orWhere('titulo', 'like', "%{$q}%")
                   ->orWhere('coordinador_nombre', 'like', "%{$q}%")
                   ->orWhere('coordinador_email', 'like', "%{$q}%");
            });
        });

        // Modalidad
        $builder->when($modalidad, fn($q) => $q->where('modalidad', $modalidad));

        // Visible
        if ($visible === '1') $builder->where('is_visible', true);
        if ($visible === '0') $builder->where('is_visible', false);

        // Orden
        if (!in_array($sort, ['orden','nombre','created_at'], true)) $sort = 'orden';
        $builder->orderBy($sort, $dir === 'desc' ? 'desc' : 'asc');

        $pregrados = $builder
            ->paginate($perPage)
            ->withQueryString()
            ->through(function (Pregrado $p) {
                return [
                    'id'          => $p->id,
                    'slug'        => $p->slug,
                    'nombre'      => $p->nombre,
                    'modalidad'   => $p->modalidad,
                    'titulo'      => $p->titulo,
                    'coordinador' => [
                        'nombre' => $p->coordinador_nombre,
                        'email'  => $p->coordinador_email,
                        'foto'   => $p->coordinador_foto,
                    ],
                    'imagen_hero' => $p->imagen_hero,
                    'malla_url'   => $p->malla_url,
                    'requisitos'  => $p->requisitos ?? [],
                    'is_visible'  => (bool) $p->is_visible,
                    'orden'       => (int) $p->orden,
                    'deleted_at'  => $p->deleted_at?->toISOString(),
                    'created_at'  => $p->created_at?->toISOString(),
                ];
            });

        // Para filtros (modalidades distintas disponibles)
        $modalidades = Pregrado::query()
            ->select('modalidad')
            ->whereNotNull('modalidad')
            ->distinct()
            ->orderBy('modalidad')
            ->pluck('modalidad')
            ->values();

        return Inertia::render('Admin/Pregrados/Index', [
            'filters'     => compact('q','modalidad','visible','trash','perPage','sort','dir'),
            'modalidades' => $modalidades,
            'pregrados'   => $pregrados,
        ]);
    }

    /**
     * POST /admin/pregrados
     * Crea un pregrado.
     * Soporta 'requisitos' como array o como string (separado por saltos de línea).
     */

    // app/Http/Controllers/Admin/AdminPregradoController.php (solo los métodos cambiados)

// ...

public function store(Request $request)
{
    $data = $request->validate([
        'nombre'             => ['required','string','max:255'],
        'duracion'           => ['nullable','string','max:120'],
        'modalidad'          => ['nullable','string','max:120'],
        'titulo'             => ['nullable','string','max:255'],
        'coordinador_nombre' => ['nullable','string','max:255'],
        'coordinador_email'  => ['nullable','email','max:255'],

        // archivos:
        'coordinador_foto'   => ['nullable','file','image','max:2048'], // 2MB
        'imagen_hero'        => ['nullable','file','image','max:4096'], // 4MB
        'foto_tarjeta'       => ['nullable','file','image','max:2048'], // 2MB
        // Usa uno u otro nombre según tu DB:
        'malla'              => ['nullable','file','mimes:pdf','max:12288'], // 12MB PDF

        'informacion'        => ['nullable','string'],
        'perfil'             => ['nullable','string'],
        'requisitos'         => ['nullable'], // array o string (lo normalizamos)
        'is_visible'         => ['nullable','boolean'],
        'orden'              => ['nullable','integer','min:0','max:65535'],
    ]);

    // slug desde nombre (único)
    $slugBase = Str::slug($data['nombre']);
    $slug = $slugBase;
    $i = 1;
    while (\App\Models\Pregrado::withTrashed()->where('slug', $slug)->exists()) {
        $slug = $slugBase.'-'.$i++;
    }

    // normalizar requisitos (array)
    $req = $data['requisitos'] ?? null;
    if (is_string($req)) $req = array_values(array_filter(array_map('trim', preg_split("/\r\n|\n|\r/", $req))));
    if ($req !== null && !is_array($req)) $req = null;

    // subir archivos si vienen
    $paths = [];
    if ($request->hasFile('coordinador_foto')) {
        $paths['coordinador_foto'] = $request->file('coordinador_foto')->store('pregrados/coordinadores','public');
    }
    if ($request->hasFile('imagen_hero')) {
        $paths['imagen_hero'] = $request->file('imagen_hero')->store('pregrados/hero','public');
    }
    if ($request->hasFile('foto_tarjeta')) {
        $paths['foto_tarjeta'] = $request->file('foto_tarjeta')->store('pregrados/tarjetas','public');
    }
    if ($request->hasFile('malla')) {
        // Si tu columna es malla_url:
        $paths['malla_url'] = $request->file('malla')->store('pregrados/mallas','public');
        // Si la renombraste a malla_archivo, usa:
        // $paths['malla_archivo'] = $request->file('malla')->store('pregrados/mallas','public');
    }

    \App\Models\Pregrado::create([
        'slug'               => $slug,
        'nombre'             => $data['nombre'],
        'duracion'           => $data['duracion'] ?? null,
        'modalidad'          => $data['modalidad'] ?? null,
        'titulo'             => $data['titulo'] ?? null,
        'coordinador_nombre' => $data['coordinador_nombre'] ?? null,
        'coordinador_email'  => $data['coordinador_email'] ?? null,
        'coordinador_foto'   => $paths['coordinador_foto'] ?? null,
        'imagen_hero'        => $paths['imagen_hero'] ?? null,
        'foto_tarjeta'       => $paths['foto_tarjeta'] ?? null,
        'informacion'        => $data['informacion'] ?? null,
        'perfil'             => $data['perfil'] ?? null,
        'malla_url'          => $paths['malla_url'] ?? null, // o malla_archivo
        'requisitos'         => $req,
        'is_visible'         => $data['is_visible'] ?? true,
        'orden'              => $data['orden'] ?? 0,
    ]);

    return back()->with('success', 'Carrera creada correctamente.');
}

public function update(Request $request, \App\Models\Pregrado $pregrado)
{
    $data = $request->validate([
        'nombre'             => ['required','string','max:255'],
        'duracion'           => ['nullable','string','max:120'],
        'modalidad'          => ['nullable','string','max:120'],
        'titulo'             => ['nullable','string','max:255'],
        'coordinador_nombre' => ['nullable','string','max:255'],
        'coordinador_email'  => ['nullable','email','max:255'],

        // archivos:
        'coordinador_foto'   => ['nullable','file','image','max:2048'],
        'imagen_hero'        => ['nullable','file','image','max:4096'],
        'foto_tarjeta'       => ['nullable','file','image','max:2048'],
        'malla'              => ['nullable','file','mimes:pdf','max:12288'],

        'informacion'        => ['nullable','string'],
        'perfil'             => ['nullable','string'],
        'requisitos'         => ['nullable'],
        'is_visible'         => ['nullable','boolean'],
        'orden'              => ['nullable','integer','min:0','max:65535'],
    ]);

    // slug actualizado desde nombre (mantiene el mismo si no cambia)
    $newSlug = Str::slug($data['nombre']);
    if ($newSlug !== $pregrado->slug) {
        $slug = $newSlug;
        $i = 1;
        while (\App\Models\Pregrado::withTrashed()->where('slug', $slug)->where('id','<>',$pregrado->id)->exists()) {
            $slug = $newSlug.'-'.$i++;
        }
        $pregrado->slug = $slug;
    }

    $req = $data['requisitos'] ?? null;
    if (is_string($req)) $req = array_values(array_filter(array_map('trim', preg_split("/\r\n|\n|\r/", $req))));
    if ($req !== null && !is_array($req)) $req = null;

    // subir archivos si vienen (y opcionalmente borrar los anteriores)
    if ($request->hasFile('coordinador_foto')) {
        $path = $request->file('coordinador_foto')->store('pregrados/coordinadores','public');
        $pregrado->coordinador_foto = $path;
    }
    if ($request->hasFile('imagen_hero')) {
        $path = $request->file('imagen_hero')->store('pregrados/hero','public');
        $pregrado->imagen_hero = $path;
    }
    if ($request->hasFile('foto_tarjeta')) {
        $path = $request->file('foto_tarjeta')->store('pregrados/tarjetas','public');
        $pregrado->foto_tarjeta = $path;
    }
    if ($request->hasFile('malla')) {
        $path = $request->file('malla')->store('pregrados/mallas','public');
        // $pregrado->malla_archivo = $path; // si renombraste
        $pregrado->malla_url = $path;        // si mantuviste malla_url
    }

    $pregrado->nombre             = $data['nombre'];
    $pregrado->duracion           = $data['duracion'] ?? $pregrado->duracion;
    $pregrado->modalidad          = $data['modalidad'] ?? $pregrado->modalidad;
    $pregrado->titulo             = $data['titulo'] ?? $pregrado->titulo;
    $pregrado->coordinador_nombre = $data['coordinador_nombre'] ?? $pregrado->coordinador_nombre;
    $pregrado->coordinador_email  = $data['coordinador_email'] ?? $pregrado->coordinador_email;
    $pregrado->informacion        = $data['informacion'] ?? $pregrado->informacion;
    $pregrado->perfil             = $data['perfil'] ?? $pregrado->perfil;
    $pregrado->requisitos         = $req ?? $pregrado->requisitos;
    if (array_key_exists('is_visible', $data)) $pregrado->is_visible = (bool)$data['is_visible'];
    if (array_key_exists('orden', $data))      $pregrado->orden = (int)$data['orden'];

    $pregrado->save();

    return back()->with('success', 'Carrera actualizada correctamente.');
}

    /**
     * DELETE /admin/pregrados/{pregrado}
     * Soft delete.
     */
    public function destroy(Pregrado $pregrado)
    {
        $pregrado->delete();
        return back()->with('success', 'Carrera enviada a la papelera.');
    }

    /**
     * PATCH /admin/pregrados/{id}/restore
     * Restaurar desde papelera.
     */
    public function restore($id)
    {
        $p = Pregrado::withTrashed()->findOrFail($id);
        if ($p->trashed()) $p->restore();
        return back()->with('success', 'Carrera restaurada.');
    }

    /**
     * DELETE /admin/pregrados/{id}/force
     * Eliminación definitiva.
     */
    public function forceDestroy($id)
    {
        $p = Pregrado::withTrashed()->findOrFail($id);
        $p->forceDelete();
        return back()->with('success', 'Carrera eliminada definitivamente.');
    }

    /**
     * PATCH /admin/pregrados/{pregrado}/toggle-visible
     * Cambia visibilidad (mostrar/ocultar en público).
     */
    public function toggleVisible(Pregrado $pregrado)
    {
        $pregrado->is_visible = !$pregrado->is_visible;
        $pregrado->save();
        return back()->with('success', $pregrado->is_visible ? 'Carrera visible.' : 'Carrera oculta.');
    }
}
