<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Postgrado;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Str;

class AdminPostgradoController extends Controller
{
    /**
     * GET /admin/postgrados
     * Filtros:
     *  - q: búsqueda por nombre, slug, título, coordinador
     *  - modalidad: exacta (opcional)
     *  - trash: 'none' | 'with' | 'only'
     *  - per_page: tamaño página
     *  - sort: nombre|created_at
     *  - dir: asc|desc
     */
    public function index(Request $request)
    {
        $q         = $request->string('q')->toString();
        $modalidad = $request->string('modalidad')->toString();
        $trash     = $request->string('trash', 'none')->toString();  // none|with|only
        $perPage   = (int) max(5, min(100, $request->integer('per_page', 10)));
        $sort      = $request->string('sort', 'nombre')->toString(); // nombre|created_at
        $dir       = $request->string('dir', 'asc')->toString();

        $builder = Postgrado::query();

        // Papelera
        if ($trash === 'with') {
            $builder->withTrashed();
        } elseif ($trash === 'only') {
            $builder->onlyTrashed();
        }

        // Búsqueda
        $builder->when($q, function ($query, $q) {
            $query->where(function ($q2) use ($q) {
                $q2->where('nombre', 'like', "%{$q}%")
                   ->orWhere('slug', 'like', "%{$q}%")
                   ->orWhere('titulo', 'like', "%{$q}%")
                   ->orWhere('coordinador_nombre', 'like', "%{$q}%")
                   ->orWhere('coordinador_email', 'like', "%{$q}%");
            });
        });

        // Modalidad
        $builder->when($modalidad, fn ($q2) => $q2->where('modalidad', $modalidad));

        // Orden
        if (!in_array($sort, ['nombre','created_at'], true)) {
            $sort = 'nombre';
        }
        $builder->orderBy($sort, $dir === 'desc' ? 'desc' : 'asc');

        $postgrados = $builder
            ->paginate($perPage)
            ->withQueryString()
            ->through(function (Postgrado $p) {
                return [
                    'id'           => $p->id,
                    'slug'         => $p->slug,
                    'nombre'       => $p->nombre,
                    'tipo'         => $p->tipo,
                    'area'         => $p->area,
                    'modalidad'    => $p->modalidad,
                    'titulo'       => $p->titulo,
                    'coordinador'  => [
                        'nombre' => $p->coordinador_nombre,
                        'email'  => $p->coordinador_email,
                        'foto'   => $p->coordinador_foto,
                    ],
                    'resolucion_senescyt' => $p->resolucion_senescyt,
                    'resumen'      => $p->resumen,
                    'imagen_hero'  => $p->imagen_hero,
                    'foto_tarjeta' => $p->foto_tarjeta,
                    'malla_url'    => $p->malla_url,
                    'brochure_url' => $p->brochure_url,
                    'requisitos'   => $p->requisitos ?? [],
                    'deleted_at'   => $p->deleted_at?->toISOString(),
                    'created_at'   => $p->created_at?->toISOString(),
                ];
            });

        // Filtro de modalidades disponibles
        $modalidades = Postgrado::query()
            ->select('modalidad')
            ->whereNotNull('modalidad')
            ->distinct()
            ->orderBy('modalidad')
            ->pluck('modalidad')
            ->values();

        return Inertia::render('Admin/Postgrados/Index', [
            'filters'     => compact('q','modalidad','trash','perPage','sort','dir'),
            'modalidades' => $modalidades,
            'postgrados'  => $postgrados,
        ]);
    }

    /**
     * POST /admin/postgrados
     * Crea un postgrado.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'nombre'             => ['required','string','max:255'],
            'duracion'           => ['nullable','string','max:120'],
            'modalidad'          => ['nullable','string','max:120'],
            'titulo'             => ['nullable','string','max:255'],
            'coordinador_nombre' => ['nullable','string','max:255'],
            'coordinador_email'  => ['nullable','email','max:255'],

            // Extras
            'tipo'               => ['nullable','string','max:120'],
            'area'               => ['nullable','string','max:255'],
            'resolucion_senescyt'=> ['nullable','string','max:255'],
            'resumen'            => ['nullable','string'],
            'informacion'        => ['nullable','string'],
            'perfil'             => ['nullable','string'],
            'requisitos'         => ['nullable'], // array o string

            // archivos
            'coordinador_foto'   => ['nullable','file','image','max:2048'],
            'imagen_hero'        => ['nullable','file','image','max:4096'],
            'foto_tarjeta'       => ['nullable','file','image','max:2048'],
            'malla'              => ['nullable','file','mimes:pdf','max:12288'],
            'brochure'           => ['nullable','file','mimes:pdf','max:12288'],

            // enlaces
            'inscripciones_url'  => ['nullable','string','max:500'],
        ]);

        // slug único
        $slugBase = Str::slug($data['nombre']);
        $slug = $slugBase;
        $i = 1;
        while (Postgrado::withTrashed()->where('slug', $slug)->exists()) {
            $slug = $slugBase.'-'.$i++;
        }

        // normalizar requisitos a array
        $req = $data['requisitos'] ?? null;
        if (is_string($req)) {
            $req = array_values(array_filter(array_map('trim', preg_split("/\r\n|\n|\r/", $req))));
        }
        if ($req !== null && !is_array($req)) $req = null;

        // subir archivos
        $paths = [];
        if ($request->hasFile('coordinador_foto')) {
            $paths['coordinador_foto'] = $request->file('coordinador_foto')->store('postgrados/coordinadores','public');
        }
        if ($request->hasFile('imagen_hero')) {
            $paths['imagen_hero'] = $request->file('imagen_hero')->store('postgrados/hero','public');
        }
        if ($request->hasFile('foto_tarjeta')) {
            $paths['foto_tarjeta'] = $request->file('foto_tarjeta')->store('postgrados/tarjetas','public');
        }
        if ($request->hasFile('malla')) {
            $paths['malla_url'] = $request->file('malla')->store('postgrados/mallas','public');
        }
        if ($request->hasFile('brochure')) {
            $paths['brochure_url'] = $request->file('brochure')->store('postgrados/brochures','public');
        }

        Postgrado::create([
            'slug'               => $slug,
            'nombre'             => $data['nombre'],
            'duracion'           => $data['duracion'] ?? null,
            'modalidad'          => $data['modalidad'] ?? null,
            'titulo'             => $data['titulo'] ?? null,
            'tipo'               => $data['tipo'] ?? null,
            'area'               => $data['area'] ?? null,
            'resolucion_senescyt'=> $data['resolucion_senescyt'] ?? null,
            'resumen'            => $data['resumen'] ?? null,
            'informacion'        => $data['informacion'] ?? null,
            'perfil'             => $data['perfil'] ?? null,

            'coordinador_nombre' => $data['coordinador_nombre'] ?? null,
            'coordinador_email'  => $data['coordinador_email'] ?? null,
            'coordinador_foto'   => $paths['coordinador_foto'] ?? null,

            'imagen_hero'        => $paths['imagen_hero'] ?? null,
            'foto_tarjeta'       => $paths['foto_tarjeta'] ?? null,
            'malla_url'          => $paths['malla_url'] ?? null,
            'brochure_url'       => $paths['brochure_url'] ?? null,
            'inscripciones_url'  => $data['inscripciones_url'] ?? null,

            'requisitos'         => $req,
        ]);

        return back()->with('success', 'Postgrado creado correctamente.');
    }

    /**
     * PUT/PATCH /admin/postgrados/{postgrado}
     * Actualiza un postgrado y soporta reemplazo de archivos.
     */
    public function update(Request $request, Postgrado $postgrado)
    {
        dd($postgrado);
        $data = $request->validate([
            'nombre'             => ['nullable','string','max:255'],
            'duracion'           => ['nullable','string','max:120'],
            'modalidad'          => ['nullable','string','max:120'],
            'titulo'             => ['nullable','string','max:255'],
            'coordinador_nombre' => ['nullable','string','max:255'],
            'coordinador_email'  => ['nullable','email','max:255'],

            'tipo'               => ['nullable','string','max:120'],
            'area'               => ['nullable','string','max:255'],
            'resolucion_senescyt'=> ['nullable','string','max:255'],
            'resumen'            => ['nullable','string'],
            'informacion'        => ['nullable','string'],
            'perfil'             => ['nullable','string'],
            'requisitos'         => ['nullable'],

            // archivos
            'coordinador_foto'   => ['nullable','file','image','max:2048'],
            'imagen_hero'        => ['nullable','file','image','max:4096'],
            'foto_tarjeta'       => ['nullable','file','image','max:2048'],
            'malla'              => ['nullable','file','mimes:pdf','max:12288'],
            'brochure'           => ['nullable','file','mimes:pdf','max:12288'],

            'inscripciones_url'  => ['nullable','string','max:500'],
        ]);

        // slug (solo si cambió el nombre)
        $newSlug = Str::slug($data['nombre']);
        if ($newSlug !== $postgrado->slug) {
            $slug = $newSlug;
            $i = 1;
            while (Postgrado::withTrashed()
                ->where('slug', $slug)
                ->where('id','<>',$postgrado->id)
                ->exists()) {
                $slug = $newSlug.'-'.$i++;
            }
            $postgrado->slug = $slug;
        }

        // normalizar requisitos
        $req = $data['requisitos'] ?? null;
        if (is_string($req)) {
            $req = array_values(array_filter(array_map('trim', preg_split("/\r\n|\n|\r/", $req))));
        }
        if ($req !== null && !is_array($req)) $req = null;

        // archivos (si vienen)
        if ($request->hasFile('coordinador_foto')) {
            $postgrado->coordinador_foto = $request->file('coordinador_foto')->store('postgrados/coordinadores','public');
        }
        if ($request->hasFile('imagen_hero')) {
            $postgrado->imagen_hero = $request->file('imagen_hero')->store('postgrados/hero','public');
        }
        if ($request->hasFile('foto_tarjeta')) {
            $postgrado->foto_tarjeta = $request->file('foto_tarjeta')->store('postgrados/tarjetas','public');
        }
        if ($request->hasFile('malla')) {
            $postgrado->malla_url = $request->file('malla')->store('postgrados/mallas','public');
        }
        if ($request->hasFile('brochure')) {
            $postgrado->brochure_url = $request->file('brochure')->store('postgrados/brochures','public');
        }

        // asignar campos
        $postgrado->nombre              = $data['nombre'];
        $postgrado->duracion            = $data['duracion'] ?? $postgrado->duracion;
        $postgrado->modalidad           = $data['modalidad'] ?? $postgrado->modalidad;
        $postgrado->titulo              = $data['titulo'] ?? $postgrado->titulo;
        $postgrado->coordinador_nombre  = $data['coordinador_nombre'] ?? $postgrado->coordinador_nombre;
        $postgrado->coordinador_email   = $data['coordinador_email'] ?? $postgrado->coordinador_email;

        $postgrado->tipo                = $data['tipo'] ?? $postgrado->tipo;
        $postgrado->area                = $data['area'] ?? $postgrado->area;
        $postgrado->resolucion_senescyt = $data['resolucion_senescyt'] ?? $postgrado->resolucion_senescyt;
        $postgrado->resumen             = $data['resumen'] ?? $postgrado->resumen;
        $postgrado->informacion         = $data['informacion'] ?? $postgrado->informacion;
        $postgrado->perfil              = $data['perfil'] ?? $postgrado->perfil;

        if ($req !== null) $postgrado->requisitos = $req;
        if (array_key_exists('inscripciones_url', $data)) {
            $postgrado->inscripciones_url = $data['inscripciones_url'];
        }

        $postgrado->save();

        return back()->with('success', 'Postgrado actualizado correctamente.');
    }

    /**
     * DELETE /admin/postgrados/{postgrado}
     * Soft delete.
     */
    public function destroy(Postgrado $postgrado)
    {
        $postgrado->delete();
        return back()->with('success', 'Postgrado enviado a la papelera.');
    }

    /**
     * PATCH /admin/postgrados/{id}/restore
     * Restaurar desde papelera.
     */
    public function restore($id)
    {
        $p = Postgrado::withTrashed()->findOrFail($id);
        if ($p->trashed()) $p->restore();
        return back()->with('success', 'Postgrado restaurado.');
    }

    /**
     * DELETE /admin/postgrados/{id}/force
     * Eliminación definitiva.
     */
    public function forceDestroy($id)
    {
        $p = Postgrado::withTrashed()->findOrFail($id);
        $p->forceDelete();
        return back()->with('success', 'Postgrado eliminado definitivamente.');
    }
}
