<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Pasantia;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;

class AdminPasantiaController extends Controller
{
    public function index(Request $request)
    {
        $q = Pasantia::query();

        // Filtros
        if ($s = $request->string('s')->trim()) {
            $q->where(function($qq) use ($s) {
                $qq->where('titulo','like',"%{$s}%")
                   ->orWhere('empresa','like',"%{$s}%")
                   ->orWhere('carrera','like',"%{$s}%");
            });
        }
        if ($modalidad = $request->get('modalidad')) {
            if ($modalidad !== 'Todas') $q->where('modalidad', $modalidad);
        }
        if ($carrera = $request->get('carrera')) {
            if ($carrera !== 'Todas') $q->where('carrera', $carrera);
        }
        if ($estado = $request->get('estado')) {
            if ($estado !== 'Todas') $q->where('estado', $estado);
        }

        $pasantias = $q->orderByDesc('created_at')
            ->paginate(10)
            ->withQueryString();

        return Inertia::render('Admin/Pasantias/Index', [
            'pasantias' => $pasantias,
            'filters' => [
                's' => (string) $request->get('s',''),
                'modalidad' => $request->get('modalidad','Todas'),
                'carrera' => $request->get('carrera','Todas'),
                'estado' => $request->get('estado','Todas'),
            ],
        ]);
    }

    public function store(Request $request)
    {
        $data = $this->validateData($request);
        $data['created_by'] = Auth::id();

        // normaliza arreglos desde string separados por coma
        $data['requisitos'] = $this->normalizeArray($data['requisitos'] ?? null);
        $data['beneficios'] = $this->normalizeArray($data['beneficios'] ?? null);

        $p = Pasantia::create($data);

        if ($p->estado === 'publicado' && !$p->published_at) {
            $p->update(['published_at' => now(), 'updated_by' => Auth::id()]);
        }

        return back()->with('success', 'Pasantía creada correctamente.');
    }

    public function update(Request $request, Pasantia $pasantia)
    {
        $data = $this->validateData($request, $pasantia->id);
        $data['updated_by'] = Auth::id();

        $data['requisitos'] = $this->normalizeArray($data['requisitos'] ?? null);
        $data['beneficios'] = $this->normalizeArray($data['beneficios'] ?? null);

        // published_at autom.
        if (($pasantia->estado !== 'publicado') && ($data['estado'] ?? null) === 'publicado' && !$pasantia->published_at) {
            $data['published_at'] = now();
        }

        $pasantia->update($data);

        return back()->with('success', 'Pasantía actualizada.');
    }

    public function destroy(Pasantia $pasantia)
    {
        $pasantia->delete();
        return back()->with('success', 'Pasantía eliminada.');
    }

    public function toggle(Pasantia $pasantia)
    {
        $nuevo = $pasantia->estado === 'publicado' ? 'borrador' : 'publicado';
        $payload = ['estado' => $nuevo, 'updated_by' => Auth::id()];
        if ($nuevo === 'publicado' && !$pasantia->published_at) {
            $payload['published_at'] = now();
        }
        $pasantia->update($payload);
        return back()->with('success', "Estado cambiado a {$nuevo}.");
    }

    private function validateData(Request $request, ?int $id = null): array
    {
        return $request->validate([
            'titulo' => ['required','string','max:180'],
            'empresa' => ['required','string','max:180'],
            'modalidad' => ['required','in:Remoto,Presencial,Híbrido'],
            'duracion' => ['nullable','string','max:80'],
            'carrera' => ['required','string','max:180'],
            'tipo' => ['required','string','max:100'],

            'fecha_inicio' => ['nullable','date'],
            'fecha_maxima_postulacion' => ['nullable','date'],

            'remunerado' => ['required','boolean'],
            'monto_remuneracion' => ['nullable','integer','min:0'],
            'cupos' => ['required','integer','min:1','max:65535'],

            // Puede venir como array o string (lo normalizamos)
            'requisitos' => ['nullable'],
            'beneficios' => ['nullable'],

            'estado' => ['required','in:borrador,publicado'],
        ]);
    }

    private function normalizeArray($value): ?array
    {
        if (is_null($value) || $value === '') return null;
        if (is_array($value)) {
            return array_values(array_filter(array_map(fn($x)=>trim((string)$x), $value)));
        }
        // string separado por comas o saltos
        $parts = preg_split('/[\n,]+/', (string)$value);
        return array_values(array_filter(array_map('trim', $parts)));
    }
}
